function wfu_recon(fileOrDirNames, options)

%
% PURPOSE: - Reconstructs GE raw data P files.  
%            This function is a wrapper function. It
%            determines the type of P file then calls the lower level 
%            reconstruction function.
%
% CATEGORY: Reconstruction
%
% INPUTS:
%
%    fileOrDirNames  - Names of the P files or directories to
%                      be reconstructed. This function is called
%                      recursively until all the P files and
%                      directories with P files are reconstructed.  
%
%                      This function reconstructs all directories first
%                      regardless of thier order in the list, before 
%                      reconstructing individual files in the list.
%
% OUTPUTS: 
%
%    None
%
% EXAMPLE:
%
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------

%
% Internal Variable
%


if nargin < 1 || isempty(fileOrDirNames)
    findOptions.displayFiles    = false;     
    [ fileOrDirNames dirNames ] = wfu_find_pfiles([],[], findOptions);     % Assume current directory 
end

if ~iscell(fileOrDirNames)
    fileOrDirNames = {fileOrDirNames};
end

defaultOptions.convertMat = true;

if nargin < 2 
   options = [];
end
   
options = wfu_set_function_options(options, defaultOptions);

%
% Loop over fileOrDirNames and sort fileNames and Directories
% 

v = ver('Matlab');
versionMatlab = str2num(v.Version(1));


nFileOrDirNames = length(fileOrDirNames);

fileNames = {};
iiFileCounter = 1;


for ii=1:nFileOrDirNames
    
    if isdir(fileOrDirNames{ii})
        
        %
        % Recursively call directories to be reconstructed
        %
        
        findFilesOptions.displayFiles = false;

        dirFileNames = wfu_find_pfiles( fileOrDirNames{ii}, false, findFilesOptions );
        wfu_recon(dirFileNames, options)

        %
        % Convert all mat files to Matlab 6.5
        %
        
        
        if (options.convertMat == true) && (versionMatlab >= 7)

            disp(sprintf('\n*** Convert all mat files to 6.5 ... \n'));
            wfu_convert_mat(fileOrDirNames{ii}, 255, true);
        else
            
            disp(sprintf('\n*** Skipping conversion of all mat files to 6.5 ... \n'));

        end
        
    else        
        fileNames{iiFileCounter} = fileOrDirNames{ii};
        iiFileCounter            = iiFileCounter + 1;                
    end    
end


%
% Loop over fileNames and reconstruct each P-file if known
%

nFileNames = length(fileNames);

if nFileNames > 0
        
    for ii=1:nFileNames
        
        absoluteFileName = fileOrDirNames{ii};
        
        %
        % Post processing P files
        %
                
        [fileDirectory fileNameBase fileNameExtension ] = fileparts(absoluteFileName);           
        fileName = strcat(fileNameBase, fileNameExtension);
        
        hdr = wfu_get_p_header(absoluteFileName);
        
        [directory, psdNameInPFile, extension] = fileparts( hdr.PulseSeqName );

        
        if wfu_is_epiflex_pfile(absoluteFileName)

            % disp(sprintf('Reconstructing EPIFLEX  data'));
            wfu_recon_epiflex(absoluteFileName, [], options);

        elseif any(strfind(  lower(psdNameInPFile), lower('spiral') ))

            %disp(sprintf('Post processing spiral data'));

        elseif any(strfind(  lower(psdNameInPFile), lower('fgre') ))

            %disp(sprintf('Post processing fgre data'));

        else

            % disp(sprintf('\tSkipping post processing of unknown P file %s (%d/%d)', fileName,ii,nFileOrDirNames));
        end

    end
end
