function varargout = wfu_segment_t1weighted_image(dirNameOrFileName)
%
%  wfu_segment_t1weighted_image(dirName)
%
% PURPOSE: Searches the analyze directory, which was created by
% autolocsort, for a T1 weighted image and then performs tissue
% segmentation.  If segmentation has already occurred from a previous call 
% the function display a message stating this.
%
% CATEGORY: 
%
% INPUTS:  
%
%     dirNameOrFileName - This input parameter is either the directory where the T1 weighted image is stored.
%                         or the name of the file to segment. If a directory is given then the directory is 
%                         searched for a suitable T1 weighted image to segement based upon criteria in the 
%                         function wfu_find_template.
%
%
% OUTPUTS: 
%
%     Function calls spm to perform tissue segmentation.  The tissue maps
%     for gray matter (*_seg1.img), white matter (*_seg2.img), and 
%     csf (*_seg3.img) are created in the same directory.
%     
% EXAMPLE:
%
%   >>  wfu_segment_t1weighted_image('/ansir1/bkraft/pasl_rr/rr001_01_01_06/analyze')
%

%==========================================================================
% C H A N G E   L O G
% 
%--------------------------------------------------------------------------

%
% Initialize Input Parameters
%

if nargin < 1 || isempty(dirNameOrFileName)
   dirNameOrFileName = pwd; 
end

% Initialize Output

t1WeightedFileName = [];
csfFileName            = [];
gmFileName             = [];
wmFileName             = [];


%
% Find and segment efgre3d t1 weighted image
%

try
    [spmFlag spmVer] = wfu_exist_spm;

    if iscell(dirNameOrFileName)
        dirNameOrFileName = dirNameOrFileName{1};
    end
    
    
    if( exist(dirNameOrFileName, 'file') == 2 )        
        t1WeightedFileName = dirNameOrFileName; % The filename to be segmented has been explicitely set as an input parameter.        
    else
        t1WeightedFileName = wfu_find_template( dirNameOrFileName );  % Find T1 weighted image for segmentation
    end
   
    
    if exist(t1WeightedFileName, 'file')

        if strcmpi( spmVer, 'SPM5')
            gmFileName   = wfu_prepend_to_filename(t1WeightedFileName,'c1');
            wmFileName   = wfu_prepend_to_filename(t1WeightedFileName,'c2');
            csfFileName  = wfu_prepend_to_filename(t1WeightedFileName,'c3');
        else
            gmFileName   = strrep(t1WeightedFileName,'.img','_seg1.img');
            wmFileName   = strrep(t1WeightedFileName,'.img','_seg2.img');
            csfFileName  = strrep(t1WeightedFileName,'.img','_seg3.img');
        end

        if ~exist(gmFileName,'file') || ~exist(wmFileName,'file') ||~exist(csfFileName,'file')

            disp(sprintf('\n\t*** %s segmenting %s ***', spmVer, t1WeightedFileName));
            
            if strcmpi( spmVer, 'SPM5') 
                t1Template  = fullfile(spm('Dir'),'templates','T1.nii');   % Grap SPM template 
                
                jobs = local_batch_segment(t1WeightedFileName);
            else
                t1Template  = fullfile(spm('Dir'),'templates','T1.mnc');   % Grap SPM template
                spm_segment(t1WeightedFileName, t1Template);
            end

        else
            
            disp(sprintf('\t*** T1 weighted image already segmented ***'));
            
        end
        
    else
        
        disp(sprintf('\t*** T1 weighted image not found for segmentation'));
        
    end
    
    %
    % Return output if requested by user.
    %
    
    if nargout > 0  
        varargout{1} = t1WeightedFileName;
    end
    
    if nargout > 1  
        varargout{2} = csfFileName;
    end
    
    if nargout > 2  
        varargout{3} = gmFileName;
    end
    
    if nargout > 3  
        varargout{4} = wmFileName;
    end
    
    
catch
		errorInfo = lasterror;
		disp(sprintf('Error received: %s\n',errorInfo.message));
    disp(sprintf('\t*** Error occurred during T1 segmentation.  Segmentation aborted.'));
    
    
end



function jobs = local_batch_segment(filename)
% spm batch job for segmentation
% the batch pathway is different than calling spm_segment


job.data = {...
  filename;...
};

%-------set the desired outputs
job.output.GM  = [0 0 1];
job.output.WM  = [0 0 1];
job.output.CSF = [0 0 1];
job.output.biascor = 1;
job.output.cleanup = 0;

%-------------set the priors
job.opts.tpm = {...
  fullfile(spm('Dir'),'tpm','grey.nii');...
  fullfile(spm('Dir'),'tpm','white.nii');...
  fullfile(spm('Dir'),'tpm','csf.nii');...
};

%---------set the segmentation options
job.opts.ngaus = [2 2 2 4];
job.opts.regtype = 'mni';
job.opts.warpreg = 1;
job.opts.warpco = 25;
job.opts.biasreg = 0.000100;
job.opts.biasfwhm = 60;
job.opts.samp = 3;
job.opts.msk = {...
  '';...
};

%--------Create the job structure
myjob = job ; job  = [];	crap{1} = myjob;
myjob = crap;	jobx{1}.preproc = myjob{1};
	jobs{1}.spatial = jobx;

%--------------Run the job
spm_jobman('run',jobs);


