%
% PURPOSE: - Saves variables in the current workspace to a Matlab mat file in
%            Matlab version 6 format regardless of the current version of
%            Matlab.
%
%
% CATEGORY: Utility
%
% INPUTS:
% 
%    fileName - File name of the mat file.
%
%    varargin - List of strings containing the names of the variables to be saved.
%               See Matlab's save function for more information
%
% OUTPUTS: 
%
%    None
%
% EXAMPLE:
%
%   >> a = 2;
%   >> wfu_convert_mat('test.mat');
%
%

function wfu_convert_mat(matFileNamesOrDirectories, recursionDepth, displayFlag) 


if nargin < 3 || isempty(displayFlag)
    displayFlag = true;
end

findOptions.displayFiles = displayFlag;


if nargin < 2 || isempty(recursionDepth)
    recursionDepth = 0;
end

if nargin == 0 || isempty(matFileNamesOrDirectories)
    matFileNamesOrDirectories = wfu_find_mat_files([], recursionDepth, findOptions);
end

if ~iscell(matFileNamesOrDirectories)
    matFileNamesOrDirectories = {matFileNamesOrDirectories};
end

%
% Check Matlab version number
%

v = ver('Matlab');
versionMatlab = str2num(v.Version(1));


if versionMatlab < 7
   error('wfu_convert_mat must be run with Matlab versions 7 or greater.'); 
end


%
% Recursive call to handle passing in a directory name
%


nFileNames = 0;

disp(sprintf('\n'));

for ii=1:length(matFileNamesOrDirectories)

    if isdir(matFileNamesOrDirectories{ii})

        matFileNames = wfu_find_mat_files(matFileNamesOrDirectories{ii}, recursionDepth, findOptions );
        wfu_convert_mat(matFileNames);

    else

        matFileName = wfu_get_spm_filename(matFileNamesOrDirectories{ii},'mat');
        
        %  Check to see if file exists         
        if exist(matFileName,'file') == 2
            
            try
                matVariables = whos('-file', matFileName);

                if length(matVariables) > 0

                    if displayFlag == true
                        disp(sprintf('\tConverting %s to Matlab 6 mat file ***', matFileName));
                    end

                    load(matFileName);

                    delete(matFileName);

                    for jj=1:length(matVariables)
                        saveVariableString =  sprintf('%s', matVariables(jj).name);
                        wfu_save_mat(matFileName, saveVariableString);
                    end
                else
                    if displayFlag == true
                        disp(sprintf('\tSkipping conversion. %s is not a Matlab mat file ***', matFileName));
                    end

                end

            catch
                if displayFlag == true
                    disp(sprintf('\tSkipping conversion. %s is not a Matlab mat file ***', matFileName));
                end
            end
        else
            if displayFlag == true
                disp(sprintf('\tSkipping conversion. %s does not exist ***', matFileName));
            end
        end
    end
end

disp(sprintf('\n'));
