function [template,rank] = wfu_find_template(dirName)
%%
%
% wfu_find_template(dirName)
%
% PURPOSE: Finds a T1 weighted image with a .dicom extension in a directory that is the most suitable for segmentation with SPM. 
%
% CATEGORY: Utility
%
% INPUTS:
%
%    dirName  - Name of the directory to search for a T1 weighted image. If no directory is given then the default directory is the current 
%               directory.  The information in the header (TR, TE, TI, flip angle, pulse sequence, etc) is read in and then used to determine 
%               the most suitable T1 weighted image for segmentation. 
% 
%
% OUTPUTS: 
%      
%    template - Filename of the T1 weighted images suitable for segmentation with SPM.
%
%    rank     - Is a structure containing all of the criteria used to select the template. 
%
% EXAMPLE:
%
%    t1wFileName = wfu_find_template(pwd);
%
%
% NOTES:
%
%    The function wfu_find_template.m is based upon the original function WFU_template/wfu_pick_template.m written by Kathy Pearson and Joe Maldjian. 
%    There was a concern that modifying wfu_pick_template.m would alter SPM processing so a new function with almost identical functionality was
%    created to avoid this potential problem.
%
%



%% ==========================================================================
% C H A N G E   L O G
% 
%--------------------------------------------------------------------------

%% Initialize inputs
if nargin<1, 
    dirName = pwd;
end


% Get full paths
dirName = wfu_get_full_path(dirName);
template = [];

    
% search for .dicom files
options.displayFiles = 0;
options.displayDirectories = 0; 
infoFiles = wfu_find_files('.dicom$',dirName,0,options);

vecLength    = length(infoFiles);
disqualified = zeros(1, vecLength);

if vecLength == 0, template = []; return 
else
    for ii = 1:vecLength
   

     infoStruct(ii)   = wfu_get_mri_info(infoFiles{ii});
        %vectors: 
        vecTR(ii)        = infoStruct(ii).RepetitionTime;
        vecTE(ii)        = infoStruct(ii).EchoTime; 
        vecTI(ii)        = infoStruct(ii).InversionTime;
        
        vecTotalSlice(ii) = infoStruct(ii).NumberOfSlices;
        vecVolumes(ii)    = infoStruct(ii).NumberOfVolumes;
        
        %
        % Adjust the number of slices acquired by the number of volumes.
        % This is necessary to compensate for the number of slices in the
        % info file reporting the total number of slices in the series
        % instead of the number of slices in an imaging volume.
        %
        
        vecSlice(ii) = vecTotalSlice(ii) / max([1 vecVolumes(ii)]);
        
        
        vecAngle(ii)      = infoStruct(ii).FlipAngle;
        vecName{ii}       = infoStruct(ii).InternalPulseSeqName;
        %vecSeriesDesc{ii} = infoStruct(ii).SeriesDescription; 

        vecPlane{ii}     = infoStruct(ii).PlaneType; 
        
        %structures: 
        [path,name]                  = fileparts(infoFiles{ii});
        rank(ii).FileName             = [name,'.img']; 
        rank(ii).RepetitionTime       = vecTR(ii);
        rank(ii).InversionTime        = vecTI(ii);
        rank(ii).EchoTime             = vecTE(ii);
        rank(ii).NumberOfSlices       = vecTotalSlice(ii);  %According to internal criteria.  Scale is 0 - 5 with 5 being the most suitable. Return the total number of slices to be consistent with
                                                          % previous fields returned in the rank structure.  This 
                                                          % minimizes the chance that a function that uses the rank will
                                                          % break with the modifications in this function. 
        rank(ii).NumberOfVolumes      = vecVolumes(ii);        
        rank(ii).FlipAngle            = vecAngle(ii);
        rank(ii).PulseSeqName         = vecName{ii};
        rank(ii).PlaneType            = vecPlane{ii}; 
    end
    
       
    % disqualify 3-plane localizers
    for i = 1:vecLength
        disqualified(ii) = ( (sum(findstr(infoStruct(ii).PulseSeqName,'3Plane')) >=1)      | ...
                            (sum(findstr(infoStruct(ii).SeriesDescription,'ASSET')) >= 1) | ...
                            (sum(findstr(infoStruct(ii).SeriesDescription,'3pl')) >= 1)     ...
                           );
    end
    
    %
    % Logic 1 
    % Look for internal PSD names of efgre3d or despot.  For efgre3d code IR Prep sequence takes precedence over all others.
    
    
    findSeqNameCell = strfind(lower(vecName), 'efgre3d');    % DESPOT code internal name should always be proceeded by efgre3d
    findSeqName      = [];                                   % According to internal criteria.  Scale is 0 - 5 with 5 being the most suitable.% Initialize to an empty array
    
    for ii = 1:vecLength
        if findSeqNameCell{ii} > 0
            findSeqName = [findSeqName ii];
        end
    end
    
    goodSeqName = local_binary(findSeqName, vecLength);
    goodTR      = local_binary(find(vecTR<=50),vecLength);                      % TR less than or equal to 700ms
    goodTE      = local_binary(find(vecTE<=10),vecLength);                      % TE less than or equal to 70ms
    goodTI      = local_binary(find((vecTI<=1200) & (vecTI>=100)), vecLength);  % TI between 100 and 1200 ms
    goodSlice   = local_binary(find(vecSlice==max(vecSlice)),vecLength);      % Maximize brain coverage

    %
    % First, try to find a suitable T1 from an IR-Prepped SPGR sequence
    %
    
    T1_score = 0 * (goodSlice + goodTR+goodTE).*goodSeqName.*goodTI.*(~disqualified)   % Remove 3-Plane files.  
    T1_Index = find( (T1_score == max(T1_score)) & (T1_score > 0) );             % Pick images that meet the most criteria

    if ~isempty(T1_Index)
        
        if length(T1_Index)>1
                    
            T1_Slices           = zeros(1,vecLength);
            T1_Slices(T1_Index) = vecSlice(T1_Index);
            
            bestT1 = find(T1_Slices==max(T1_Slices));   %T1 w/ most slices

            if length(bestT1)>1  
                bestT1 = bestT1(end);    % If more than one bestT1 is found assume that the 
                                         % techs repeated the scan and the
                                         % final image is the best.
            end
            
            template = infoFiles{bestT1};
        
        else
            template = infoFiles{T1_Index};             %Take the only T1; 
        end
    end
    
    
    %
    % This is the original logic before the function started searching for IR-SPGR. 
    %
    
    if isempty(template)
        
        %
        % Logic 2
        %

        %
        % Criteria calculations calculated for Logic 1.
        %
        
        % select 1/0 arrays for desireable images
        %goodTR = local_binary(find(vecTR<=700),vecLength);  % TR less than or equal to 700ms
        %goodTE = local_binary(find(vecTE<=70),vecLength);   % TE less than or equal to 70ms

        %goodSlice = local_binary(find(vecSlice==max(vecSlice)),vecLength);
        %goodAngle = local_binary(find(vecAngle==min(vecAngle)),vecLength); 

        % First, try to find a suitable T1 
        T1_Index = find((goodTR+goodTE).*~disqualified==2); 

        if ~isempty(T1_Index)
            if length(T1_Index)>1
                T1_Slices           = zeros(1,vecLength);
                T1_Slices(T1_Index) = vecSlice(T1_Index);

                bestT1 = find(T1_Slices==max(T1_Slices));   %T1 w/ most slices

                if length(bestT1)>1  
                    bestT1 = bestT1(end);    % If more than one bestT1 is found assume that the 
                                             % techs repeated the scan and the
                                             % final image is the best.
                end

                template = infoFiles{bestT1};
            else
                template = infoFiles{T1_Index};             %Take the only T1; 
            end
        end

    
    end
    
    % try to find suitable T2's 
    %---- need to do this later
    
    [tp,tn] = fileparts(template); 
    template = fullfile(tp,[tn,'.img']);
    
    
end


%_______________________________________________________
%_______________________________________________________
function outV = local_binary(inV,vecLength)
outV = zeros(1,vecLength);
outV(inV) = 1;