function wfu_grid_run(option, item, partition, vlength, cmd, mtype, griddir, append)
%___________________________________________________________________________________________
%
% function wfu_grid_run(option, item, partition, vlength, cmd, mtype, griddir, append)
%
%     run the specified command as an array job on the Sun Grid Engine after
%     creating the mask required for each task
%
%     results will be stored in a directory named "sge" within griddir;
%     if append, the output directory becomes "sge" concatenated with date and time;
%     if already existing, the output directory will be removed and then re-created
%
% option     - one of {'flist', 'image', 'loop'}
% item       - one of {flist file name, image file name, num2str(number of loops)}
% partition  - one of {'point', 'slice', 'vector'}
% vlength    - vector length if partition is 'vector' (ignored otherwise)
% cmd        - matlab command string such as 'grid_test1(mask)'
% mtype      - mask type
%                   1 = list of vector indices
%                   2 = entire list with 0 or 1 to indicate mask off, on
%                   3 = masked item
% griddir    - directory for holding generated scripts, commands, and results
% append     - append date and time to griddir name? (false if not specified)
% 
% Example function calls:
%
% wfu_grid_run('flist', flistname,  'point',  1,     'grid_flist(mask)', 3, pwd, 1);
% wfu_grid_run('flist', flistname,  'vector', 3,     'grid_flist(mask)', 3, pwd, 1);
% wfu_grid_run('image', imagefname, 'vector', 23*55, 'grid_image(mask)', 3, pwd, 1);
% wfu_grid_run('image', imagefname, 'slice',  1,     'grid_image(mask)', 3, pwd, 1);
% wfu_grid_run('loop',  '20',       'point',  1,     'grid_loop(mask)', 3, pwd, 1);
%___________________________________________________________________________________________
% JAM, KLP 4/26/06

if nargin < 8
    append = 0;
end
vectorsize = 1;
if strcmp(partition, 'vector')
    vectorsize = vlength;
end
switch option
    case {'flist'}
        flistname = item;
        filenames = cellstr(wfu_bpm_read_flist(flistname));
        n = length(filenames);
    case {'image'}
	imgname = item;
        v = spm_vol(imgname);
        n = prod(v.dim(1:3));
        if strcmp(partition, 'slice')
	    vectorsize = prod(v.dim(1:2));
        end       
    case {'loop'}
        n = str2num(item);
end
ntask = ceil(n/vectorsize);
%
% check to see if the number of tasks will be greater than current limits allow
%
system_task_max = 75000;
temp_task_max = 100;
grid_task_max = min(system_task_max, temp_task_max);
if ntask > grid_task_max
    disp(sprintf('Requested # of tasks (%d) limit of (%d)!\n', ntask, grid_task_max));
    ntask = grid_task_max;
end
disp(['Number of tasks = ' num2str(ntask)]);
%
% append sge to griddir;
%
outdir = fullfile(griddir, 'sge');
%
% append date and time to outdir name if requested
%
if append
    outdir = [outdir datestr(clock, 30)];
end
%
% MATLAB 6.5 may give inconsistent results using mkdir, rmdir;
% unless MATLAB7, call UNIX for these utilities
%
% remove any existing directory, then create a new one
if strncmp(version, '7', 1)
    if exist(outdir, 'dir')
        [success, message, messageid] = rmdir(outdir);
        if ~success
            error([message ';' messageid]);
        end
    end
    [success, message, messageid] = mkdir(outdir); 
    if ~success
        error([message ';' messageid]);
    end
else
    if exist(outdir, 'dir')
        [status, result] = unix(sprintf('rm -rf %s', outdir));
        if status
            error('unix error on rm');
        end
    end
    [status, result] = unix(sprintf('mkdir -p %s', outdir));
    if status
        error('unix error on mkdir');
    end
end 
%
% create the matlab grid command file to be submitted in the grid job
%
fname = fullfile(outdir, 'gridcmds.m');
[f, message] = fopen(fname, 'wt');
if f == -1
    error(['fopen error: ' message]);
end
fprintf(f, 'wfu_startup([],2);\n');
fprintf(f, 'addpath(''%s'');\n', pwd);
fprintf(f, 'mask=wfu_grid_mask(''%s'',''%s'',%d,%d);\n', option, item, vectorsize, mtype);
if strcmp(option, 'image')
    fprintf(f, 'imagefname = ''%s'';\n', item);
end
fprintf(f, '%s;\n', cmd);
fprintf(f, 'exit;\n');
fclose(f);
%
% create the grid script to be submitted
%
fname = fullfile(outdir, 'grid');
[f, message] = fopen(fname, 'wt');
if f == -1
    error(['fopen error: ' message]);
end
fprintf(f, '#!/bin/csh\n');
fprintf(f, '#\n');
fprintf(f, '# grid ntask outdir datdir\n');
fprintf(f, '#\n');
fprintf(f, '# QSUB options\n');
fprintf(f, '#$ -p -20\n');
fprintf(f, '#$ -l unix=1\n');
fprintf(f, '#$ -l matlab=1\n');
fprintf(f, '#$ -v DISPLAY=alexia:0.0\n');
fprintf(f, '#\n');
fprintf(f, 'set ntask = $1\n');
fprintf(f, 'set outdir = $2\n');
fprintf(f, 'set datdir = $3\n');
fprintf(f, 'set cmdfile = $outdir/gridcmds.m\n');
fprintf(f, 'echo Host: `hostname`\n');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, 'date ''+Date: %cm/%cd/%cy''\n', '%', '%', '%');
fprintf(f, 'date ''+Time: %cH:%cM:%cS''\n', '%', '%', '%');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, 'set id = `printenv SGE_TASK_ID`\n');
fprintf(f, 'set len = `expr "$id" : ".*"`\n');
fprintf(f, 'if ( "$len" < 1 ) then\n');
fprintf(f, '    set id = 1\n');
fprintf(f, '    echo ''undefined task id; setting to 1''\n');
fprintf(f, 'endif\n');
fprintf(f, '#\n');
fprintf(f, '# any special per-task directories should be created here\n');
fprintf(f, '#mkdir $datdir/data$id\n');
fprintf(f, '#mkdir $datdir/results$id\n');
fprintf(f, '#\n');
fprintf(f, 'set priority = 19\n');
if strncmp(version, '7', 1)
    fprintf(f, 'set cmdline = ''nice +''$priority'' ''"`alias matlab7x`"'' < ''$cmdfile\n');
else
    fprintf(f, 'set cmdline = ''nice +''$priority'' ''"`alias matlabx`"'' < ''$cmdfile\n');
end
fprintf(f, 'source ~/.cshrc\n');
fprintf(f, 'echo "executing ..."\n');
fprintf(f, 'echo "    $cmdline"\n');
fprintf(f, '$cmdline\n');
fprintf(f, 'echo '' ''\n');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, 'date ''+Date: %cm/%cd/%cy''\n', '%', '%', '%');
fprintf(f, 'date ''+Time: %cH:%cM:%cS''\n', '%', '%', '%');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, '#\n');
fprintf(f, '# if transient, any special per-task directories should be deleted here\n');
fprintf(f, '#rm -rf $datdir/data$id\n');
fprintf(f, '#rm -rf $datdir/results$id\n');
fprintf(f, '#\n');
fprintf(f, 'exit\n');
fclose(f);
%
% make the grid script executeable
%
[status, result] = unix(sprintf('chmod ugo+x %s', fname));
if status
    error('unix error on chmod');
end
%
% create the grid2 script to be submitted for clean-up
%
% there may be multiple uses for a clean-up script;
% in this case, the default SGE output and error logs
% written without group write permission are granted it
% after all tasks in the array job are completed
%
fname = fullfile(outdir, 'grid2');
[f, message] = fopen(fname, 'wt');
if f == -1
    error(['fopen error: ' message]);
end
fprintf(f, '#!/bin/csh\n');
fprintf(f, '#\n');
fprintf(f, '# grid2 logdir\n');
fprintf(f, '#\n');
fprintf(f, '# QSUB options\n');
fprintf(f, '#$ -p -20\n');
fprintf(f, '#$ -l unix=1\n');
fprintf(f, '#$ -o /dev/null\n');
fprintf(f, '#$ -e /dev/null\n');
fprintf(f, '#\n');
fprintf(f, 'set logdir = $1\n');
fprintf(f, 'echo Host: `hostname`\n');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, 'date ''+Date: %cm/%cd/%cy''\n', '%', '%', '%');
fprintf(f, 'date ''+Time: %cH:%cM:%cS''\n', '%', '%', '%');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, '#\n');
fprintf(f, '# add permissions on all SGE output/error logs for group write access\n');
fprintf(f, '#\n');
fprintf(f, 'chmod -R ug+w $logdir\n');
fprintf(f, 'echo '' ''\n');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, 'date ''+Date: %cm/%cd/%cy''\n', '%', '%', '%');
fprintf(f, 'date ''+Time: %cH:%cM:%cS''\n', '%', '%', '%');
fprintf(f, 'echo ''---------------''\n');
fprintf(f, 'exit\n');
fclose(f);
%
% make the grid2 script executeable
%
[status, result] = unix(sprintf('chmod ugo+x %s', fname));
if status
    error('unix error on chmod');
end
%
% create the grid master script
%
fname = fullfile(outdir, 'gridmaster'); 
[f, message] = fopen(fname, 'wt');
if f == -1
    error(['fopen error: ' message]);
end
fprintf(f, '#!/bin/csh\n');
fprintf(f, '#\n');
fprintf(f, '# gridmaster ntask outdir\n');
fprintf(f, '#\n');
fprintf(f, 'set nonomatch\n');
fprintf(f, 'set ntask = $1\n');
fprintf(f, 'set outdir = $2\n');
fprintf(f, 'set datdir = $outdir/dat\n');
fprintf(f, 'set logdir = $outdir/log\n');
fprintf(f, 'rm -rf $datdir\n');
fprintf(f, 'rm -rf $logdir\n');
fprintf(f, 'mkdir $datdir\n');
fprintf(f, 'mkdir $logdir\n');
fprintf(f, '#\n');
fprintf(f, '# submit the grid job to the queue and obtain its job number\n');
fprintf(f, '#\n');
fprintf(f, 'set script = "$outdir/grid $ntask $outdir $datdir"\n');
fprintf(f, 'set arrayjob = "-t 1-$ntask`echo :1`"\n');
fprintf(f, 'set cmd = "qsub -l h_cpu=23:0:0 -o $logdir -e $logdir $arrayjob $script"\n');
fprintf(f, 'echo cmd is: $cmd\n');
fprintf(f, 'set jobnum = `$cmd | cut -f3 -d " " | cut -f1 -d .`\n');
fprintf(f, 'echo grid jobnum is: $jobnum\n');
fprintf(f, '#\n');
fprintf(f, '# submit a "grid2" clean-up script to the queue\n');
fprintf(f, '# holding it until all the tasks in the array job are complete\n');
fprintf(f, '#\n');
fprintf(f, 'set script = "$outdir/grid2 $logdir"\n');
fprintf(f, 'set cmd = "qsub -hold_jid $jobnum $script"\n');
fprintf(f, 'echo cmd is: $cmd\n');
fprintf(f, 'set jobnum = `$cmd | cut -f3 -d " "`\n');
fprintf(f, 'echo grid2 jobnum is: $jobnum\n');
fclose(f);
%
% make the gridmaster script executeable
%
[status, result] = unix(sprintf('chmod ugo+x %s', fname));
if status
    error('unix error on chmod');
end
%
% pass the grid master script to the operating system
%
unixcmd = sprintf('%s %d %s %s', fullfile(outdir, 'gridmaster'), ntask, outdir);
disp(unixcmd)
[status, result] = unix(sprintf('%s', unixcmd));
if status
    error('error passing grid master script to UNIX');
 else
    disp(result);
    [status, result] = unix('qstat');
    if ~status
        disp(result);
    end
end
