static const char rcsid[] = "$Id: fmriqa_phantomqa.c,v 1.36 2009-01-21 19:57:51 gadde Exp $";

/*
 * fmriqa_phantomqa.cpp --
 * 
 *  Based on Matlab script provided to the fBIRN by Gary Glover.
 *  Description from that script:
 *    To perform a quantitation of snr, sfnr, stability and drift
 *    includes weisskoff plot  MRM 36:643 (1996)
 *
 *  Script subsequently modified to handle XML image descriptors
 *  by Beau Mack and Charles Michelich.
 *
 *  Conversion to C/C++ by Syam Gadde.
 *
 *  Though this program produces images, it does not actually
 *  generate plots -- it provides the data in textual form,
 *  to be plotted by an external program.
 */

#include <bxh_config.h>

#include <stdio.h>

#ifndef HAVE_LIBGSL
int main(int argc, char * argv[])
{
    fprintf(stderr, "Sorry -- to use this program, this package must be compiled with GSL\n(GNU Scientific Library) support!\n");
    return -1;
}
#else /* #ifndef HAVE_GSL */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>

#include <gsl/gsl_multifit.h>
#include <gsl/gsl_fft_real.h>

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    const char * inputfile = NULL;
    int dimnum;
    int msbfirst = 1;
    FILE * fp = NULL;
    int * timepoints = NULL;
    const char * opt_select[] = { ":", ":", NULL, NULL };
    char zselectbuf[16];
    char tselectbuf[16];
    int opt_summaryonly = 0;
    int opt_nofluct = 0;
    int opt_noroi = 0;
    unsigned int opt_roisize = (unsigned int)-1;
    double opt_forcetr = 0;
    char * opt_maskfile = NULL;
    int opt_version = 0;
    
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdr;
    struct bxhdataread maskbdr;
    double * dataptr = NULL;

    const int numopts = 11;
    opt_data opts[11] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_phantomqa [opts] xmlfile [outputbase]\n\n"
	  "This program is usually called by fmriqa_phantomqa.pl, and "
	  "is not likely to be useful to users on its own.  "
	  "This program takes a 4-D BXH- or XCEDE- wrapped dataset and "
	  "calculates and writes various QA measures, designed for fMRI "
	  "images of the BIRN calibration phantom."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'.  "
	  "Default is to ignore first 2 timepoints (2:), or 3 if the total "
	  "number of timepoints is odd." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Chooses the slice number on which to compute the statistics.  "
	  "Must be a single unsigned integer within the range "
	  "0 <= x <= (numslices-1).  Default is middle slice." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_summaryonly, 1, "summaryonly",
	  "Don't generate ave, nave, std images." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_nofluct, 1, "nofluct",
	  "Don't run fluctuation analysis." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_noroi, 1, "noroi",
	  "Don't run ROI-based analysis." },
	{ OPT_FLAGS_FULL, OPT_VAL_UINT, &opt_roisize, 1, "roisize",
	  "Override the default ROI size of 30x30 (for 128x128 slices) "
	  "or 15x15 (for everything else).  Specify the length of the edge "
	  "of the ROI box in voxels." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "maskfile",
	  "Use this mask (should be an XML file) instead of "
	  "30x30 (for 128x128 slices) or 15x15 (or whatever is specified "
	  "by --roisize).  If 2-D, must match "
	  "slice dimensions of input data.  If 3-D, all three spatial "
	  "dimensions must match (but only slice specified in zselect "
	  "will be used)." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_forcetr, 1, "forcetr",
	  "If specified, this value (in seconds) will replace the TR "
	  "specified in the input image file, if any." }
    };

    memset(&bdr, '\0', sizeof(bdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc < 2 || argc > 3) {
	fprintf(stderr, "Usage: %s [opts] xmlfile [outputbase]\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	goto FAIL;
    }

    inputfile = argv[1];

    msbfirst = (((char *)&msbfirst)[0] == 0);

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, NULL, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    fprintf(stdout, "##orig. dimensions: %ux%ux%ux%u\n", (unsigned int)bdr.dimsizes[0], (unsigned int)bdr.dimsizes[1], (unsigned int)bdr.dimsizes[2], (unsigned int)bdr.dimsizes[3]);
    {
	char * xunits = "[units?]";
	char * yunits = xunits;
	char * zunits = xunits;
	char * tunits = xunits;
	if (bdr.datarec->dimensions[0].units) {
	    xunits = bdr.datarec->dimensions[0].units;
	}
	if (bdr.datarec->dimensions[1].units) {
	    yunits = bdr.datarec->dimensions[1].units;
	}
	if (bdr.datarec->dimensions[2].units) {
	    zunits = bdr.datarec->dimensions[2].units;
	}
	if (bdr.datarec->dimensions[3].units) {
	    tunits = bdr.datarec->dimensions[3].units;
	}
	fprintf(stdout, "##orig. spacing: %g%sx%g%sx%g%sx%g%s\n", bdr.datarec->dimensions[0].spacing, xunits, bdr.datarec->dimensions[1].spacing, yunits, bdr.datarec->dimensions[2].spacing, zunits, bdr.datarec->dimensions[3].spacing, tunits);
	fprintf(stdout, "##orig. gap: %g%sx%g%sx%g%sx%g%s\n", bdr.datarec->dimensions[0].gap, xunits, bdr.datarec->dimensions[1].gap, yunits, bdr.datarec->dimensions[2].gap, zunits, bdr.datarec->dimensions[3].gap, tunits);
    }
    bxh_datareaddata_free(&bdr);

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (bdr.datarec->numdims != 4) {
	fprintf(stderr, "Data must be 4-dimensional.\n");
	goto FAIL;
    }
    if (bdr.datarec->dimensions[3].size == 0) {
	fprintf(stderr, "Number of time points must be greater than 0!\n");
	goto FAIL;
    }

    /* grab timepoint indices before bdr dimensions change */
    timepoints = bxh_datarec_constructSelector(opt_select[3], 0, bdr.datarec->dimensions[3].size-1, NULL);
	
    {
	int reread = 0;
	if (opt_select[2] == NULL) {
	    opt_select[2] = &zselectbuf[0];
	    sprintf(&zselectbuf[0], "%u", (unsigned int)((bdr.datarec->dimensions[2].size / 2.0) + 0.51 - 1));
	    reread = 1;
	}
	if (opt_select[3] == NULL) {
	    opt_select[3] = &tselectbuf[0];
	    sprintf(&tselectbuf[0], "%u:", (unsigned int)(2 + (bdr.datarec->dimensions[3].size % 2)));
	    if (bdr.datarec->dimensions[3].size <= 2) {
		fprintf(stderr, "Time selector '%s' is out of range of number of timepoints (%u)!\n", &tselectbuf[0], (unsigned int)bdr.datarec->dimensions[3].size);
		goto FAIL;
	    }
	    reread = 1;
	}
	if (reread) {
	    free(timepoints);
	    timepoints = bxh_datarec_constructSelector(opt_select[3], 0, bdr.datarec->dimensions[3].size-1, NULL);
	    bxh_datareaddata_free(&bdr); /* restart to use new z/t selectors */
	    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
		fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
		goto FAIL;
	    }
	}
    }

    if (bdr.datarec->dimensions[2].size != 1) {
	fprintf(stderr, "Z selector must select only one slice!\n");
	exit(-1);
    }
    if (bdr.datarec->dimensions[3].size == 0) {
	fprintf(stderr, "Number of time points must be greater than 0!\n");
	exit(-1);
    }
    if (bdr.datarec->dimensions[3].size % 2 != 0) {
	fprintf(stderr, "Number of time points must be divisible by 2!\n");
	exit(-1);
    }
    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, ordereddimnames, opt_select, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3 && maskbdr.datarec->numdims != 2) {
	    fprintf(stderr, "Mask must be 2- or 3-dimensional.\n");
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, bdr.dimsizes, sizeof(maskbdr.dimsizes[0]) * maskbdr.datarec->numdims) != 0) {
	    fprintf(stderr, "Mask spatial dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
    }
    if (bxh_dataReadFinish(&bdr, "double") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (opt_maskfile) {
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
    }

    dataptr = (double *)bdr.dataptr;

    /*** Do the dirty work here ***/
    {
	size_t nTimePts;
	size_t R;
	size_t npox2;
	size_t npoy2;
	size_t r1, r2;
	char * mask = NULL;
	double * roi = NULL;  /* ROI mean for each time pt (largest radius) */
	double * roir = NULL; /* ROI mean for each time point and radius */
	double * Iodd = NULL; /* Sum of odd images */
	double * Ieven = NULL; /* Sum of even images */
	float * Isub = NULL; /* Difference image */
	float * Sy = NULL; /* Sum image */
	float * Iave = NULL; /* Average image */
	float * Isd = NULL;
	float * sfnr = NULL;
	
	double TR;
	BXHElementPtr trelemp = NULL;

	size_t indx, indy, indt, indxy, r;

	int outnum;
	char * outputbxh = NULL;
	char * outputimg = NULL;
	float * outputdata = NULL;

	fprintf(stdout, "##Using slice %s\n", opt_select[2]);
	fprintf(stdout, "##Using time points %s\n", opt_select[3]);

	TR = opt_forcetr;
	if (opt_forcetr == 0) {
	    if ((trelemp = bxh_getChildElement(bdr.acqdatap, "tr")) == NULL ||
		bxh_getElementDoubleValue(trelemp, &TR) != 0) {
		if ((TR = bdr.datarec->dimensions[3].spacing) == 0) {
		    fprintf(stderr, "Unable to read TR or time dimension spacing from image header!\n");
		    goto FAIL;
		}
	    }
	    TR /= 1000.0; /* convert to seconds */
	}

	nTimePts = bdr.dimsizes[3];
	if (opt_roisize != (unsigned int)-1) {
	    R = opt_roisize;
	} else {
	    if (bdr.dimsizes[0] == 128)
		R = 30;
	    else
		R = 15;
	}
	npox2 = bdr.dimsizes[0]/2;
	npoy2 = bdr.dimsizes[1]/2;
	r1 = 1;
	r2 = R;
	
	fprintf(stdout, "##Using ROI size %u\n", (unsigned int)R);

	if (opt_maskfile) {
	    mask = maskbdr.dataptr;
	} else {
	    /* create a mask from the largest ROI */
	    mask = (char *)malloc(sizeof(char)*bdr.pagesizes[1]);
	    memset(mask, '\0', sizeof(char)*bdr.pagesizes[1]);
	    {
		size_t X1,Y1;
		X1 = npox2 - (R / 2) - 1;
		Y1 = npoy2 - (R / 2) - 1;
		for (indy = Y1; indy < Y1 + R; indy++) {
		    size_t pageposy = indy * bdr.dimsizes[0];
		    for (indx = X1; indx < X1 + R; indx++) {
			mask[pageposy + indx] = 1;
		    }
		}
	    }
	}

	roi = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
	roir = (double *)malloc(sizeof(double)*bdr.dimsizes[3]*(r2-r1+1));
	memset(roi, '\0', sizeof(double)*bdr.dimsizes[3]);
	memset(roir, '\0', sizeof(double)*bdr.dimsizes[3]*(r2-r1+1));

	Iodd = (double *)malloc(sizeof(double)*bdr.pagesizes[1]);
	Ieven = (double *)malloc(sizeof(double)*bdr.pagesizes[1]);
	memset(Iodd, '\0', sizeof(double)*bdr.pagesizes[1]);
	memset(Ieven, '\0', sizeof(double)*bdr.pagesizes[1]);
	
	for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
	    size_t pagepost = indt * bdr.pagesizes[1];
	    for (indxy = 0; indxy < bdr.pagesizes[1]; indxy++) {
		double val = dataptr[pagepost + indxy];
		/* use 1-based indexing like Matlab to determine odd/even */
		if (indt % 2 == 0) {
		    Iodd[indxy] += val;
		} else {
		    Ieven[indxy] += val;
		}
		if (mask[indxy]) {
		    roi[indt] += val;
		}
	    }
	    roi[indt] /= (R*R); /* mean of largest ROI */
	    for (r = r1; r <= r2; r++) {
		size_t x1 = npox2 - (r / 2) - 1;
		size_t y1 = npoy2 - (r / 2) - 1;
		size_t roirind = ((r-1) * bdr.dimsizes[3]) + indt;
		for (indy = y1; indy < y1 + r; indy++) {
		    size_t pageposy = pagepost + (indy * bdr.pagesizes[0]);
		    for (indx = x1; indx < x1 + r; indx++) {
			double val = dataptr[pageposy + indx];
			roir[roirind] += val;
		    }
		}
		roir[roirind] /= (r * r);
	    }
	}

	/* --- Calculate stat images (mean, difference, std, sfnr) --- */
	Isub = (float *)malloc(sizeof(float)*bdr.pagesizes[1]);
	Sy = (float *)malloc(sizeof(float)*bdr.pagesizes[1]);
	Iave = (float *)malloc(sizeof(float)*bdr.pagesizes[1]);
	for (indy = 0; indy < bdr.dimsizes[1]; indy++) {
	    size_t pageposy = indy * bdr.dimsizes[0];
	    for (indx = 0; indx < bdr.dimsizes[0]; indx++) {
		size_t pageposx = pageposy + indx;
		Isub[pageposx] = (float)(Iodd[pageposx] - Ieven[pageposx]);
		Sy[pageposx] = (float)(Iodd[pageposx] + Ieven[pageposx]);
		Iave[pageposx] = (float)(Sy[pageposx] / bdr.dimsizes[3]);
	    }
	}

	free(Iodd); Iodd = NULL;
	free(Ieven); Ieven = NULL;


	/* Calculate Std Dev of residuals of 2nd-order polynomial fit */
	Isd = (float *)malloc(sizeof(float)*bdr.pagesizes[1]);
	sfnr = (float *)malloc(sizeof(float)*bdr.pagesizes[1]);
	{
	    gsl_matrix * gslX = NULL;
	    gsl_vector * gsly = NULL;
	    gsl_vector * gslc = NULL;
	    gsl_matrix * gslcov = NULL;
	    double gslchisq;
	    gsl_multifit_linear_workspace * gslwork = NULL;

	    /* set up 2nd-order polynomial fit */
	    gslX = gsl_matrix_alloc(bdr.dimsizes[3], 3);
	    gsly = gsl_vector_alloc(bdr.dimsizes[3]);
	    gslc = gsl_vector_alloc(3);
	    gslcov = gsl_matrix_alloc(3, 3);
	    gslwork = gsl_multifit_linear_alloc(bdr.dimsizes[3], 3);
	    for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		/* 1-based indexing for matrix vals like Matlab version */
		gsl_matrix_set(gslX, indt, 0, 1);
		gsl_matrix_set(gslX, indt, 1, indt+1);
		gsl_matrix_set(gslX, indt, 2, (indt+1)*(indt+1));
	    }
	    
	    for (indxy = 0; indxy < bdr.pagesizes[1]; indxy++) {
		double c0, c1, c2;
		double sum;
		double sqsum;
		/* set gsly with timeseries of this voxel */
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    gsl_vector_set(gsly, indt, dataptr[indt*bdr.pagesizes[1] + indxy]);
		}
		/* do the fit */
		gsl_multifit_linear(gslX, gsly, gslc, gslcov, &gslchisq, gslwork);
		/* grab the coefficients */
		c0 = gsl_vector_get(gslc, 0);
		c1 = gsl_vector_get(gslc, 1);
		c2 = gsl_vector_get(gslc, 2);
		/* calculate standard deviation of the residuals */
		sum = 0;
		sqsum = 0;
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    double fitval =
			(c0 * 1) +
			(c1 * (indt + 1)) +
			(c2 * (indt + 1) * (indt + 1));
		    double resid =
			dataptr[indt*bdr.pagesizes[1] + indxy] - fitval;
		    sum += resid;
		    sqsum += resid * resid;
		}
		Isd[indxy] = (float)sqrt((sqsum - ((sum*sum)/bdr.dimsizes[3])) / bdr.dimsizes[3]);
		sfnr[indxy] = (float)(Iave[indxy] / (Isd[indxy] + FLT_EPSILON));
	    }
	    gsl_multifit_linear_free(gslwork);
	    gsl_matrix_free(gslX);
	    gsl_vector_free(gsly);
	    gsl_vector_free(gslc);
	    gsl_matrix_free(gslcov);
	}

	/* extract some header info and print out */
	{
	    BXHElementPtr * elemarray = NULL;
	    if ((elemarray = bxh_getChildElementArray(bdr.acqdatap, "*")) != NULL) {
		BXHElementPtr * curelempp = elemarray;
		while (*curelempp) {
		    BXHElementPtr curelemp = *curelempp;
		    char * name = NULL;
		    char * value = NULL;
		    name = bxh_getElementName(curelemp);
		    if (name != NULL && strcmp(name, "acqParam") == 0) {
			free(name); name = NULL;
			name = bxh_getAttributeStringValue(curelemp, "name");
		    }
		    if (name != NULL) {
			if (bxh_getElementStringValue(curelemp, &value) == 0) {
			    char * newline = value;
			    while ((newline = strchr(newline, '\n')) != NULL) {
				*newline = '|';
				newline++;
			    }
			    fprintf(stdout, "##acquisitiondata:%s = %s\n", name, value);
			}
		    }
		    free(name); name = NULL;
		    free(value); value = NULL;
		    bxh_element_unref(curelemp); curelemp = NULL;
		    curelempp++;
		}
		free(elemarray); elemarray = NULL;
	    }
	}


	/* create output files */
	if (!opt_summaryonly) {
	    char * ofdiff_hdr = NULL;
	    char * ofave_hdr = NULL;
	    char * ofstddev_hdr = NULL;
	    char * ofsfnr_hdr = NULL;
	    char * ofdiff_img = NULL;
	    char * ofave_img = NULL;
	    char * ofstddev_img = NULL;
	    char * ofsfnr_img = NULL;
	    char * outputbase = NULL;
	    char * extpos = NULL;
	    size_t baselen = 0;
	    if (argc > 2) {
		outputbase = strdup(argv[2]);
	    } else {
		outputbase = strdup(argv[1]);
	    }
	    extpos = strrchr(outputbase, '.');
#ifdef WIN32
	    if (extpos != NULL && extpos > strrchr(outputbase, '\\')) {
		baselen = (size_t)(extpos - outputbase);
	    } else {
		baselen = strlen(outputbase);
	    }
#else
	    if (extpos != NULL && extpos > strrchr(outputbase, '/')) {
		baselen = (size_t)(extpos - outputbase);
	    } else {
		baselen = strlen(outputbase);
	    }
#endif
	    ofdiff_hdr = (char *)malloc(sizeof(char)*(baselen+10));
	    ofave_hdr = (char *)malloc(sizeof(char)*(baselen+10));
	    ofstddev_hdr = (char *)malloc(sizeof(char)*(baselen+10));
	    ofsfnr_hdr = (char *)malloc(sizeof(char)*(baselen+10));
	    ofdiff_img = (char *)malloc(sizeof(char)*(baselen+10));
	    ofave_img = (char *)malloc(sizeof(char)*(baselen+10));
	    ofstddev_img = (char *)malloc(sizeof(char)*(baselen+10));
	    ofsfnr_img = (char *)malloc(sizeof(char)*(baselen+10));
	    outputbase[baselen] = '\0';
	    strcpy(ofdiff_hdr, outputbase);
	    strcpy(ofave_hdr, outputbase);
	    strcpy(ofstddev_hdr, outputbase);
	    strcpy(ofsfnr_hdr, outputbase);
	    strcpy(ofdiff_img, outputbase);
	    strcpy(ofave_img, outputbase);
	    strcpy(ofstddev_img, outputbase);
	    strcpy(ofsfnr_img, outputbase);
	    strcat(ofdiff_hdr, "_nave.bxh");
	    strcat(ofave_hdr, "_ave.bxh");
	    strcat(ofstddev_hdr, "_sd.bxh");
	    strcat(ofsfnr_hdr, "_sfnr.bxh");
	    strcat(ofdiff_img, "_nave.img");
	    strcat(ofave_img, "_ave.img");
	    strcat(ofstddev_img, "_sd.img");
	    strcat(ofsfnr_img, "_sfnr.img");

	    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], &inputfile, 1) != 0) {
		fprintf(stderr, "Error adding history entry\n");
		goto FAIL;
	    }
	    for (outnum = 0; outnum < 4; outnum++) {
		bxhrawdatarec * datareccopy = NULL;
		switch (outnum) {
		case 0:
		    outputbxh = ofdiff_hdr;
		    outputimg = ofdiff_img;
		    outputdata = Isub;
		    break;
		case 1:
		    outputbxh = ofave_hdr;
		    outputimg = ofave_img;
		    outputdata = Iave;
		    break;
		case 2:
		    outputbxh = ofstddev_hdr;
		    outputimg = ofstddev_img;
		    outputdata = Isd;
		    break;
		case 3:
		    outputbxh = ofsfnr_hdr;
		    outputimg = ofsfnr_img;
		    outputdata = sfnr;
		    break;
		}
		/* delete all higher order dimensions */
		datareccopy = bxh_datarec_copy(bdr.datarec);
		for (dimnum = 2; dimnum < datareccopy->numdims; dimnum++) {
		    bxh_datarec_dimdata_free(&datareccopy->dimensions[dimnum]);
		}
		datareccopy->numdims = 2;
		free(datareccopy->elemtype);
		datareccopy->elemtype = strdup("float32");
		bxh_datarec_frags_free(datareccopy);
		bxh_datarec_addfrag(datareccopy, outputimg, 0, sizeof(float) * bdr.pagesizes[1], outputbxh, 1);
		if (bxh_datarec_writeToElement(bdr.imagedatap, datareccopy) != 0) {
		    fprintf(stderr, "Failed writing datarec\n");
		    goto FAIL;
		}
		bxh_datarec_free(datareccopy);
		if (bxh_writeFile(bdr.docp, outputbxh) != 0) {
		    fprintf(stderr, "Error writing output header %s\n", outputbxh);
		    goto FAIL;
		}
		/* write out results */
		if ((fp = fopen(outputimg, "wb")) == NULL) {
		    fprintf(stderr, "Error opening file %s\n", outputimg);
		    goto FAIL;
		}
		if (fwrite(outputdata, sizeof(float)*bdr.pagesizes[1], 1, fp) != 1) {
		    fprintf(stderr, "Error writing to file %s\n", outputimg);
		    goto FAIL;
		}
	    }

	    fprintf(stdout, "##Difference image written to %s\n", ofdiff_hdr);
	    fprintf(stdout, "##Mean image written to %s\n", ofave_hdr);
	    fprintf(stdout, "##StdDev image written to %s\n", ofstddev_hdr);
	    fprintf(stdout, "##SFNR image written to %s\n", ofsfnr_hdr);

	    free(ofdiff_hdr);
	    free(ofave_hdr);
	    free(ofstddev_hdr);
	    free(ofsfnr_hdr);
	    free(ofdiff_img);
	    free(ofave_img);
	    free(ofstddev_img);
	    free(ofsfnr_img);

	    free(outputbase);
	}

	/* --- Calculate and display summary values --- */
	/* varI = var(Isub(mask)); */
	/* meanI = mean(Iave(mask)); */
	/* sfnrI = mean(sfnr(mask)); */
	/* snr = meanI/sqrt(varI/nTimePts); */
	{
	    static char summarystring[1024];
	    double varI = 0;
	    double meanI = 0;
	    double sfnrI = 0;
	    double masksize = 0;
	    double submean = 0;
	    double snr;
	    for (indxy = 0; indxy < bdr.pagesizes[1]; indxy++) {
		if (mask[indxy] == 0) continue;
		masksize += 1;
	    }
	    for (indxy = 0; indxy < bdr.pagesizes[1]; indxy++) {
		if (mask[indxy] == 0) continue;
		submean += Isub[indxy];
	    }
	    submean /= masksize;
	    for (indxy = 0; indxy < bdr.pagesizes[1]; indxy++) {
		if (mask[indxy] == 0) continue;
		varI += (Isub[indxy] - submean) * (Isub[indxy] - submean);
	    }
	    varI /= (masksize - 1);
	    for (indxy = 0; indxy < bdr.pagesizes[1]; indxy++) {
		if (mask[indxy] == 0) continue;
		meanI += Iave[indxy];
	    }
	    meanI /= masksize;
	    for (indxy = 0; indxy < bdr.pagesizes[1]; indxy++) {
		if (mask[indxy] == 0) continue;
		sfnrI += sfnr[indxy];
	    }
	    sfnrI /= masksize;
	    snr = meanI/sqrt(varI/bdr.dimsizes[3]);
	    sprintf(&summarystring[0], "(mean, SNR, SFNR) = (%5.1f  %5.1f  %5.1f)", meanI, snr, sfnrI);
	    fprintf(stdout, "##%s\n", summarystring);
	}

	free(Isub); Isub = NULL;
	free(Sy); Sy = NULL;
	free(Iave); Iave = NULL;
	free(Isd); Isd = NULL;
	free(sfnr); sfnr = NULL;

	/* Do fluctuation analysis */
	if (!opt_nofluct) {
	    double * y = NULL;
	    double * yfit = NULL;
	    double m;
	    double sd;
	    double drift;
	    double driftfit;

	    yfit = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
	    y = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
	    {
		gsl_matrix * gslX = NULL;
		gsl_vector * gsly = NULL;
		gsl_vector * gslc = NULL;
		gsl_matrix * gslcov = NULL;
		double gslchisq;

		gslX = gsl_matrix_alloc(bdr.dimsizes[3], 3);
		gsly = gsl_vector_alloc(bdr.dimsizes[3]);
		gslc = gsl_vector_alloc(3);
		gslcov = gsl_matrix_alloc(3, 3);
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    /* 1-based indexing for matrix vals like Matlab version */
		    gsl_matrix_set(gslX, indt, 0, 1);
		    gsl_matrix_set(gslX, indt, 1, indt+1);
		    gsl_matrix_set(gslX, indt, 2, (indt+1)*(indt+1));
		    gsl_vector_set(gsly, indt, roi[indt]);
		}
		{
		    gsl_multifit_linear_workspace * work = NULL;
		    work = gsl_multifit_linear_alloc(bdr.dimsizes[3], 3);
		    gsl_multifit_linear(gslX, gsly, gslc, gslcov, &gslchisq, work);
		    gsl_multifit_linear_free(work);
		}
	
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    yfit[indt] =
			(gsl_vector_get(gslc, 0) * 1)
			+ (gsl_vector_get(gslc, 1) * (indt + 1))
			+ (gsl_vector_get(gslc, 2) * (indt + 1) * (indt + 1));
		    y[indt] = roi[indt] - yfit[indt];
		}
		gsl_matrix_free(gslX);
		gsl_vector_free(gsly);
		gsl_vector_free(gslc);
		gsl_matrix_free(gslcov);
	    }

	    m = 0;
	    for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		m += roi[indt];
	    }
	    m /= bdr.dimsizes[3];

	    {
		double ym = 0;
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    ym += y[indt];
		}
		ym /= bdr.dimsizes[3];
		sd = 0;
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    sd += (y[indt] - ym) * (y[indt] - ym);
		}
		sd /= (bdr.dimsizes[3] - 1);
		sd = sqrt(sd);
	    }

	    {
		double minroi, maxroi;
		minroi = roi[0];
		maxroi = roi[0];
		for (indt = 1; indt < bdr.dimsizes[3]; indt++) {
		    if (roi[indt] < minroi) minroi = roi[indt];
		    if (roi[indt] > maxroi) maxroi = roi[indt];
		}
		drift = (maxroi - minroi) / m;
	    }
	    {
		double minfit, maxfit;
		minfit = yfit[0];
		maxfit = yfit[0];
		for (indt = 1; indt < bdr.dimsizes[3]; indt++) {
		    if (yfit[indt] < minfit) minfit = yfit[indt];
		    if (yfit[indt] > maxfit) maxfit = yfit[indt];
		}
		driftfit = (maxfit - minfit) / m;
	    }
	    fprintf(stdout, "##(std, percent fluc, drift, driftfit) = (%5.2f  %6.2f %6.2f %6.2f)\n",
		    sd, 100 * sd / m, 100 * drift, 100 * driftfit);
	    fprintf(stdout, "#FrameNum RawSignal(ROI) RawSignal(Fit)\n");
	    for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		fprintf(stdout, "%lu %.10g %.10g\n",
			(unsigned long)(indt+1), roi[indt], yfit[indt]);
	    }
	    

	    {
		double fs = 1.0 / TR;
		size_t nf = (bdr.dimsizes[3] / 2) + 1;
		double * z = NULL;
		z = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    z[indt] = y[indt];
		}
		{
		    gsl_fft_real_wavetable * wavetable = NULL;
		    gsl_fft_real_workspace * work = NULL;
		    wavetable = gsl_fft_real_wavetable_alloc(bdr.dimsizes[3]);
		    work = gsl_fft_real_workspace_alloc(bdr.dimsizes[3]);
		    gsl_fft_real_transform(z, 1, bdr.dimsizes[3], wavetable, work);
		    gsl_fft_real_wavetable_free(wavetable);
		    gsl_fft_real_workspace_free(work);
		}
		fprintf(stdout, "#frequency(Hz) spectrum(mean_scaled)\n");
		/* start at 1 to ignore DC */
		for (indt = 1; indt < nf; indt++) {
		    double real;
		    double imag;
		    if (indt == 0) {
			real = z[0];
			imag = 0;
		    } else if ((bdr.dimsizes[3] % 2 == 0) && indt == nf - 1) {
			real = z[bdr.dimsizes[3]-1];
			imag = 0;
		    } else {
			real = z[(indt * 2) - 1];
			imag = z[indt * 2];
		    }
		    fprintf(stdout, "%g %g\n",
			    0.5*indt*fs/(nf-1), /* first (indt==0) is 0, last (indt==nf-1) is 0.5/TR */
			    sqrt((real * real) + (imag * imag)) * 100.0 / m); /* scaled by mean intensity across ROI across time */
		}
		free(z);
	    }
	    free(y);
	    free(yfit);
	}
    
	/* now do analysis for each roi size */
	if (!opt_noroi) {
	    double * y = NULL;
	    double * yfit = NULL;
	    double * F = NULL;
	    double * fcalc = NULL;
	    double rdc;

	    gsl_matrix * gslX = NULL;
	    gsl_vector * gsly = NULL;
	    gsl_vector * gslc = NULL;
	    gsl_matrix * gslcov = NULL;
	    double gslchisq;

	    gsl_multifit_linear_workspace * work = NULL;

	    work = gsl_multifit_linear_alloc(bdr.dimsizes[3], 3);

	    yfit = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
	    y = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
	    F = (double *)malloc(sizeof(double)*(r2-r1+1));
	    fcalc = (double *)malloc(sizeof(double)*(r2-r1+1));

	    gslX = gsl_matrix_alloc(bdr.dimsizes[3], 3);
	    gsly = gsl_vector_alloc(bdr.dimsizes[3]);
	    gslc = gsl_vector_alloc(3);
	    gslcov = gsl_matrix_alloc(3, 3);
	    for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		gsl_matrix_set(gslX, indt, 0, 1);
		gsl_matrix_set(gslX, indt, 1, indt+1);
		gsl_matrix_set(gslX, indt, 2, (indt+1)*(indt+1));
	    }

	    for (r = r1; r <= r2; r++) {
		double m;
		double m2;
		double sd2;
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    gsl_vector_set(gsly, indt, roir[((r-1) * bdr.dimsizes[3]) + indt]);
		}
		gsl_multifit_linear(gslX, gsly, gslc, gslcov, &gslchisq, work);
	
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    y[indt] = gsl_vector_get(gsly, indt);
		    yfit[indt] =
			(gsl_vector_get(gslc, 0) * 1)
			+ (gsl_vector_get(gslc, 1) * (indt + 1))
			+ (gsl_vector_get(gslc, 2) * (indt + 1) * (indt + 1));
		    y[indt] -= yfit[indt];
		}
	    
		/* F(r) = 100 * std(y - yfit)/mean(yfit); */
		m = 0; /* mean(yfit) */
		m2 = 0; /* mean(y - yfit) */
		/* [y now actually contains (y - yfit)] */
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    m += yfit[indt];
		    m2 += y[indt];
		}
		m /= bdr.dimsizes[3];
		m2 /= bdr.dimsizes[3];
		sd2 = 0; /* std(y - yfit) */
		for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		    double dev = y[indt] - m2;
		    sd2 += dev * dev;
		}
		sd2 /= (bdr.dimsizes[3] - 1);
		sd2 = sqrt(sd2);
		F[r-1] = 100.0 * (sd2 / m); /* percent */
	    }

	    for (r = r1; r <= r2; r++) {
		fcalc[r-1] = F[0] / r;
	    }
	    rdc = F[0] / F[r2-1];

	    fprintf(stdout, "##rdc = %3.1f pixels\n", rdc);
	    fprintf(stdout, "#ROIFullWidth(pixels) MeasuredRelativeSTD(percent)\n");
	    for (r = r1; r <= r2; r++) {
		fprintf(stdout, "%g %g\n", (double)r, (double)F[r-1]);
	    }
	    fprintf(stdout, "#ROIFullWidth(pixels) CalculatedRelativeSTD(percent)\n");
	    for (r = r1; r <= r2; r++) {
		fprintf(stdout, "%g %g\n", (double)r, (double)fcalc[r-1]);
	    }
	
	    free(yfit);
	    free(y);
	    free(F);
	    free(fcalc);

	    gsl_matrix_free(gslX);
	    gsl_vector_free(gsly);
	    gsl_vector_free(gslc);
	    gsl_matrix_free(gslcov);

	    gsl_multifit_linear_free(work);
	}

	if (!opt_maskfile) {
	    free(mask);
	}
	free(roi);
	free(roir);
    }
    goto EXIT;
    
  FAIL:
    retval = -1;

  EXIT:
    free(timepoints);
    bxh_datareaddata_free(&bdr);
    if (opt_maskfile)
	bxh_datareaddata_free(&maskbdr);
    return retval;
}

#endif /* #ifndef HAVE_GSL #else */

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.35  2009/01/15 20:55:17  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.34  2008/10/17 16:21:54  gadde
 * Add ROI size to output
 *
 * Revision 1.33  2007/02/27 18:18:53  gadde
 * Deal better with acqParam objects.
 *
 * Revision 1.32  2007/01/15 20:15:13  gadde
 * Fix reported dimensions.
 *
 * Revision 1.31  2007/01/12 16:22:50  gadde
 * Put ROI back to 15x15.
 *
 * Revision 1.30  2007/01/11 21:15:26  gadde
 * Add acquisition data to output.
 *
 * Revision 1.29  2007/01/11 20:13:59  gadde
 * Change default ROI size
 *
 * Revision 1.28  2007/01/08 21:54:05  gadde
 * Lee's requests:
 * 1) simplify stddev of linear detrend by just doing 2nd-order polynomial fit
 * 2) calculate drift as min-max divided by mean, rather than using last-first
 * 3) fix x-axis of spectrum plot, and show spectrum as a percentage of the mean
 *
 * Revision 1.27  2006/12/05 21:29:30  gadde
 * When constructing selector, use the original ("read") datarec, rather
 * than the already selected datarec.
 *
 * Revision 1.26  2006/06/01 20:16:48  gadde
 * const fixes
 *
 * Revision 1.25  2006/04/11 17:39:38  gadde
 * Calculate "timepoints" list correctly.
 *
 * Revision 1.24  2006/04/07 14:45:53  gadde
 * Fix diagnostic message.
 *
 * Revision 1.23  2005/09/20 18:37:52  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.22  2005/09/19 16:31:53  gadde
 * Documentation and help message updates.
 *
 * Revision 1.21  2005/09/14 15:19:17  gadde
 * Some -Wall fixes.
 *
 * Revision 1.20  2005/09/14 14:49:24  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.19  2005/07/19 18:28:16  gadde
 * Fix order of statements (resulted in access of uninitialized memory.
 *
 * Revision 1.18  2005/04/20 16:49:54  gadde
 * Using dimt->spacing if non-zero and if acquisitiondata/tr doesn't exist.
 *
 * Revision 1.17  2005/03/07 19:21:22  gadde
 * Add forcetr option.
 *
 * Revision 1.16  2005/02/25 17:06:01  gadde
 * Allow X and Y dimensions to be different sizes.
 *
 * Revision 1.15  2004/12/30 16:26:18  gadde
 * Fix error message.
 *
 * Revision 1.14  2004/12/30 16:24:04  gadde
 * Add default timeselect of 2: for even number of timepoints and 3: otherwise.
 *
 * Revision 1.13  2004/12/20 19:36:21  gadde
 * Simple option validation fix.
 *
 * Revision 1.12  2004/12/20 18:56:30  gadde
 * Move to new bxh_dataReadStart/Finish interface.
 * Add option to specify mask file.
 *
 * Revision 1.11  2004/12/20 18:10:09  gadde
 * Add --nofluct and --noroi options.
 *
 * Revision 1.10  2004/06/18 15:21:50  gadde
 * Standardize frag creation (redux)
 *
 * Revision 1.9  2004/06/18 14:09:30  gadde
 * Standardize frag creation
 *
 * Revision 1.8  2004/06/15 16:16:10  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.7  2004/06/11 15:00:24  gadde
 * Add --summaryonly option.
 *
 * Revision 1.6  2004/06/10 19:54:43  gadde
 * Make sure indexing is consistent between this and Matlab version.
 *
 * Revision 1.5  2004/06/03 21:46:37  gadde
 * Output slice used.
 *
 * Revision 1.4  2004/06/03 21:30:56  gadde
 * Write out SFNR image too.
 *
 * Revision 1.3  2004/05/31 16:38:07  gadde
 * Some memory fixes
 *
 * Revision 1.2  2004/05/28 20:24:55  gadde
 * Code reorganization and memory free fix.
 *
 * Revision 1.1  2004/05/27 22:01:10  gadde
 * *** empty log message ***
 *
 *
 */
