/*
 * fmriqa_spectrum.c --
 * 
 *  Calculates a per-voxel frequency spectrum across time
 */

#include <bxh_config.h>

#include <stdio.h>

#ifndef HAVE_LIBGSL
int main(int argc, char * argv[])
{
    fprintf(stderr, "Sorry -- to use this program, this package must be compiled with GSL\n(GNU Scientific Library) support!\n");
    return -1;
}
#else /* #ifndef HAVE_GSL */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>

#include <gsl/gsl_multifit.h>
#include <gsl/gsl_fft_real.h>

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    int oldargc = argc;
    struct stat statbuf;
    const char * inputfile = NULL;
    int * timepoints = NULL;
    const char * opt_select[] = { ":", ":", ":", ":" };
    char tselectbuf[16];
    double opt_forcetr = 0;
    char * opt_maskfile = NULL;
    int opt_version = 0;
    size_t indt;
    
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    const char * mask_select[] = { ":", ":", ":" };
    struct bxhdataread bdr;
    struct bxhdataread maskbdr;
    double * dataptr = NULL;
    char * maskdataptr = NULL;

    char * outputfile = NULL;
    char * outputbxh = NULL;
    char * outputbase = NULL;
    char * extpos = NULL;

    const int numopts = 6;
    opt_data opts[6] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_spectrum [opts] xmlfile outputbase\n\n"
	  "This program takes a 4-D BXH- or XCEDE- wrapped dataset and "
	  "calculates and writes a 4-D image representing the frequency "
	  "spectrum for each voxel.  Fourth dimension (theta) in the output "
	  "will represent frequency."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'.  "
	  "Default is to ignore first 2 timepoints (2:), or 3 if the total "
	  "number of timepoints is odd.  Selected timepoints must all be "
	  "contiguous."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "maskfile",
	  "Use this 3-D mask.  Values outside the mask will be 0.  "
	  "Sizes of all three spatial dimensions in the mask must match that "
	  "of the input data."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_forcetr, 1, "forcetr",
	  "If specified, this value (in seconds) will replace the TR "
	  "specified in the input image file, if any." }
    };

    memset(&bdr, '\0', sizeof(bdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 3) {
	fprintf(stderr, "Usage: %s [opts] xmlfile outputbase\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	goto FAIL;
    }

    inputfile = argv[1];
    outputbase = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 1));
    outputfile = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 8));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 5));
    strcpy(outputbase, argv[argc-1]);
    extpos = strrchr(outputbase, '.');
    if (extpos != NULL) {
	/* output file already ends in .bxh */
	*extpos = '\0';
	strcpy(outputfile, outputbase);
	strcat(outputfile, ".nii.gz");
	strcpy(outputbxh, outputbase);
	strcat(outputbxh, ".bxh");
    } else {
	strcpy(outputfile, outputbase);
	strcat(outputfile, ".nii.gz");
	strcpy(outputbxh, outputbase);
    }
    if (stat(outputfile, &statbuf) == 0) {
	fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputfile);
	return -1;
    }
    if (stat(outputbxh, &statbuf) == 0) {
	fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputbxh);
	return -1;
    }

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (bdr.datarec->numdims != 4) {
	fprintf(stderr, "Data must be 4-dimensional.\n");
	goto FAIL;
    }
    if (bdr.datarec->dimensions[3].size == 0) {
	fprintf(stderr, "Number of time points must be greater than 0!\n");
	goto FAIL;
    }
    if (bdr.datarec->dimensions[3].size % 2 != 0) {
	fprintf(stderr, "Number of time points must be divisible by 2!\n");
	exit(-1);
    }
    /* grab timepoint indices from before timepoint selection */
    timepoints = bxh_datarec_constructSelector(opt_select[3], 0, bdr.datarec->dimensions[3].size-1, NULL);
    for (indt = 1; indt < bdr.datarec->dimensions[3].size; indt++) {
	if (timepoints[indt] - timepoints[indt-1] != 1) {
	    fprintf(stderr, "Selected timepoints must be monotonically increasing and contiguous.  (%d and %d are not, selector=%s, dimsize=%d)\n", (int)timepoints[indt-1], (int)timepoints[indt], (opt_select[3] == NULL) ? "<null>" : opt_select[3], bdr.datarec->dimensions[3].size-1);
	    goto FAIL;
	}
    }
    free(timepoints); timepoints = NULL;

    {
	/* If we need to use default time selection, set it here, and re-start data read */
	int reread = 0;
	if (opt_select[3] == NULL) {
	    opt_select[3] = &tselectbuf[0];
	    sprintf(&tselectbuf[0], "%u:", (unsigned int)(2 + (bdr.datarec->dimensions[3].size % 2)));
	    if (bdr.datarec->dimensions[3].size <= 2) {
		fprintf(stderr, "Default time selector '%s' is out of range of number of timepoints (%u)!\n", &tselectbuf[0], (unsigned int)bdr.datarec->dimensions[3].size);
		goto FAIL;
	    }
	    fprintf(stderr, "Using default time selector '%s'\n", &tselectbuf[0]);
	    reread = 1;
	}
	if (reread) {
	    bxh_datareaddata_free(&bdr); /* restart to use new z/t selectors */
	    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
		fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
		goto FAIL;
	    }
	}
    }

    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, ordereddimnames, mask_select, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be 3-dimensional.\n");
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, bdr.dimsizes, sizeof(maskbdr.dimsizes[0]) * 3) != 0) {
	    fprintf(stderr, "Mask spatial dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
    }

    if (bxh_dataReadFinish(&bdr, "double") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (opt_maskfile) {
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
    }

    dataptr = (double *)bdr.dataptr;
    maskdataptr = (char *)maskbdr.dataptr;

    /*** Do the dirty work here ***/
    {
	double TR;
	BXHElementPtr trelemp = NULL;
	double * results = NULL;
	double * fftdata = NULL;
	double * seriesfit = NULL;
	gsl_matrix * gslX = NULL;
	gsl_vector * gsly = NULL;
	gsl_vector * gslc = NULL;
	gsl_matrix * gslcov = NULL;
	double gslchisq;
	gsl_multifit_linear_workspace * fitwork = NULL;
	gsl_fft_real_wavetable * wavetable = NULL;
	gsl_fft_real_workspace * fftwork = NULL;

	size_t nf = (bdr.dimsizes[3] / 2) + 1;
	size_t indxyz;
	bxhrawdatarec * outdatarec = NULL;

	TR = opt_forcetr;
	if (opt_forcetr == 0) {
	    if ((trelemp = bxh_getChildElement(bdr.acqdatap, "tr")) == NULL ||
		bxh_getElementDoubleValue(trelemp, &TR) != 0) {
		if ((TR = bdr.datarec->dimensions[3].spacing) == 0) {
		    fprintf(stderr, "Unable to read TR or time dimension spacing from image header!\n");
		    goto FAIL;
		}
	    }
	    TR /= 1000.0; /* convert to seconds */
	}

	/* since we do one voxel at a time, output can go (destructively)
	 * back into dataptr to save memory */
	results = dataptr;
	seriesfit = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
	fitwork = gsl_multifit_linear_alloc(bdr.dimsizes[3], 3);
	gslX = gsl_matrix_alloc(bdr.dimsizes[3], 3);
	gsly = gsl_vector_alloc(bdr.dimsizes[3]);
	gslc = gsl_vector_alloc(3);
	gslcov = gsl_matrix_alloc(3, 3);
	fftdata = (double *)malloc(sizeof(double)*bdr.dimsizes[3]);
	wavetable = gsl_fft_real_wavetable_alloc(bdr.dimsizes[3]);
	fftwork = gsl_fft_real_workspace_alloc(bdr.dimsizes[3]);

	/* set up linear and quadratic arrays */
	for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
	    gsl_matrix_set(gslX, indt, 0, 1);
	    gsl_matrix_set(gslX, indt, 1, indt+1);
	    gsl_matrix_set(gslX, indt, 2, (indt+1)*(indt+1));
	}

	for (indxyz = 0; indxyz < bdr.pagesizes[2]; indxyz++) {
	    if (maskdataptr && maskdataptr[indxyz] == 0) {
		for (indt = 1; indt < nf; indt++) {
		    results[((indt - 1) * bdr.pagesizes[2]) + indxyz] = 0;
		}
		continue;
	    }
	    /* remove any linear and quadratic drift */
	    for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		gsl_vector_set(gsly, indt, dataptr[(indt * bdr.pagesizes[2]) + indxyz]);
	    }
	    gsl_multifit_linear(gslX, gsly, gslc, gslcov, &gslchisq, fitwork);
	    for (indt = 0; indt < bdr.dimsizes[3]; indt++) {
		seriesfit[indt] =
		    (gsl_vector_get(gslc, 0) * 1) +
		    (gsl_vector_get(gslc, 1) * (indt + 1)) +
		    (gsl_vector_get(gslc, 2) * (indt + 1) * (indt + 1));
		fftdata[indt] = gsl_vector_get(gsly, indt) - seriesfit[indt];
	    }
	    gsl_fft_real_transform(fftdata, 1, bdr.dimsizes[3], wavetable, fftwork);
	    /* skip DC (indt==0) */
	    for (indt = 1; indt < nf; indt++) {
		double real;
		double imag;
		if (indt == 0) {
		    real = fftdata[0];
		    imag = 0;
		} else if ((bdr.dimsizes[3] % 2 == 0) && indt == nf - 1) {
		    real = fftdata[bdr.dimsizes[3]-1];
		    imag = 0;
		} else {
		    real = fftdata[(indt * 2) - 1];
		    imag = fftdata[indt * 2];
		}
		results[((indt-1) * bdr.pagesizes[2]) + indxyz] = sqrt((real * real) + (imag * imag));
	    }
	}

	free(seriesfit);
	free(fftdata); fftdata = NULL;
	gsl_fft_real_wavetable_free(wavetable); wavetable = NULL;
	gsl_fft_real_workspace_free(fftwork); fftwork = NULL;
	gsl_matrix_free(gslX);
	gsl_vector_free(gsly);
	gsl_vector_free(gslc);
	gsl_matrix_free(gslcov);
	
	gsl_multifit_linear_free(fitwork); fitwork = NULL;

	/* valid data in dataptr/results now has dimensions
	 * nx*ny*nz*((nt/2)+1) */

	/* write out results */
	{
	    double fs = 1.0 / TR;
	    float * tempresults = NULL;
	    size_t numelems = bdr.pagesizes[2]*(nf-1);
	    size_t i;
	    /* convert data to float in place -- very slight possibility
	     * that this is dangerous, but alignment *should* work */
	    tempresults = (float *)results;
	    for (i = 0; i < numelems; i++) {
		float val = (float)results[i];
		tempresults[i] = val;
	    }
	    outdatarec = bxh_datarec_copy(bdr.datarec);
	    free(outdatarec->elemtype);
	    outdatarec->elemtype = strdup("float32");
	    bxh_datarec_dimdata_free(&outdatarec->dimensions[3]);
	    memset(&outdatarec->dimensions[3], '\0', sizeof(outdatarec->dimensions[3]));
	    outdatarec->dimensions[3].type = strdup("theta");
	    outdatarec->dimensions[3].units = strdup("Hz");
	    outdatarec->dimensions[3].size = nf-1;
	    outdatarec->dimensions[3].origin = 0.5 * fs / (nf - 1);
	    outdatarec->dimensions[3].spacing = 0.5 * fs / (nf - 1);
	    bxh_datarec_frags_free(outdatarec);
	    bxh_datarec_addfrag(outdatarec, outputfile, 0, sizeof(float) * bdr.pagesizes[3], outputbxh, 1);
	    if (bxh_datarec_writeToElement(bdr.imagedatap, outdatarec) != 0) {
		fprintf(stderr, "Failed writing datarec\n");
		bxh_datarec_free(outdatarec);
		goto FAIL;
	    }
	    bxh_datarec_free(bdr.datarec);
	    bdr.datarec = outdatarec;
	    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], (const char **)&argv[1], oldargc - 1) != 0) {
		fprintf(stderr, "Error adding history entry\n");
		goto FAIL;
	    }
	    writeBXHAndNIIGZ(outputbase, &bdr, tempresults, 0);
	}
    }
    goto EXIT;
    
  FAIL:
    retval = -1;

  EXIT:
    if (timepoints) {
	free(timepoints);
	timepoints = NULL;
    }
    bxh_datareaddata_free(&bdr);
    if (opt_maskfile)
	bxh_datareaddata_free(&maskbdr);
    if (outputbase)
	free(outputbase);
    if (outputbxh)
	free(outputbxh);
    if (outputfile)
	free(outputfile);
    return retval;
}

#endif /* #ifndef HAVE_GSL #else */

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.35  2009/01/15 20:55:17  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.34  2008/10/17 16:21:54  gadde
 * Add ROI size to output
 *
 * Revision 1.33  2007/02/27 18:18:53  gadde
 * Deal better with acqParam objects.
 *
 * Revision 1.32  2007/01/15 20:15:13  gadde
 * Fix reported dimensions.
 *
 * Revision 1.31  2007/01/12 16:22:50  gadde
 * Put ROI back to 15x15.
 *
 * Revision 1.30  2007/01/11 21:15:26  gadde
 * Add acquisition data to output.
 *
 * Revision 1.29  2007/01/11 20:13:59  gadde
 * Change default ROI size
 *
 * Revision 1.28  2007/01/08 21:54:05  gadde
 * Lee's requests:
 * 1) simplify stddev of linear detrend by just doing 2nd-order polynomial fit
 * 2) calculate drift as min-max divided by mean, rather than using last-first
 * 3) fix x-axis of spectrum plot, and show spectrum as a percentage of the mean
 *
 * Revision 1.27  2006/12/05 21:29:30  gadde
 * When constructing selector, use the original ("read") datarec, rather
 * than the already selected datarec.
 *
 * Revision 1.26  2006/06/01 20:16:48  gadde
 * const fixes
 *
 * Revision 1.25  2006/04/11 17:39:38  gadde
 * Calculate "timepoints" list correctly.
 *
 * Revision 1.24  2006/04/07 14:45:53  gadde
 * Fix diagnostic message.
 *
 * Revision 1.23  2005/09/20 18:37:52  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.22  2005/09/19 16:31:53  gadde
 * Documentation and help message updates.
 *
 * Revision 1.21  2005/09/14 15:19:17  gadde
 * Some -Wall fixes.
 *
 * Revision 1.20  2005/09/14 14:49:24  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.19  2005/07/19 18:28:16  gadde
 * Fix order of statements (resulted in access of uninitialized memory.
 *
 * Revision 1.18  2005/04/20 16:49:54  gadde
 * Using dimt->spacing if non-zero and if acquisitiondata/tr doesn't exist.
 *
 * Revision 1.17  2005/03/07 19:21:22  gadde
 * Add forcetr option.
 *
 * Revision 1.16  2005/02/25 17:06:01  gadde
 * Allow X and Y dimensions to be different sizes.
 *
 * Revision 1.15  2004/12/30 16:26:18  gadde
 * Fix error message.
 *
 * Revision 1.14  2004/12/30 16:24:04  gadde
 * Add default timeselect of 2: for even number of timepoints and 3: otherwise.
 *
 * Revision 1.13  2004/12/20 19:36:21  gadde
 * Simple option validation fix.
 *
 * Revision 1.12  2004/12/20 18:56:30  gadde
 * Move to new bxh_dataReadStart/Finish interface.
 * Add option to specify mask file.
 *
 * Revision 1.11  2004/12/20 18:10:09  gadde
 * Add --nofluct and --noroi options.
 *
 * Revision 1.10  2004/06/18 15:21:50  gadde
 * Standardize frag creation (redux)
 *
 * Revision 1.9  2004/06/18 14:09:30  gadde
 * Standardize frag creation
 *
 * Revision 1.8  2004/06/15 16:16:10  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.7  2004/06/11 15:00:24  gadde
 * Add --summaryonly option.
 *
 * Revision 1.6  2004/06/10 19:54:43  gadde
 * Make sure indexing is consistent between this and Matlab version.
 *
 * Revision 1.5  2004/06/03 21:46:37  gadde
 * Output slice used.
 *
 * Revision 1.4  2004/06/03 21:30:56  gadde
 * Write out SFNR image too.
 *
 * Revision 1.3  2004/05/31 16:38:07  gadde
 * Some memory fixes
 *
 * Revision 1.2  2004/05/28 20:24:55  gadde
 * Code reorganization and memory free fix.
 *
 * Revision 1.1  2004/05/27 22:01:10  gadde
 * *** empty log message ***
 *
 *
 */
