static const char rcsid[] = "$Id: bxh_convert_analyze.c,v 1.1 2009-01-15 20:55:18 gadde Exp $";

/*
 * analyze2bxh.cc --
 * 
 * Creates a BXH file based on given Analyze 7.5/SPM file.
 */

#include "bxh_utils.h"
#include "bxh_datarec.h"

#include <time.h>
#include <ctype.h>
#include <string.h>
#if defined(WIN32) && !defined(__MINGW32__)
#include <pstdint.h>
#else
#include <inttypes.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>

#include <math.h>

#include <zlib.h>

#include "nifti1_io.h"

extern char * domutil_errorbuf;

typedef struct {
    int sizeof_hdr;
    char data_type[10];
    char db_name[18];
    int extents;
    short int session_error;
    char regular;
    char hkey_un0;
} analyze_header_key;

typedef struct {
    short int dim[8];
    short int unused8;
    short int unused9;
    short int unused10;
    short int unused11;
    short int unused12;
    short int unused13;
    short int unused14;
    short int datatype;
    short int bitpix;
    short int dim_un0;
    float pixdim[8];
    float vox_offset;
    float funused1;
    float funused2;
    float funused3;
    float cal_max;
    float cal_min;
    float compressed;
    float verified;
    int glmax,glmin;
} analyze_image_dimension;

typedef struct {
    char descrip[80];
    char aux_file[24];
    char orient;
    char originator[10];
    char generated[10];
    char scannum[10];
    char patient_id[10];
    char exp_date[10];
    char exp_time[10];
    char hist_un0[3];
    int views;
    int vols_added;
    int start_field;
    int field_skip;
    int omax, omin;
    int smax, smin;
} analyze_data_history;

typedef struct {
    analyze_header_key hk;
    analyze_image_dimension dime;
    analyze_data_history hist;
} analyze_dsr;

typedef struct {
    unsigned long volume;
    unsigned long slice;
    unsigned long offset;
    unsigned long length;
    unsigned long cmp;
    unsigned long format;
} avw_slice;

typedef struct {
    char * filename;
    int datatype;
    unsigned long width;
    unsigned long height;
    unsigned long depth;
    unsigned long numvols;
    struct avw_slice * slices;
} avw_hdr;


#define DT_NONE			0
#define DT_UNKNOWN		0
#define DT_BINARY		1	
#define DT_UNSIGNED_CHAR	2
#define DT_SIGNED_SHORT		4
#define DT_SIGNED_INT		8
#define DT_FLOAT		16
#define DT_COMPLEX		32
#define DT_DOUBLE		64
#define DT_RGB			128
#define DT_SIGNED_CHAR		130 /* SPM */
#define DT_UNSIGNED_SHORT	132 /* SPM */
#define DT_UNSIGNED_INT		136 /* SPM */
#define DT_ALL			255

static int needswap = 0;

static unsigned short
swapushort(unsigned short in) {
    register unsigned short ret;
    ret = (((in & 0xff) << 8) | ((in & 0xff00) >> 8));
    return ret;
}

static unsigned int
swapuint(unsigned int in) {
    register unsigned int ret;
    ret = (((in & 0xff) << 24) |
	   ((in & 0xff00) << 8) |
	   ((in & 0xff0000) >> 8) |
	   ((in & 0xff000000) >> 24));
    return ret;
}

static int32_t
swapint32(int32_t in) {
    register int32_t ret;
    ret = (((uint32_t)(in & 0xff) << 24) |
	   ((uint32_t)(in & 0xff00) << 8) |
	   ((uint32_t)(in & 0xff0000) >> 8) |
	   ((uint32_t)(in & 0xff000000) >> 24));
    return ret;
}

static double
swapdouble(double in) {
    double ret;
    double in2 = in;
    char * inp = (char *)&in2;
    char * retp = (char *)&ret;
    retp[0] = inp[7];
    retp[1] = inp[6];
    retp[2] = inp[5];
    retp[3] = inp[4];
    retp[4] = inp[3];
    retp[5] = inp[2];
    retp[6] = inp[1];
    retp[7] = inp[0];
    return ret;
}

typedef struct {
    float real_v;
    float imag_v;
} COMPLEX;

#define MATFILE_PC      0000
#define MATFILE_SPARC   1000
#define MATFILE_DOUBLE  00
#define MATFILE_FULL    0

typedef struct
{
    int32_t  type ;
    int32_t  mrows ;
    int32_t  ncols ;
    int32_t  imagf ;
    int32_t  namlen ;
} MATHD4;

typedef struct
{
    char  text[116];
    char  subsystemoffset[8];
    short version;
    short endianness;
} MATHD5;

typedef struct
{
    unsigned int type;
    unsigned int len;
} MATDATATAG5;

typedef struct
{
    unsigned short len;
    unsigned short type;
} MATSMALLDATATAG5;

typedef enum {
    miINT8 = 1,
    miUINT8 = 2,
    miINT16 = 3,
    miUINT16 = 4,
    miINT32 = 5,
    miUINT32 = 6,
    miSINGLE = 7,
    miDOUBLE = 9,
    miINT64 = 12,
    miUINT64 = 13,
    miMATRIX = 14,
    miCOMPRESSED = 15,
    miUTF8 = 16,
    miUTF16 = 17,
    miUTF32 = 18,
} MATDATATYPE5;

typedef enum {
    mxCELL_CLASS = 1,
    mxSTRUCT_CLASS = 2,
    mxOBJECT_CLASS = 3,
    mxCHAR_CLASS = 4,
    mxSPARSE_CLASS = 5,
    mxDOUBLE_CLASS = 6,
    mxSINGLE_CLASS = 7,
    mxINT8_CLASS = 8,
    mxUINT8_CLASS = 9,
    mxINT16_CLASS = 10,
    mxUINT16_CLASS = 11,
    mxINT32_CLASS = 12,
    mxUINT32_CLASS = 13,
} MATARRAYTYPE5;

#if 0
typedef struct
{
    long  type ;
    long  mrows ;
    long  ncols ;
    long  imagf ;
    long  namlen ;
    char  *data ;
    char  *idata ;
} MATFILE ;
#endif

typedef struct {
    int type;
    int hdrlen;
    size_t datalen;
    char * data;
} matelem5;

typedef struct {
    int flags;
    int class;
    int nzmax;
    int numdims;
    int * dimsizes;
    char * name;
    size_t datalen;
    char * arraydata;
} matmatrix5;

matelem5
readmatelem5(char * buf, off_t bufoff, size_t buflen, int swap)
{
    matelem5 ret;
    MATDATATAG5 tag;
    MATSMALLDATATAG5 smalltag;
    memset(&ret, '\0', sizeof(ret));
    if (bufoff + sizeof(MATDATATAG5) >= buflen) {
	fprintf(stderr, "readmatelem5: Reached end-of-buffer reading data tag from .mat file\n");
	return ret;
    }
    memcpy(&tag, buf + bufoff, 8);
    if (swap) { tag.type = swapint32(tag.type); }
    if ((tag.type & 0xffff0000) == 0) {
	/* long tag */
	if (swap) { tag.len = swapint32(tag.len); }
	ret.type = tag.type;
	ret.hdrlen = 8;
	ret.datalen = tag.len;
    } else {
	/* small tag */
#if 0
	if (swap) { tag.type = swapint32(tag.type); } /* swap back */
	memcpy(&smalltag, &tag, sizeof(smalltag));
	if (swap) {
	    smalltag.type = swapushort(smalltag.type);
	    smalltag.len = swapushort(smalltag.len);
	}
#else
	smalltag.type = (tag.type & 0x0000ffff);
	smalltag.len = (tag.type & 0xffff0000) >> 16;
#endif
	ret.type = smalltag.type;
	ret.hdrlen = 4;
	ret.datalen = smalltag.len;
    }
    ret.data = (char *)malloc(ret.datalen);
    memcpy(ret.data, buf + bufoff + ret.hdrlen, ret.datalen);
    return ret;
}

matmatrix5
readmatmatrix5double(char * buf, off_t bufoff, size_t buflen, int swap)
{
    matmatrix5 ret;
    matelem5 me;
    int numelems = 0;
    int elemnum = 0;
    int elemsize = 0;
    unsigned int mathead1 = 0;
    memset(&ret, '\0', sizeof(ret));
    me = readmatelem5(buf, bufoff, buflen, swap);
    if (me.data == NULL) {
	fprintf(stderr, "readmatmatrix5double: Error reading data element from .mat file.\n");
	return ret;
    }
    memcpy(&mathead1, me.data, 4);
    memcpy(&ret.nzmax, me.data + 4, 4);
    if (swap) {
	mathead1 = swapuint(mathead1);
	ret.nzmax = swapint32(ret.nzmax);
    }
    ret.flags = (mathead1 & 0xff00) >> 8;
    ret.class = mathead1 & 0xff;
    if (ret.class != mxDOUBLE_CLASS) {
	fprintf(stderr, "readmatmatrix5double: .mat file matrix is not of class 'double' (has class %d).\n", ret.class);
	return ret;
    }
    free(me.data);
    bufoff += me.hdrlen + me.datalen + ((me.hdrlen - me.datalen) % me.hdrlen);
    me = readmatelem5(buf, bufoff, buflen, swap);
    if (me.type != 5) { /* miINT32 */
	fprintf(stderr, "readmatmatrix5double: dimension element is not INT32?\n");
	return ret;
    }
    ret.dimsizes = (int *)malloc(me.datalen);
    memcpy(ret.dimsizes, me.data, me.datalen);
    ret.numdims = me.datalen / 4;
    free(me.data);
    if (swap) {
	*(int *)&ret.dimsizes[0] = swapuint(*(unsigned int *)&ret.dimsizes[0]);
	*(int *)&ret.dimsizes[1] = swapuint(*(unsigned int *)&ret.dimsizes[1]);
    }
    bufoff += me.hdrlen + me.datalen + ((me.hdrlen - me.datalen) % me.hdrlen);
    me = readmatelem5(buf, bufoff, buflen, swap);
    if (me.type != 1) { /* miINT8 */
	fprintf(stderr, "readmatmatrix5double: array name is not INT8?\n");
	free(ret.dimsizes); ret.dimsizes = NULL;
	return ret;
    }
    ret.name = (char *)malloc(me.datalen + 1);
    memcpy(ret.name, me.data, me.datalen);
    ret.name[me.datalen] = '\0';
    free(me.data);
    bufoff += me.hdrlen + me.datalen + ((me.hdrlen - me.datalen) % me.hdrlen);
    me = readmatelem5(buf, bufoff, buflen, swap);
    switch (me.type) {
    case 1: /* miINT8 */ elemsize = 1; break;
    case 2: /* miUINT8 */ elemsize = 1; break;
    case 3: /* miINT16 */ elemsize = 2; break;
    case 4: /* miUINT16 */ elemsize = 2; break;
    case 5: /* miINT32 */ elemsize = 4; break;
    case 6: /* miUINT32 */ elemsize = 4; break;
    case 7: /* miSINGLE */ elemsize = 4; break;
    case 9: /* miDOUBLE */ elemsize = 8; break;
    case 12: /* miINT64 */ elemsize = 8; break;
    case 13: /* miUINT64 */ elemsize = 8; break;
    default:
	fprintf(stderr, "readmatmatrix5double: Unsupported matrix type %d\n", me.type);
	free(ret.name); ret.name = NULL;
	free(ret.dimsizes); ret.dimsizes = NULL;
	return ret;
    }
    numelems = me.datalen / elemsize;
    ret.datalen = me.datalen;
    ret.arraydata = (char *)malloc(sizeof(double)*numelems);
    for (elemnum = 0; elemnum < numelems; elemnum++) {
	double val = 0;
	char * srcptr = me.data + (elemnum * elemsize);
	if (swap) {
	    switch (me.type) {
	    case 1: /* miINT8 */
		break;
	    case 2: /* miUINT8 */
		break;
	    case 3: /* miINT16 */
		*(short *)srcptr = swapushort(*(unsigned short *)srcptr);
		break;
	    case 4: /* miUINT16 */
		*(unsigned short *)srcptr = swapushort(*(unsigned short *)srcptr);
		break;
	    case 5: /* miINT32 */
		*(int *)srcptr = swapuint(*(unsigned int *)srcptr);
		break;
	    case 6: /* miUINT32 */
		*(unsigned int *)srcptr = swapuint(*(unsigned int *)srcptr);
		break;
	    case 7: /* miSINGLE */
		*(float *)srcptr = swapuint(*(unsigned int *)srcptr);
		break;
	    case 9: /* miDOUBLE */
		*(double *)srcptr = swapdouble(*(double *)srcptr);
		break;
	    case 12: /* miINT64 */
		*(long long *)srcptr = swapdouble(*(double *)srcptr);
		break;
	    case 13: /* miUINT64 */
		*(unsigned long long *)srcptr = swapdouble(*(double *)srcptr);
		break;
	    }
	}
	switch (me.type) {
	case 1: /* miINT8 */ val = (double)*srcptr; break;
	case 2: /* miUINT8 */ val = (double)*(unsigned char *)srcptr; break;
	case 3: /* miINT16 */ val = (double)*(short *)srcptr; break;
	case 4: /* miUINT16 */ val = (double)*(unsigned short *)srcptr; break;
	case 5: /* miINT32 */ val = (double)*(int *)srcptr; break;
	case 6: /* miUINT32 */ val = (double)*(unsigned int *)srcptr; break;
	case 7: /* miSINGLE */ val = (double)*(float *)srcptr; break;
	case 9: /* miDOUBLE */ val = (double)*(double *)srcptr; break;
	case 12: /* miINT64 */ val = (double)*(long long *)srcptr; break;
	case 13: /* miUINT64 */ val = (double)*(unsigned long long *)srcptr; break;
	}
	((double *)ret.arraydata)[elemnum] = val;
    }
    return ret;
}

static void
grablong(void * out_i, char * buf, size_t pos)
{
    char * out = (char *)out_i;
    memcpy(out, &buf[pos], 4);
    if (needswap) {
	char swapper;
	swapper = out[0];
	out[0] = out[3];
	out[3] = swapper;
	swapper = out[1];
	out[1] = out[2];
	out[2] = swapper;
    }
}

static void
grabshort(void * out_i, char * buf, size_t pos)
{
    char * out = (char *)out_i;
    memcpy(out, &buf[pos], 2);
    if (needswap) {
	char swapper;
	swapper = out[0];
	out[0] = out[1];
	out[1] = swapper;
    }
}

BXHDocPtr
createDocFromAnalyze(const char **inifnames, int numinfiles, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient, char * orient, int strictanalyze, char * avwbyteorder, int nofixpixdim)
{
    char **ifnames = NULL;
    char hdrbuf[348];
    BXHDocPtr docp = NULL;
    bxhrawdatarec * datarec = NULL;
    bxhdimension * dimx = NULL;
    bxhdimension * dimy = NULL;
    bxhdimension * dimz = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    BXHElementPtr subject = NULL;
    char * sliceorderstr = NULL;
    int dimnum;
    int infilenum;
    int msbfirst;
    int orientwarned = 0;
    struct stat statbuf;

    if (forceorient != NULL && orient == NULL) {
	orient = forceorient;
    }

    ifnames = (char **)malloc(sizeof(char*)*numinfiles);
    for (infilenum = 0; infilenum < numinfiles; infilenum++) {
	ifnames[infilenum] = strdup(inifnames[infilenum]);
    }

    msbfirst = 1;
    msbfirst = (((char *)&msbfirst)[0] == 0);

    if (avwbyteorder != NULL &&
	strcmp(avwbyteorder, "b") != 0 &&
	strcmp(avwbyteorder, "l") != 0) {
	fprintf(stderr, "createDocFromAnalyze: --avwbyteorder must be either 'b' or 'l'\n");
	goto FAIL;
    }
    
    if (avwbyteorder == NULL) {
	if (msbfirst)
	    avwbyteorder = "b";
	else
	    avwbyteorder = "l";
    }

    for (infilenum = 0; infilenum < numinfiles; infilenum++) {
	const char * ifname = ifnames[infilenum];
	FILE * iffp = NULL;
	static char linebuf[2048];
	size_t iflen = 0;

	iflen = strlen(ifname);

	if ((iffp = fopen(ifname, "rb")) == NULL) {
	    fprintf(stderr, "createDocFromAnalyze: Error opening file %s\n", ifname);
	    perror("fopen");
	    goto FAIL;
	}
	if (fread(&linebuf[0], 14, 1, iffp) != 1) {
	    fprintf(stderr, "createDocFromAnalyze: Error reading first 13 bytes of file %s\n", ifname);
	    goto FAIL;
	}

	if (strncmp(&linebuf[0], "AVW_ImageFile", 13) == 0) {
	    /* AVW_ImageFile format */
	    unsigned int hdrend;
	    long curpos;
	    static char namebuf[128];
	    static char valuebuf[128];
	    unsigned long width = 1;
	    unsigned long height = 1;
	    unsigned long depth = 1;
	    unsigned long numvols = 1;
	    int endian_msb = 1;
	    char * elemtype = NULL;
	    unsigned int elemsize = 0;

	    /* read rest of first line */
	    if (fgets(&linebuf[0], sizeof(linebuf), iffp) == NULL)
		goto FREADFAIL;

	    if (sscanf(&linebuf[0], " %*f %u", &hdrend) != 1) {
		fprintf(stderr, "createDocFromAnalyze: Malformed first line in AVW_ImageFile %s\n", ifname);
		goto FAIL;
	    }
	    while ((curpos = ftell(iffp)) < (off_t)hdrend) {
		if (curpos < 0) {
		    fprintf(stderr, "createDocFromAnalyze: Error getting position in file %s\n", ifname);
		    goto FAIL;
		}
		if (fgets(&linebuf[0], sizeof(linebuf), iffp) == NULL)
		    goto FREADFAIL;
		namebuf[0] = '\0';
		valuebuf[0] = '\0';
		sscanf(&linebuf[0], " %128[^=]=%128s",
		       &namebuf[0], &valuebuf[0]);
		if (strcmp(&namebuf[0], "DataType") == 0) {
		    if (strcmp(&valuebuf[0], "AVW_UNSIGNED_CHAR") == 0) {
			elemtype = strdup("uint8");
			elemsize = 1;
		    } else if (strcmp(&valuebuf[0], "AVW_SIGNED_CHAR") == 0) {
			elemtype = strdup("int8");
			elemsize = 1;
		    } else if (strcmp(&valuebuf[0], "AVW_UNSIGNED_SHORT") == 0) {
			elemtype = strdup("uint16");
			elemsize = 2;
		    } else if (strcmp(&valuebuf[0], "AVW_SIGNED_SHORT") == 0) {
			elemtype = strdup("int16");
			elemsize = 2;
		    } else if (strcmp(&valuebuf[0], "AVW_UNSIGNED_INT") == 0) {
			elemtype = strdup("uint32");
			elemsize = 4;
		    } else if (strcmp(&valuebuf[0], "AVW_SIGNED_INT") == 0) {
			elemtype = strdup("int32");
			elemsize = 4;
		    } else if (strcmp(&valuebuf[0], "AVW_FLOAT") == 0) {
			elemtype = strdup("float32");
			elemsize = 4;
		    } else if (strcmp(&valuebuf[0], "AVW_DOUBLE_FLOAT") == 0) {
			elemtype = strdup("float64");
			elemsize = 8;
		    } else if (strcmp(&valuebuf[0], "AVW_COMPLEX") == 0) {
			fprintf(stderr, "createDocFromAnalyze: Analyze COMPLEX format not currently supported!\n");
			goto FAIL;
		    } else if (strcmp(&valuebuf[0], "AVW_COLOR") == 0) {
			fprintf(stderr, "createDocFromAnalyze: Analyze COLOR format not currently supported!\n");
			goto FAIL;
		    } else {
			fprintf(stderr, "createDocFromAnalyze: Unrecognized Analyze format %s!\n", &valuebuf[0]);
			goto FAIL;
		    }
		} else if (strcmp(&namebuf[0], "Width") == 0) {
		    width = strtoul(&valuebuf[0], NULL, 10);
		} else if (strcmp(&namebuf[0], "Height") == 0) {
		    height = strtoul(&valuebuf[0], NULL, 10);
		} else if (strcmp(&namebuf[0], "Depth") == 0) {
		    depth = strtoul(&valuebuf[0], NULL, 10);
		} else if (strcmp(&namebuf[0], "NumVols") == 0) {
		    numvols = strtoul(&valuebuf[0], NULL, 10);
		} else if (strcmp(&namebuf[0], "ColormapSize") == 0) {
		} else if (strcmp(&namebuf[0], "BeginInformation") == 0) {
		} else if (strcmp(&namebuf[0], "MaximumDataValue") == 0) {
		} else if (strcmp(&namebuf[0], "MinimumDataValue") == 0) {
		} else if (strcmp(&namebuf[0], "Endian") == 0) {
		    if (strcmp(&valuebuf[0], "Little") == 0)
			endian_msb = 0;
		    else
			endian_msb = 1;
		} else if (strcmp(&namebuf[0], "EndInformation") == 0) {
		} else if (strcmp(&namebuf[0], "MoreInformation") == 0) {
		    if (strtol(&valuebuf[0], NULL, 10) == -1)
			break;
		}
	    }
	    if (fgets(&linebuf[0], sizeof(linebuf), iffp) == NULL)
		goto FREADFAIL;
	    namebuf[0] = '\0';
	    if (sscanf(&linebuf[0], "Vol Slc Offset Length Cmp %s", &namebuf[0]) != 1 || strcmp(&namebuf[0], "Format") != 0) {
		fprintf(stderr, "createDocFromAnalyze: Didn't find slice table in AVW_ImageFile %s\n", ifname);
		goto FAIL;
	    }

	    if (datarec == NULL) {
		datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
		memset(datarec, '\0', sizeof(bxhrawdatarec));

		datarec->numdims = 4;
		datarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension)*4);
		memset(datarec->dimensions, '\0', sizeof(bxhdimension)*4);
		datarec->dimensions[0].type = strdup("x");
		datarec->dimensions[1].type = strdup("y");
		datarec->dimensions[2].type = strdup("z");
		datarec->dimensions[3].type = strdup("t");
		datarec->dimensions[0].size = width;
		datarec->dimensions[1].size = height;
		datarec->dimensions[2].size = depth;
		datarec->dimensions[3].size = numvols;
		datarec->msbfirstfrags = endian_msb;
		datarec->elemtype = strdup(elemtype);
		datarec->elemsize = elemsize;
	    } else {
		if (datarec->numdims < 3 || datarec->numdims > 4) {
		    fprintf(stderr, "Number of dimensions in %s doesn't match earlier files\n", ifname);
		    goto FAIL;
		}
		if (datarec->dimensions[0].size != width ||
		    datarec->dimensions[1].size != height ||
		    datarec->dimensions[2].size != depth) {
		    fprintf(stderr, "Width, Height and Depth of %s don't match earlier files\n", ifname);
		    goto FAIL;
		}
		if (datarec->msbfirstfrags != endian_msb) {
		    fprintf(stderr, "Endian-ness of %s doesn't match earlier files\n", ifname);
		    goto FAIL;
		}
		if (strcmp(datarec->elemtype, elemtype) != 0) {
		    fprintf(stderr, "Element type of %s doesn't match earlier files\n", ifname);
		    goto FAIL;
		}
		datarec->dimensions[3].size += numvols;
	    }

	    while ((curpos = ftell(iffp)) < (off_t)hdrend) {
		unsigned long volnum = 0;
		unsigned long slcnum = 0;
		unsigned long off = 0;
		unsigned long len = 0;
		unsigned long cmp = 0;
		
		if (curpos < 0) {
		    fprintf(stderr, "createDocFromAnalyze: Error getting position in file %s\n", ifname);
		    goto FAIL;
		}
		if (fgets(&linebuf[0], sizeof(linebuf), iffp) == NULL)
		    goto FREADFAIL;
		if (sscanf(&linebuf[0], " %lu %lu %lu %lu %lu",
			   &volnum, &slcnum, &off, &len, &cmp) != 5) {
		    if (strncmp(&linebuf[0], ".CONTIG", 7) == 0) {
			if (fstat(fileno(iffp), &statbuf) != 0) {
			    fprintf(stderr, "Error finding size of file %s\n", ifname);
			    perror("fstat");
			    goto FAIL;
			}
			bxh_datarec_addfrag(datarec, ifname, hdrend, statbuf.st_size - hdrend, ofname, 1);
			datarec->datasize += statbuf.st_size - hdrend;
			break;
		    }
		    if (strncmp(&linebuf[0], "EndSliceTable", 13) == 0)
			break;
		    fprintf(stderr, "createDocFromAnalyze: Malformed slice table line in AVW_ImageFile %s\n", ifname);
		    goto FAIL;
		}

		bxh_datarec_addfrag(datarec, ifname, off, len, ofname, 1);
		datarec->datasize += len;
	    }

	    goto FREADEXIT;
	    
	  FREADFAIL:
	    if (feof(iffp)) {
		fprintf(stderr, "createDocFromAnalyze: Prematurely reached end of file %s\n", ifname);
	    } else {
		fprintf(stderr, "createDocFromAnalyze: Unknown erro when reading file %s\n", ifname);
	    }
	    fclose(iffp);
	    goto FAIL;
	    
	  FREADEXIT:
	    fclose(iffp);
	} else if (strncmp(&linebuf[0], "AVW_VolumeFile", 14) == 0) {
	    int numadded = 0;
	    /* AVW_VolumeFile format */

	    /* get rid of rest of first line */
	    if (fgets(&linebuf[0], sizeof(linebuf), iffp) == NULL)
		goto FREADFAIL2;
	    
	    /* "uncount" the volume file, individual files will be counted */
	    memmove(&ifnames[infilenum], &ifnames[infilenum+1], sizeof(char *) * (numinfiles - (infilenum + 1)));
	    numinfiles--;
	    /* add individual files */
	    while (fgets(&linebuf[0], sizeof(linebuf), iffp) != NULL) {
		size_t lineend;
		if (linebuf[0] == '#')
		    continue;
		lineend = strlen(&linebuf[0]) - 1;
		while (lineend >= 0 && (linebuf[lineend] == '\n' || linebuf[lineend] == '\r')) {
		    linebuf[lineend] = '\0';
		    lineend--;
		}
		ifnames = (char **)realloc(ifnames, sizeof(char *) * (numinfiles + 1));
		memmove(&ifnames[infilenum + numadded + 1], &ifnames[infilenum + numadded], sizeof(char *) * (numinfiles - (infilenum + numadded)));
		ifnames[infilenum + numadded] = bxh_datarec_getfullfilename(ifname,linebuf);
		numinfiles++;
		numadded++;
	    }
	    /* need to re-check the new file at current index, so decrement */
	    infilenum--;

	    goto FREADEXIT2;

	  FREADFAIL2:
	    if (feof(iffp)) {
		fprintf(stderr, "createDocFromAnalyze: Prematurely reached end of file %s\n", ifname);
	    } else {
		fprintf(stderr, "createDocFromAnalyze: Unknown erro when reading file %s\n", ifname);
	    }
	    fclose(iffp);
	    goto FAIL;
	    
	  FREADEXIT2:
	    fclose(iffp);
	} else if ((iflen >= 4 &&
		    (strcmp(ifname+iflen-4, ".hdr") == 0 ||
		     strcmp(ifname+iflen-4, ".img") == 0 ||
		     strcmp(ifname+iflen-4, ".nii") == 0)) ||
		   (iflen >= 7 &&
		    strcmp(ifname+iflen-7, ".nii.gz") == 0)) {
	    /* Analyze 7.5 format (or NIfTI-1) */

	    typedef double mmatrix[16];
	    mmatrix * mmatp = NULL;
	    gzFile * hdrfp = NULL;
	    FILE * matfp = NULL;
	    char * hdrname = NULL;
	    char * imgname = NULL;
	    char * matname = NULL;
	    analyze_dsr hdr;
	    analyze_dsr * hdrp = &hdr;
	    struct nifti_1_header *nifp = (struct nifti_1_header *)&hdr;
	    int isnifti = 0;
	    int isnii = 0;
	    int iscompressed = 0;
	    bxhrawdatarec * olddatarec = datarec;
	    int spm_byteswap = 0;

	    datarec = NULL;

	    fclose(iffp);

	    memset(hdrp, '\0', sizeof(*hdrp));

	    if (iflen >= 4 && strcmp(ifname+iflen-4, ".nii") == 0) {
		hdrname = strdup(ifname);
		imgname = strdup(ifname);
		isnii = 1;
	    } else if (iflen >= 7 && strcmp(ifname+iflen-7, ".nii.gz") == 0) {
		hdrname = strdup(ifname);
		imgname = strdup(ifname);
		isnii = 1;
		iscompressed = 1;
	    } else {
		hdrname = strdup(ifname);
		strcpy(hdrname+iflen-4, ".hdr");
		imgname = strdup(ifname);
		strcpy(imgname+iflen-4, ".img");
		matname = strdup(ifname);
		strcpy(matname+iflen-4, ".mat");
	    }

	    if ((hdrfp = gzopen(hdrname, "rb")) == NULL) {
		fprintf(stderr, "createDocFromAnalyze: Error opening header file '%s'\n", hdrname);
		perror("fopen");
		goto FAIL;
	    }
    
	    if (gzread(hdrfp, &hdrbuf[0], sizeof(hdrbuf)) != sizeof(hdrbuf)) {
		fprintf(stderr, "createDocFromAnalyze: Error reading header from '%s'\n", hdrname);
		goto FAIL;
	    }

	    gzclose(hdrfp);
	    hdrfp = NULL;

	    if (*(int *)&hdrbuf[0] == 0x15C) {
		needswap = 0;
	    } else if (*(int *)&hdrbuf[0] == 0x5C010000) {
		needswap = 1;
	    } else {
		fprintf(stderr, "createDocFromAnalyze: Header size is not 348 (either byte order)\n");
		goto FAIL;
	    }

	    if (strcmp(&hdrbuf[344], "ni1") == 0 ||
		strcmp(&hdrbuf[344], "n+1") == 0) {
		isnifti = 1;
	    }

	    /* find .mat files if they exist */
	    if (!isnii &&
		!isnifti &&
		!strictanalyze &&
		(matfp = fopen(matname, "rb")) != NULL) {
		char magic[4];
		if (fread(&magic[0], 4, 1, matfp) != 1) {
		    fprintf(stderr, "createDocFromAnalyze: Error reading magic string from '%s'\n", matname);
		    goto FAIL;
		}
		if (fseek(matfp, 0, SEEK_SET) != 0) {
		    fprintf(stderr, "createDocFromAnalyze: Error seeking in '%s'\n", matname);
		    goto FAIL;
		}
		if (magic[0] == 0 || magic[1] == 0 ||
		    magic[2] == 0 || magic[3] == 0) {
		    mmatrix tmpmat;
		    char * buf = NULL;
		    int i;
		    MATHD4 mh;
		    if (fread(&mh, sizeof(MATHD4), 1, matfp) != 1) {
			fprintf(stderr, "createDocFromAnalyze: Error reading matrix header from '%s'\n", matname);
			goto FAIL;
		    }
		    /* Level 4 .mat file */
		    if (mh.type < 0 || mh.type > 1000) {
			mh.type = swapuint(mh.type);
		    }
		    if (((mh.type / 1000) == 1) != msbfirst) {
			mh.mrows = swapuint(mh.mrows);
			mh.ncols = swapuint(mh.ncols);
			mh.imagf = swapuint(mh.imagf);
			mh.namlen = swapuint(mh.namlen);
		    }
		    if (!(mh.type == 0 || mh.type == 1000) ||
			mh.mrows != 4 || mh.ncols != 4) {
			fprintf(stderr, "createDocFromAnalyze: .mat file '%s' has incorrect dimensions (%ldx%ld) or data type (%ld)\n", matname, (long int)mh.mrows, (long int)mh.ncols, (long int)mh.type);
			goto FAIL;
		    }
		    buf = (char *)malloc(sizeof(char)*mh.namlen);
		    if (fread(buf, mh.namlen, 1, matfp) != 1) {
			fprintf(stderr, "createDocFromAnalyze: Error reading matrix name from '%s'\n", matname);
			if (buf) {
			    free(buf);
			    buf = NULL;
			}
			goto FAIL;
		    }
		    if (buf) {
			free(buf);
			buf = NULL;
		    }
		    memset(&tmpmat[0], '\0', sizeof(mmatrix));
		    if (fread(&tmpmat[0], sizeof(double)*16, 1, matfp) != 1) {
			fprintf(stderr, "createDocFromAnalyze: Error reading matrix from '%s'\n", matname);
			goto FAIL;
		    }
		    mmatp = (mmatrix *)malloc(sizeof(mmatrix));
		    /* convert matrix to row-major format */
		    for (i = 0; i < 16; i++) {
			double tmpdouble = tmpmat[(i%4)*4 + (i/4)];
			if (((mh.type / 1000) == 1) != msbfirst) {
			    tmpdouble = swapdouble(tmpdouble);
			}
			(*mmatp)[i] = tmpdouble;
		    }
		} else {
		    /* Level 5 .mat file */
		    char * buf = NULL;
		    off_t bufoff = 0;
		    MATHD5 mh;
		    char normalendianstr[3] = "MI";
		    char endianstr[3] = "";
		    int swap = 0;
		    struct stat statbuf;
		    matelem5 me;
		    matmatrix5 mm;
		    matmatrix5 mmM;
		    matmatrix5 mmmat;
		    int foundM = 0;
		    int foundmat = 0;
		    int i;
		    if (stat(matname, &statbuf) != 0) {
			fprintf(stderr, "createDocFromAnalyze: Error stat'ing .mat file '%s'\n", matname);
			goto FAIL;
		    }
		    buf = (char *)malloc(statbuf.st_size);
		    if (fread(buf, statbuf.st_size, 1, matfp) != 1) {
			fprintf(stderr, "createDocFromAnalyze: Error reading .mat file from '%s'\n", matname);
			goto FAIL;
		    }
		    memcpy(&mh, buf + bufoff, sizeof(MATHD5));
		    bufoff += sizeof(MATHD5);;
		    endianstr[0] = ((*((unsigned short *)&mh.endianness)) & 0xff00) >> 8;
		    endianstr[1] = (*((unsigned short *)&mh.endianness)) & 0xff;
		    endianstr[2] = '\0';
		    if (strcmp(&endianstr[0], &normalendianstr[0]) != 0) {
			swap = 1;
		    }
		    mh.version = swapuint(mh.version);
		    while (bufoff < statbuf.st_size && (!foundM || !foundmat)) {
			matelem5 origme;
			me = readmatelem5(buf, bufoff, statbuf.st_size, swap);
			origme = me;
			if (origme.type == 15) { /* miCOMPRESSED */
			    matelem5 me_old = me;
			    size_t newbuflen = 512;
			    off_t newbufpos = 0;
			    char * newbuf = NULL;
			    z_stream zs;
			    int zerr;
			    memset(&zs, '\0', sizeof(zs));
			    zs.zalloc = Z_NULL;
			    zs.zfree = Z_NULL;
			    zs.opaque = Z_NULL;
			    zs.next_in = me_old.data;
			    zs.avail_in = me_old.datalen;
			    if (inflateInit(&zs) != Z_OK) {
				fprintf(stderr, "Error initizializing zlib state\n");
				goto FAIL;
			    }
			    zerr = Z_OK;
			    while (zerr != Z_STREAM_END) {
				if (newbuf == NULL || newbufpos >= newbuflen) {
				    newbuflen *= 2;
				    newbuf = (char *)realloc(newbuf, newbuflen);
				}
				zs.next_out = newbuf + newbufpos;
				zs.avail_out = newbuflen - newbufpos;
				zerr = inflate(&zs, Z_SYNC_FLUSH);
				if (zerr != Z_OK && zerr != Z_STREAM_END) {
				    fprintf(stderr, "Error uncompressing data with zlib\n");
				    goto FAIL;
				}
			    }
			    me = readmatelem5(newbuf, 0, zs.total_out, swap);
			    free(me_old.data); me_old.data = NULL;
			}
			bufoff += origme.hdrlen + origme.datalen;
			if (origme.type != 15) {
			    bufoff += (origme.hdrlen - origme.datalen) % origme.hdrlen;
			}
			if (me.type == 14) {
			    mm = readmatmatrix5double(me.data, 0, me.datalen, swap);
			    if (mm.name != NULL &&
				mm.dimsizes != NULL &&
				mm.arraydata != NULL &&
				mm.numdims == 2 &&
				mm.dimsizes[0] == 4 &&
				mm.dimsizes[1] == 4 &&
				mm.class == mxDOUBLE_CLASS) {
				if (strcmp(mm.name, "M") == 0) {
				    mmM = mm;
				    foundM = 1;
				}
				if (strcmp(mm.name, "mat") == 0) {
				    mmmat = mm;
				    foundmat = 1;
				}
			    }
			} /* miMATRIX */
		    }
		    if (!foundmat && !foundM) {
			fprintf(stderr, "createDocFromAnalyze: Didn't find 4x4 double-precision floating-point 'M' or 'mat'\nmatrix in .mat file '%s'\n", matname);
			goto FAIL;
		    }
		    if (foundmat) {
			mm = mmmat;
		    } else if (foundM) {
			fprintf(stderr, "Using SPM99 'M' matrix -- L/R orientation may not be correct!\n");
			mm = mmM;
		    }
		    mmatp = (mmatrix *)malloc(sizeof(mmatrix));
		    /* convert matrix to row-major format */
		    for (i = 0; i < 16; i++) {
			double tmpdouble = ((double *)mm.arraydata)[(i%4)*4 + (i/4)];
/*			if (swap) { tmpdouble = swapdouble(tmpdouble); } */
			(*mmatp)[i] = tmpdouble;
		    }
		    free(me.data); me.data = NULL;
		    free(mm.name); mm.name = NULL;
		    free(mm.dimsizes); mm.dimsizes = NULL;
		    free(mm.arraydata); mm.arraydata = NULL;
		}
		fclose(matfp);
	    }

	    if (isnifti) {
		grablong(&nifp->sizeof_hdr, hdrbuf, 4);
		strncpy(&nifp->data_type[0], &hdrbuf[4], 10);
		strncpy(&nifp->db_name[0], &hdrbuf[14], 18);
		grablong(&nifp->extents, hdrbuf, 32);
		grabshort(&nifp->session_error, hdrbuf, 36);
		nifp->regular = hdrbuf[38];
		nifp->dim_info = hdrbuf[39];

		grabshort(&nifp->dim[0], hdrbuf, 40+0);
		grabshort(&nifp->dim[1], hdrbuf, 40+2);
		grabshort(&nifp->dim[2], hdrbuf, 40+4);
		grabshort(&nifp->dim[3], hdrbuf, 40+6);
		grabshort(&nifp->dim[4], hdrbuf, 40+8);
		grabshort(&nifp->dim[5], hdrbuf, 40+10);
		grabshort(&nifp->dim[6], hdrbuf, 40+12);
		grabshort(&nifp->dim[7], hdrbuf, 40+14);
		grablong(&nifp->intent_p1, hdrbuf, 40+16);
		grablong(&nifp->intent_p2, hdrbuf, 40+20);
		grablong(&nifp->intent_p3, hdrbuf, 40+24);
		grabshort(&nifp->intent_code, hdrbuf, 40+28);
		grabshort(&nifp->datatype, hdrbuf, 40+30);
		grabshort(&nifp->bitpix, hdrbuf, 40+32);
		grabshort(&nifp->slice_start, hdrbuf, 40+34);
		grablong(&nifp->pixdim[0], hdrbuf, 40+36);
		grablong(&nifp->pixdim[1], hdrbuf, 40+40);
		grablong(&nifp->pixdim[2], hdrbuf, 40+44);
		grablong(&nifp->pixdim[3], hdrbuf, 40+48);
		grablong(&nifp->pixdim[4], hdrbuf, 40+52);
		grablong(&nifp->pixdim[5], hdrbuf, 40+56);
		grablong(&nifp->pixdim[6], hdrbuf, 40+60);
		grablong(&nifp->pixdim[7], hdrbuf, 40+64);
		grablong(&nifp->vox_offset, hdrbuf, 40+68);
		grablong(&nifp->scl_slope, hdrbuf, 40+72);
		grablong(&nifp->scl_inter, hdrbuf, 40+76);
		grabshort(&nifp->slice_end, hdrbuf, 40+80);
		nifp->slice_code = hdrbuf[40+82];
		nifp->xyzt_units = hdrbuf[40+83];
		grablong(&nifp->cal_max, hdrbuf, 40+84);
		grablong(&nifp->cal_min, hdrbuf, 40+88);
		grablong(&nifp->slice_duration, hdrbuf, 40+92);
		grablong(&nifp->toffset, hdrbuf, 40+96);
		grablong(&nifp->glmax, hdrbuf, 40+100);
		grablong(&nifp->glmin, hdrbuf, 40+104);

		strncpy(&nifp->descrip[0], &hdrbuf[148+0], 80);
		strncpy(&nifp->aux_file[0], &hdrbuf[148+80], 24);
		grabshort(&nifp->qform_code, hdrbuf, 148+104);
		grabshort(&nifp->sform_code, hdrbuf, 148+106);
		grablong(&nifp->quatern_b, hdrbuf, 148+108);
		grablong(&nifp->quatern_c, hdrbuf, 148+112);
		grablong(&nifp->quatern_d, hdrbuf, 148+116);
		grablong(&nifp->qoffset_x, hdrbuf, 148+120);
		grablong(&nifp->qoffset_y, hdrbuf, 148+124);
		grablong(&nifp->qoffset_z, hdrbuf, 148+128);
		grablong(&nifp->srow_x[0], hdrbuf, 148+132);
		grablong(&nifp->srow_x[1], hdrbuf, 148+136);
		grablong(&nifp->srow_x[2], hdrbuf, 148+140);
		grablong(&nifp->srow_x[3], hdrbuf, 148+144);
		grablong(&nifp->srow_y[0], hdrbuf, 148+148);
		grablong(&nifp->srow_y[1], hdrbuf, 148+152);
		grablong(&nifp->srow_y[2], hdrbuf, 148+156);
		grablong(&nifp->srow_y[3], hdrbuf, 148+160);
		grablong(&nifp->srow_z[0], hdrbuf, 148+164);
		grablong(&nifp->srow_z[1], hdrbuf, 148+168);
		grablong(&nifp->srow_z[2], hdrbuf, 148+172);
		grablong(&nifp->srow_z[3], hdrbuf, 148+176);
		strncpy(&nifp->intent_name[0], &hdrbuf[148+180], 16);
		strncpy(&nifp->magic[0], &hdrbuf[148+196], 4);

		if (!nofixpixdim) {
		    int i = 0;
		    for (i = 1; i <= nifp->dim[0]; i++) {
			// nifti1_io.c "corrects" bad pixdims so we'll do it too
			if (nifp->pixdim[i] == 0.0) {
			    fprintf(stderr, "Warning: \"correcting\" pixdim[%d] to 1.0 (original value is 0.0).  Specify --nofixpixdim to leave it 0.0\n", i);
			    nifp->pixdim[i] = 1.0;
			}
		    }
		}
	    } else {
		grablong(&hdrp->hk.sizeof_hdr, hdrbuf, 4);
		strncpy(&hdrp->hk.data_type[0], &hdrbuf[4], 10);
		strncpy(&hdrp->hk.db_name[0], &hdrbuf[14], 18);
		grablong(&hdrp->hk.extents, hdrbuf, 32);
		grabshort(&hdrp->hk.session_error, hdrbuf, 36);
		hdrp->hk.regular = hdrbuf[38];
		hdrp->hk.hkey_un0 = hdrbuf[39];

		grabshort(&hdrp->dime.dim[0], hdrbuf, 40+0);
		grabshort(&hdrp->dime.dim[1], hdrbuf, 40+2);
		grabshort(&hdrp->dime.dim[2], hdrbuf, 40+4);
		grabshort(&hdrp->dime.dim[3], hdrbuf, 40+6);
		grabshort(&hdrp->dime.dim[4], hdrbuf, 40+8);
		grabshort(&hdrp->dime.dim[5], hdrbuf, 40+10);
		grabshort(&hdrp->dime.dim[6], hdrbuf, 40+12);
		grabshort(&hdrp->dime.dim[7], hdrbuf, 40+14);
		grabshort(&hdrp->dime.unused8, hdrbuf, 40+16);
		grabshort(&hdrp->dime.unused9, hdrbuf, 40+18);
		grabshort(&hdrp->dime.unused10, hdrbuf, 40+20);
		grabshort(&hdrp->dime.unused11, hdrbuf, 40+22);
		grabshort(&hdrp->dime.unused12, hdrbuf, 40+24);
		grabshort(&hdrp->dime.unused13, hdrbuf, 40+26);
		grabshort(&hdrp->dime.unused14, hdrbuf, 40+28);
		grabshort(&hdrp->dime.datatype, hdrbuf, 40+30);
		grabshort(&hdrp->dime.bitpix, hdrbuf, 40+32);
		grabshort(&hdrp->dime.dim_un0, hdrbuf, 40+34);
		grablong(&hdrp->dime.pixdim[0], hdrbuf, 40+36);
		grablong(&hdrp->dime.pixdim[1], hdrbuf, 40+40);
		grablong(&hdrp->dime.pixdim[2], hdrbuf, 40+44);
		grablong(&hdrp->dime.pixdim[3], hdrbuf, 40+48);
		grablong(&hdrp->dime.pixdim[4], hdrbuf, 40+52);
		grablong(&hdrp->dime.pixdim[5], hdrbuf, 40+56);
		grablong(&hdrp->dime.pixdim[6], hdrbuf, 40+60);
		grablong(&hdrp->dime.pixdim[7], hdrbuf, 40+64);
		grablong(&hdrp->dime.vox_offset, hdrbuf, 40+68);
		grablong(&hdrp->dime.funused1, hdrbuf, 40+72);
		grablong(&hdrp->dime.funused2, hdrbuf, 40+76);
		grablong(&hdrp->dime.funused3, hdrbuf, 40+80);
		grablong(&hdrp->dime.cal_max, hdrbuf, 40+84);
		grablong(&hdrp->dime.cal_min, hdrbuf, 40+88);
		grablong(&hdrp->dime.compressed, hdrbuf, 40+92);
		grablong(&hdrp->dime.verified, hdrbuf, 40+96);
		grablong(&hdrp->dime.glmax, hdrbuf, 40+100);
		grablong(&hdrp->dime.glmin, hdrbuf, 40+104);

		strncpy(&hdrp->hist.descrip[0], &hdrbuf[148+0], 80);
		strncpy(&hdrp->hist.aux_file[0], &hdrbuf[148+80], 24);
		hdrp->hist.orient = hdrbuf[148+104];
		strncpy(&hdrp->hist.originator[0], &hdrbuf[148+105], 10);
		strncpy(&hdrp->hist.generated[0], &hdrbuf[148+115], 10);
		strncpy(&hdrp->hist.scannum[0], &hdrbuf[148+125], 10);
		strncpy(&hdrp->hist.patient_id[0], &hdrbuf[148+135], 10);
		strncpy(&hdrp->hist.exp_date[0], &hdrbuf[148+145], 10);
		strncpy(&hdrp->hist.exp_time[0], &hdrbuf[148+155], 10);
		strncpy(&hdrp->hist.hist_un0[0], &hdrbuf[148+165], 3);
		grablong(&hdrp->hist.views, hdrbuf, 148+168);
		grablong(&hdrp->hist.vols_added, hdrbuf, 148+172);
		grablong(&hdrp->hist.start_field, hdrbuf, 148+176);
		grablong(&hdrp->hist.field_skip, hdrbuf, 148+180);
		grablong(&hdrp->hist.omax, hdrbuf, 148+184);
		grablong(&hdrp->hist.omin, hdrbuf, 148+188);
		grablong(&hdrp->hist.smax, hdrbuf, 148+192);
		grablong(&hdrp->hist.smin, hdrbuf, 148+196);

		if (!nofixpixdim) {
		    int i = 0;
		    for (i = 1; i <= hdrp->dime.dim[0]; i++) {
			// nifti1_io.c "corrects" bad pixdims so we'll do it too
			if (hdrp->dime.pixdim[i] == 0.0) {
			    fprintf(stderr, "Warning: \"correcting\" pixdim[%d] to 1.0 (original value is 0.0).  Specify --nofixpixdim to leave it 0.0\n", i);
			    hdrp->dime.pixdim[i] = 1.0;
			}
		    }
		}
	    }

	    datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
	    memset(datarec, '\0', sizeof(bxhrawdatarec));

	    if (hdrp->dime.datatype == DT_BINARY) {
		fprintf(stderr, "createDocFromAnalyze: Analyze BINARY format not currently supported!\n");
		goto FAIL;
	    } else if (isnifti && nifp->datatype == NIFTI_TYPE_INT8) {
		datarec->elemtype = strdup("int8");
		datarec->elemsize = 1;
	    } else if (isnifti && nifp->datatype == NIFTI_TYPE_UINT16) {
		datarec->elemtype = strdup("uint16");
		datarec->elemsize = 2;
	    } else if (isnifti && nifp->datatype == NIFTI_TYPE_UINT32) {
		datarec->elemtype = strdup("uint32");
		datarec->elemsize = 4;
	    } else if (isnifti && nifp->datatype == NIFTI_TYPE_INT64) {
		datarec->elemtype = strdup("int64");
		datarec->elemsize = 8;
	    } else if (isnifti && nifp->datatype == NIFTI_TYPE_UINT64) {
		datarec->elemtype = strdup("uint64");
		datarec->elemsize = 8;
	    } else if (hdrp->dime.datatype == DT_UNSIGNED_CHAR) {
		datarec->elemtype = strdup("uint8");
		datarec->elemsize = 1;
	    } else if (hdrp->dime.datatype == DT_SIGNED_CHAR) {
		datarec->elemtype = strdup("int8");
		datarec->elemsize = 1;
	    } else if (hdrp->dime.datatype == DT_UNSIGNED_SHORT) {
		datarec->elemtype = strdup("uint16");
		datarec->elemsize = 2;
	    } else if (hdrp->dime.datatype == DT_SIGNED_SHORT) {
		datarec->elemtype = strdup("int16");
		datarec->elemsize = 2;
	    } else if (hdrp->dime.datatype == DT_UNSIGNED_INT) {
		datarec->elemtype = strdup("uint32");
		datarec->elemsize = 4;
	    } else if (hdrp->dime.datatype == DT_SIGNED_INT) {
		datarec->elemtype = strdup("int32");
		datarec->elemsize = 4;
	    } else if (hdrp->dime.datatype == DT_FLOAT) {
		datarec->elemtype = strdup("float32");
		datarec->elemsize = 4;
	    } else if (hdrp->dime.datatype == DT_COMPLEX) {
		fprintf(stderr, "createDocFromAnalyze: Analyze COMPLEX format not currently supported!\n");
		goto FAIL;
	    } else if (hdrp->dime.datatype == DT_DOUBLE) {
		datarec->elemtype = strdup("float64");
		datarec->elemsize = 8;
	    } else if (hdrp->dime.datatype == DT_RGB) {
		datarec->elemtype = strdup("rgb24");
		datarec->elemsize = 3;
	    /* the following are SPM "byte-swapped" datatypes */
	    } else if (!isnifti && hdrp->dime.datatype == DT_UNSIGNED_CHAR * 256) {
		datarec->elemtype = strdup("uint8");
		datarec->elemsize = 1;
		spm_byteswap = 1;
	    } else if (!isnifti && hdrp->dime.datatype == DT_SIGNED_CHAR * 256) {
		/* this will never happen because of range of dime.datatype */
		datarec->elemtype = strdup("int8");
		datarec->elemsize = 1;
		spm_byteswap = 1;
	    } else if (!isnifti && hdrp->dime.datatype == DT_UNSIGNED_SHORT * 256) {
		/* this will never happen because of range of dime.datatype */
		datarec->elemtype = strdup("uint8");
		datarec->elemsize = 2;
		spm_byteswap = 1;
	    } else if (!isnifti && hdrp->dime.datatype == DT_SIGNED_SHORT * 256) {
		datarec->elemtype = strdup("int16");
		datarec->elemsize = 2;
		spm_byteswap = 1;
	    } else if (!isnifti && hdrp->dime.datatype == DT_UNSIGNED_INT * 256) {
		/* this will never happen because of range of dime.datatype */
		datarec->elemtype = strdup("uint32");
		datarec->elemsize = 4;
		spm_byteswap = 1;
	    } else if (!isnifti && hdrp->dime.datatype == DT_SIGNED_INT * 256) {
		datarec->elemtype = strdup("int32");
		datarec->elemsize = 4;
		spm_byteswap = 1;
	    } else if (!isnifti && hdrp->dime.datatype == DT_FLOAT * 256) {
		datarec->elemtype = strdup("float32");
		datarec->elemsize = 4;
		spm_byteswap = 1;
	    } else if (!isnifti && hdrp->dime.datatype == DT_DOUBLE * 256) {
		datarec->elemtype = strdup("float64");
		datarec->elemsize = 8;
		spm_byteswap = 1;
	    } else {
		fprintf(stderr, "createDocFromAnalyze: Analyze pixel format %d not supported!\n", hdrp->dime.datatype);
		goto FAIL;
	    }

	    datarec->numdims = hdrp->dime.dim[0];
	    while (datarec->numdims > 3) {
		if (hdrp->dime.dim[datarec->numdims - 1 + 1] > 1)
		    break;
		datarec->numdims--;
	    }
	    if (numinfiles > 1 && hdrp->dime.dim[datarec->numdims - 1 + 1] != 1)
		datarec->numdims++;
	    datarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension)*datarec->numdims);
	    memset(datarec->dimensions, '\0', sizeof(bxhdimension)*datarec->numdims);
	    if (datarec->numdims >= 1) {
		bxhdimension * dim = &datarec->dimensions[0];
		dim->type = strdup("x");
		dim->units = NULL;
		dim->size = hdrp->dime.dim[1];
		dim->origin = 0;
		dim->gap = 0;
		dim->spacing = fabs((double)hdrp->dime.pixdim[1]);
		dim->direction = (double *)malloc(sizeof(double)*3);
		memset(dim->direction, '\0', sizeof(double)*3);
		dim->numdpstructs = 0;
		dim->dpstructs = NULL;
		dimx = dim;
	    }
	    if (datarec->numdims >= 2) {
		bxhdimension * dim = &datarec->dimensions[1];
		dim->type = strdup("y");
		dim->units = NULL;
		dim->size = hdrp->dime.dim[2];
		dim->origin = 0;
		dim->gap = 0;
		dim->spacing = fabs((double)hdrp->dime.pixdim[2]);
		dim->direction = (double *)malloc(sizeof(double)*3);
		memset(dim->direction, '\0', sizeof(double)*3);
		dim->numdpstructs = 0;
		dim->dpstructs = NULL;
		dimy = dim;
	    }
	    if (datarec->numdims >= 3) {
		bxhdimension * dim = &datarec->dimensions[2];
		dim->type = strdup("z");
		dim->units = NULL;
		dim->size = hdrp->dime.dim[3];
		dim->origin = 0;
		dim->gap = 0;
		dim->spacing = fabs((double)hdrp->dime.pixdim[3]);
		dim->direction = (double *)malloc(sizeof(double)*3);
		memset(dim->direction, '\0', sizeof(double)*3);
		dim->numdpstructs = 0;
		dim->dpstructs = NULL;
		dimz = dim;
	    }
	    if (datarec->numdims >= 4) {
		bxhdimension * dim = &datarec->dimensions[3];
		dim->type = strdup("t");
		dim->units = NULL;
		dim->size = hdrp->dime.dim[4];
		dim->origin = 0;
		dim->gap = 0;
		dim->spacing = hdrp->dime.pixdim[4];
		dim->direction = NULL;
		dim->numdpstructs = 0;
		dim->dpstructs = NULL;
	    }
	    for (dimnum = 4; dimnum < datarec->numdims; dimnum++) {
		bxhdimension * dim = &datarec->dimensions[dimnum];
		dim->type = strdup("00000000");
		sprintf(dim->type, "%d", dimnum);
		dim->units = NULL;
		dim->size = hdrp->dime.dim[dimnum+1];
		dim->origin = 0;
		dim->gap = 0;
		dim->spacing = 0;
		dim->direction = NULL;
		dim->numdpstructs = 0;
		dim->dpstructs = NULL;
	    }

/* 	    if (numinfiles > 1) { */
/* 		bxhdimension * lastdim = NULL; */
/* 		if (datarec->dimensions[datarec->numdims - 1].size != 1) { */
/* 		    lastdim = &datarec->dimensions[datarec->numdims - 1]; */
/* 		    memset(lastdim, '\0', sizeof(*lastdim)); */
/* 		} */
/* 		lastdim = &datarec->dimensions[datarec->numdims - 1]; */
/* 		lastdim->size = numinfiles; */
/* 	    } */

	    /* set default orientation to RAS (may be overridden later) */
	    for (dimnum = 0; dimnum < 3 && dimnum < datarec->numdims; dimnum++) {
		double * direction = datarec->dimensions[dimnum].direction;
		memset(direction, '\0', sizeof(double)*3);
		direction[dimnum] = 1; /* dim<R>[R]=1, dim<A>[A]=1, dim<S>[S]=1 */
		if (hdrp->dime.pixdim[dimnum+1] < 0) {
		    direction[dimnum] = -1;
		}
	    }

	    /* set default origin to -1/2 FOV (may be overridden later)
	     * (deals with negative pixdim as used in old Analyze files)
	     * (also adjusted to point to center of first voxel) */
	    for (dimnum = 0; dimnum < 3 && dimnum < datarec->numdims; dimnum++) {
		bxhdimension * dimp = &datarec->dimensions[dimnum];
		dimp->origin = -0.5 * hdrp->dime.pixdim[dimnum+1] * (dimp->size-1);
	    }

	    /* Check NIfTI-1 orientation fields */
	    if (isnifti && (nifp->qform_code > 0 || nifp->sform_code > 0)) {
		mat44 R;
		memset(&R, '\0', sizeof(R));
		if (nifp->sform_code > 0) {
		    R.m[0][0] = nifp->srow_x[0];
		    R.m[0][1] = nifp->srow_x[1];
		    R.m[0][2] = nifp->srow_x[2];
		    R.m[1][0] = nifp->srow_y[0];
		    R.m[1][1] = nifp->srow_y[1];
		    R.m[1][2] = nifp->srow_y[2];
		    R.m[2][0] = nifp->srow_z[0];
		    R.m[2][1] = nifp->srow_z[1];
		    R.m[2][2] = nifp->srow_z[2];
		    R.m[0][3] = nifp->srow_x[3];
		    R.m[1][3] = nifp->srow_y[3];
		    R.m[2][3] = nifp->srow_z[3];
		} else if (nifp->qform_code > 0) {
		    R = nifti_quatern_to_mat44(
			nifp->quatern_b, nifp->quatern_c, nifp->quatern_d,
			nifp->qoffset_x, nifp->qoffset_y, nifp->qoffset_z,
			1, 1, 1,
			nifp->pixdim[0]);
		}
		for (dimnum = 0; dimnum < 3 && dimnum < datarec->numdims; dimnum++) {
		    bxhdimension * dimp = &datarec->dimensions[dimnum];
		    double * dir = dimp->direction;
		    double len = sqrt(
			((double)R.m[0][dimnum] * (double)R.m[0][dimnum]) +
			((double)R.m[1][dimnum] * (double)R.m[1][dimnum]) +
			((double)R.m[2][dimnum] * (double)R.m[2][dimnum]));
		    dir[0] = (double)R.m[0][dimnum] / len;
		    dir[1] = (double)R.m[1][dimnum] / len;
		    dir[2] = (double)R.m[2][dimnum] / len;
                    double adr = fabs(dir[0]);
                    double ada = fabs(dir[1]);
                    double ads = fabs(dir[2]);
                    if (adr > ada && adr > ads) {
		        dimp->origin = R.m[0][3];
                    } else if (ada > adr && ada > ads) {
		        dimp->origin = R.m[1][3];
                    } else if (ads > adr && ads > ada) {
		        dimp->origin = R.m[2][3];
                    }
		}
	    } else if (!strictanalyze && mmatp) {
		/* Use .mat file if it exists */
		mmatrix mmat;
		double oR, oA, oS;
		int i;
		for (i = 0; i < 16; i++) {
		    mmat[i] = (*mmatp)[i];
		}
		/* .mat file matrix uses (1,1,1) as the origin --
		 * calculate it using M * [1 1 1 1]' */
		oR = mmat[ 0] * 1 + mmat[ 1] * 1 + mmat[ 2] * 1 + mmat[ 3] * 1;
		oA = mmat[ 4] * 1 + mmat[ 5] * 1 + mmat[ 6] * 1 + mmat[ 7] * 1;
		oS = mmat[ 8] * 1 + mmat[ 9] * 1 + mmat[10] * 1 + mmat[11] * 1;
	
		for (dimnum = 0; dimnum < 3 && dimnum < datarec->numdims; dimnum++) {
		    double dR, dA, dS;
		    double dlen = 0;
		    double * direction = datarec->dimensions[dimnum].direction;
		    double * originp = &datarec->dimensions[dimnum].origin;

		    dR = mmat[0*4 + dimnum];
		    dA = mmat[1*4 + dimnum];
		    dS = mmat[2*4 + dimnum];
		    dlen = sqrt((dR*dR) + (dA*dA) + (dS*dS));
		    dR /= dlen;
		    dA /= dlen;
		    dS /= dlen;
		    if (fabs(dR) >= fabs(dA) && fabs(dR) >= fabs(dS)) {
			*originp = oR; /* this dimension is R-L */
		    } else if (fabs(dA) >= fabs(dR) && fabs(dA) >= fabs(dS)) {
			*originp = oA; /* this dimension is A-P */
		    } else if (fabs(dS) >= fabs(dR) && fabs(dS) >= fabs(dA)) {
			*originp = oS; /* this dimension is S-I */
		    }

		    direction[0] = dR;
		    direction[1] = dA;
		    direction[2] = dS;
		}
	    }

	    if (isnifti) {
		double temporalfactor = 1.0;
		double spatialfactor = 1.0;
		switch (nifp->xyzt_units & 0x07) {
		case NIFTI_UNITS_UNKNOWN:
		    break;
		case NIFTI_UNITS_METER:
		    spatialfactor = 1000.0;
		    break;
		case NIFTI_UNITS_MM:
		    break;
		case NIFTI_UNITS_MICRON:
		    spatialfactor = 1.0/1000.0;
		    break;
		default: break;
		}
		switch(nifp->xyzt_units & 0x38) {
		case NIFTI_UNITS_UNKNOWN:
		    break;
		case NIFTI_UNITS_SEC:
		    temporalfactor = 1000.0;
		    break;
		case NIFTI_UNITS_MSEC:
		    break;
		case NIFTI_UNITS_USEC:
		    temporalfactor = 1.0/1000.0;
		    break;
		default: break;
		}
		for (dimnum = 0; dimnum < 3 && dimnum < datarec->numdims; dimnum++) {
		    datarec->dimensions[dimnum].units = strdup("mm");
		    datarec->dimensions[dimnum].origin *= spatialfactor;
		    datarec->dimensions[dimnum].gap *= spatialfactor;
		    datarec->dimensions[dimnum].spacing *= spatialfactor;
		}
		if (datarec->numdims >= 4) {
		    datarec->dimensions[3].units = strdup("ms");
		    datarec->dimensions[3].origin *= temporalfactor;
		    datarec->dimensions[3].gap *= temporalfactor;
		    datarec->dimensions[3].spacing *= temporalfactor;
		}

		if (dimz != NULL && nifp->slice_code != 0 && nifp->slice_start == 0 && nifp->slice_end == dimz->size - 1) {
		    char * curstrptr = NULL;
		    int maxdigits = 1;
		    int numslices = dimz->size;
		    int numsliceshalf = (dimz->size + 1) / 2;
		    int sliceind;
		    char * numbuf = NULL;
		    char ** datapoints = (char **)malloc(sizeof(char *) * numslices);
		    dimz->dpstructs = (bxhdatapoints *)realloc(dimz->dpstructs, sizeof(bxhdatapoints) * (dimz->numdpstructs + 1));
		    bxhdatapoints * dpstructp = &dimz->dpstructs[dimz->numdpstructs];
		    dimz->numdpstructs++;
		    dpstructp->label = strdup("acquisitiontimeindex");
		    dpstructp->values = datapoints;
		    dpstructp->numvalues = numslices;
		    {
			int tmpsize = numslices;
			while (tmpsize > 10) {
			    maxdigits++;
			    tmpsize /= 10;
			}
		    }
		    numbuf = (char *)malloc(sizeof(char) * (maxdigits + 2));
		    sliceorderstr = (char *)malloc(sizeof(char) * ((maxdigits + 1) * numslices) + 1);
		    curstrptr = sliceorderstr;
		    for (sliceind = 0; sliceind < numslices; sliceind++) {
			int sliceorderval = 0;
			int acqtimeind = 0;
			if (nifp->slice_code == NIFTI_SLICE_SEQ_INC) {
			    sliceorderval = sliceind + 1;
			    acqtimeind = sliceind + 1;
			} else if (nifp->slice_code == NIFTI_SLICE_SEQ_DEC) {
			    sliceorderval = numslices - sliceind;
			    acqtimeind = numslices - sliceind;
			} else if (nifp->slice_code == NIFTI_SLICE_ALT_INC) {
			    if (sliceind < numsliceshalf) {
				sliceorderval = (sliceind * 2) + 1;
			    } else {
				sliceorderval = ((sliceind - numsliceshalf) * 2) + 2;
			    }
			    if (sliceind % 2 == 0) {
				acqtimeind = sliceind / 2 + 1;
			    } else {
				acqtimeind = numsliceshalf + (sliceind / 2) + 1;
			    }
			} else if (nifp->slice_code == NIFTI_SLICE_ALT_DEC) {
			    if (sliceind < numsliceshalf) {
				sliceorderval = numslices - 1 - (sliceind * 2);
			    } else {
				sliceorderval = numslices - 1 - ((sliceind - numsliceshalf) * 2) - 1;
			    }
			    if ((numslices - 1 - sliceind) % 2 == 0) {
				acqtimeind = (numslices - 1 - sliceind) / 2 + 1;
			    } else {
				acqtimeind = numsliceshalf + ((numslices - 1 - sliceind) / 2) + 1;
			    }
			} else if (nifp->slice_code == NIFTI_SLICE_ALT_INC2) {
			    if (sliceind < numsliceshalf) {
				sliceorderval = (sliceind * 2) + 2;
			    } else {
				sliceorderval = ((sliceind - numsliceshalf) * 2) + 1;
			    }
			    if (sliceind % 2 == 0) {
				acqtimeind = numsliceshalf + (sliceind / 2) + 1;
			    } else {
				acqtimeind = sliceind / 2 + 1;
			    }
			} else if (nifp->slice_code == NIFTI_SLICE_ALT_DEC2) {
			    if (sliceind < numsliceshalf) {
				sliceorderval = numslices - 1 - (sliceind * 2) - 1;
			    } else {
				sliceorderval = numslices - 1 - ((sliceind - numsliceshalf) * 2);
			    }
			    if ((numslices - 1 - sliceind) % 2 == 0) {
				acqtimeind = numsliceshalf + ((numslices - 1 - sliceind) / 2) + 1;
			    } else {
				acqtimeind = (numslices - 1 - sliceind) / 2 + 1;
			    }
			}
			sprintf(numbuf, "%d", acqtimeind);
			datapoints[sliceind] = strdup(numbuf);
			if (sliceind == 0) {
			    sprintf(numbuf, "%d", sliceorderval);
			} else {
			    sprintf(numbuf, ",%d", sliceorderval);
			}
			strcpy(curstrptr, numbuf);
			curstrptr += strlen(curstrptr);
		    }
		    free(numbuf); numbuf = NULL;
		}
	    } else {
		if (datarec->numdims >= 4) {
		    datarec->dimensions[3].spacing *= 1000.0; /* convert from secs to msec */
		}
	    }

	    /* use Analyze 'orient' field if --strictanalyze and
	     * not overridden by --orientation. */
	    if (strictanalyze && orient == NULL) {
		switch (hdrp->hist.orient) {
		case 0: orient = "las"; break;
		case 1: orient = "lsa"; break;
		case 2: orient = "asl"; break;
		case 3: orient = "lps"; break;
		case 4: orient = "lia"; break;
		case 5: orient = "ail"; break;
		default:
		    fprintf(stderr, "Bad value (%d) in Analyze 'orient' field\n",
			    hdrp->hist.orient);
		    goto FAIL;
		}
	    }

	    /* override if user specifies orientation explicitly */
	    if (orient) {
		for (dimnum = 0; dimnum < 3 && dimnum < datarec->numdims; dimnum++) {
		    double * direction = datarec->dimensions[dimnum].direction;
		    datarec->dimensions[dimnum].origin = 0; /* reset origin */
		    memset(direction, '\0', sizeof(double)*3);
		    if (tolower(orient[dimnum]) == 'r') {
			direction[0] = 1;
		    } else if (tolower(orient[dimnum]) == 'l') {
			direction[0] = -1;
		    } else if (tolower(orient[dimnum]) == 'a') {
			direction[1] = 1;
		    } else if (tolower(orient[dimnum]) == 'p') {
			direction[1] = -1;
		    } else if (tolower(orient[dimnum]) == 's') {
			direction[2] = 1;
		    } else if (tolower(orient[dimnum]) == 'i') {
			direction[2] = -1;
		    }
		}
	    }

	    if (!orientwarned && !strictanalyze && orient == NULL && !mmatp) {
		if (!isnifti) {
		    fprintf(stderr, "createDocFromAnalyze: Warning: will assume default orientation of RAS (pure axial, stored in neurological convention)\n");
		    orientwarned = 1;
		} else if (nifp->qform_code == 0 && nifp->sform_code == 0) {
		    /* this NIFTI file has no orientation -- assume it knows what it's doing */
		    for (dimnum = 0; dimnum < 3 && dimnum < datarec->numdims; dimnum++) {
			if (datarec->dimensions[dimnum].direction != NULL) {
			    free(datarec->dimensions[dimnum].direction);
			    datarec->dimensions[dimnum].direction = NULL;
			}
		    }
		}
	    }

	    /* first byte of header size will be 0x00 if big-endian */
	    if (hdrbuf[0] == 0)
		datarec->msbfirstfrags = 1;
	    else
		datarec->msbfirstfrags = 0;

	    if (spm_byteswap) {
		datarec->msbfirstfrags = !datarec->msbfirstfrags;
	    }

	    if (olddatarec != NULL) {
		int dimnum;
		/* double check that this .hdr has the same values
		 * as all previously read headers */
		if (datarec->numdims != olddatarec->numdims) {
		    fprintf(stderr, "Field 'numdims'");
		    goto HDRCMPFAIL;
		}
		for (dimnum = 0; dimnum < datarec->numdims - 1; dimnum++) {
		    if (datarec->dimensions[dimnum].size !=
			olddatarec->dimensions[dimnum].size) {
			fprintf(stderr, "Size of dimension %d (%d != %d)", dimnum, (int)datarec->dimensions[dimnum].size, (int)olddatarec->dimensions[dimnum].size);
			goto HDRCMPFAIL;
		    }
		}
		if (strcmp(datarec->elemtype, olddatarec->elemtype) != 0) {
		    fprintf(stderr, "Datatype");
		    goto HDRCMPFAIL;
		}
		if (datarec->msbfirstfrags != olddatarec->msbfirstfrags) {
		    fprintf(stderr, "Endian-ness");
		    goto HDRCMPFAIL;
		}

	    }

	    {
		size_t fragsize = 1;
		char * newimgname = strdup(imgname);
		for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		    fragsize *= datarec->dimensions[dimnum].size;
		}
		if (numinfiles > 1) {
		    fragsize /= datarec->dimensions[datarec->numdims - 1].size;
		}
		fragsize *= datarec->elemsize;
		
		if (olddatarec) {
		    olddatarec->dimensions[datarec->numdims-1].size +=
			datarec->dimensions[datarec->numdims-1].size;
		    bxh_datarec_free(datarec);
		    datarec = olddatarec;
		}

		if (iscompressed) {
		    /* get rid of .gz extension */
		    newimgname[strlen(newimgname)-3] = '\0';
		}

		bxh_datarec_addfrag(datarec, newimgname, (size_t)hdrp->dime.vox_offset, fragsize, ofname, 1);
		free(newimgname);
	    }

	    goto HDRCMPEXIT;

	  HDRCMPFAIL:
	    fprintf(stderr, " does not match in file '%s' and '%s'\n", ifnames[0], ifnames[infilenum]);
	    goto FAIL;

	  HDRCMPEXIT:
	    if (matname) {
		free(matname);
		matname = NULL;
	    }
	    if (hdrname) {
		free(hdrname);
		hdrname = NULL;
	    }
	    if (imgname) {
		free(imgname);
		imgname = NULL;
	    }
	    if (mmatp) {
		free(mmatp);
		mmatp = NULL;
	    }
	    /* null */;
	} else {
	    fprintf(stderr, "createDocFromAnalyze: file %s is not Analyze 7.5, AVW_ImageFile, or AVW_VolumeFile?\n(also, AVW_VolumeFile pointing to non-Analyze types is not supported)\n", ifnames[infilenum]);
	    goto FAIL;
	}
    }    

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;

    if ((subject = bxh_getSubject(docp)) == NULL)
	goto FAIL;

    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;

    if (sliceorderstr != NULL &&
	sliceorderstr[0] != '\0' &&
	bxh_appendChildElement(acquisitiondata, "sliceorder", sliceorderstr) != 0)
	goto FAIL;

    {
	char * hintnames[4] = { "x", "y", "z", "t" };
	int dimnum;
	if (hintsize[0] != (size_t)-1 ||
	    hintsize[1] != (size_t)-1 ||
	    hintsize[2] != (size_t)-1 ||
	    hintsize[3] != (size_t)-1) {
	    fprintf(stderr, "Warning: --hintsize* options are ignored for Analyze files\n");
	}
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &datarec->dimensions[dimnum];
	    int hintnum;
	    for (hintnum = 0; hintnum < 4; hintnum++) {
		if (strcmp(dimp->type, hintnames[hintnum]) == 0)
		    break;
	    }
	    if (hintnum == 4) continue;
	    if (hintorigin[hintnum] != HUGE_VAL)
		dimp->origin = hintorigin[hintnum];
	    if (hintspacing[hintnum] != HUGE_VAL)
		dimp->spacing = hintspacing[hintnum];
	    if (hintgap[hintnum] != HUGE_VAL)
		dimp->gap = hintgap[hintnum];
	}
    }

    bxh_datarec_writeToElement(imagedata, datarec);

    goto EXIT;

FAIL:
    fprintf(stderr, "createDocFromAnalyze: conversion failed.\n");
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    if (docp) {
	bxh_freeDocument(docp);
    }
    docp = NULL;

EXIT:
    if (ifnames) {
	for (infilenum = 0; infilenum < numinfiles; infilenum++) {
	    free(ifnames[infilenum]); ifnames[infilenum] = NULL;
	}
	free(ifnames); ifnames = NULL;
    }
    if (datarec)
	bxh_datarec_free(datarec);
    if (subject)
	bxh_element_unref(subject);
    if (acquisitiondata)
	bxh_element_unref(acquisitiondata);
    if (imagedata)
	bxh_element_unref(imagedata);
    if (sliceorderstr) {
	free(sliceorderstr);
	sliceorderstr == NULL;
    }

    return docp;
}

int
createBXHFromAnalyze(const char **inifnames, int numinfiles, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient, char * orient, int strictanalyze, char * avwbyteorder, int nofixpixdim)
{
    int result = 0;
    struct stat statbuf;
    BXHDocPtr docp = NULL;

    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "createBXHFromAnalyze: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromAnalyze(inifnames, numinfiles, ofname, hintsize, hintorigin, hintspacing, hintgap, forceorient, orient, strictanalyze, avwbyteorder, nofixpixdim)) == NULL)
	goto FAIL;
    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

  FAIL:
    result = -1;

  EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }
    return result;
}

BXHDocPtr
bxh_analyze2doc(const char * filename)
{
    size_t hintsizes[4] = { (size_t)-1, (size_t)-1, (size_t)-1, (size_t)-1 };
    double hintorigin[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintspacing[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintgap[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    return createDocFromAnalyze(&filename, 1, "dummy.bxh", &hintsizes[0], &hintorigin[0], &hintspacing[0], &hintgap[0], NULL, NULL, 0, NULL, 0);
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.47  2008/12/08 16:53:34  gadde
 * Fix types for 64-bit machines.
 *
 * Revision 1.46  2008/08/08 20:17:30  gadde
 * Fix extraction of sform.
 *
 * Revision 1.45  2008/05/22 15:49:00  gadde
 * Convert to double before math expressions to avoid accumulating rounding errors.
 *
 * Revision 1.44  2007/09/14 20:45:58  gadde
 * Change default origin from 0,0,0 to half FOV.
 *
 * Revision 1.43  2007/08/17 15:33:57  gadde
 * Fix memory bug.
 *
 * Revision 1.42  2007/02/23 20:08:59  gadde
 * When reading .mat files, use 'mat' matrix (SPM2 convention) if available,
 * otherwise use 'M'.
 *
 * Revision 1.41  2007/01/16 21:49:19  gadde
 * Fix temporal units for output XML file.
 *
 * Revision 1.40  2007/01/16 20:22:40  gadde
 * Make sure output .bxh has units of ms and mm.
 *
 * Revision 1.39  2006/10/20 17:01:49  gadde
 * Fix simple unsigned/signed bug.
 *
 * Revision 1.38  2006/10/04 21:16:30  gadde
 * Deal with negative pixdims (interpret them as reversing the
 * default RAS convention for that dimension).
 * Work better with compressed .mat files and .mat files that
 * need to be byte-swapped.
 *
 * Revision 1.37  2006/09/22 14:19:10  gadde
 * Fix t spacing for NIfTI
 *
 * Revision 1.36  2006/08/04 16:29:26  gadde
 * Fix populating of .mat files.
 *
 * Revision 1.35  2006/07/25 19:27:34  gadde
 * Fix .mat file reading and also only use it if NIFTI transforms are missing.
 *
 * Revision 1.34  2006/05/19 14:37:42  gadde
 * Support gzipped nifti.
 *
 * Revision 1.33  2006/05/05 14:55:18  gadde
 * Actually represent the SPM byteswap this time.
 *
 * Revision 1.32  2006/05/05 14:51:57  gadde
 * Add SPM byte-swap datatypes.
 *
 * Revision 1.31  2006/02/08 19:01:31  gadde
 * Update to support nifti one-file format and sform_code
 *
 * Revision 1.30  2006/01/03 19:23:57  gadde
 * Support compressed elements in Level 5 .mat files.
 *
 * Revision 1.29  2005/12/16 18:25:57  gadde
 * Add level 5 MAT-file support.
 *
 * Revision 1.28  2005/09/19 16:31:55  gadde
 * Documentation and help message updates.
 *
 * Revision 1.27  2005/09/14 14:49:27  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.26  2005/08/11 18:55:40  gadde
 * Add pixdim[4] for t-dimension spacing.
 *
 * Revision 1.25  2005/05/10 19:48:54  gadde
 * Fix some memory bugs.
 *
 * Revision 1.24  2005/04/26 18:39:37  gadde
 * Update for new version of NIfTI library.
 *
 * Revision 1.23  2005/03/08 18:09:57  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.22  2004/09/17 13:54:20  gadde
 * Fix some indexing errors.
 *
 * Revision 1.21  2004/06/18 15:21:51  gadde
 * Standardize frag creation (redux)
 *
 * Revision 1.20  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.19  2004/04/07 19:31:40  gadde
 * Add units and slice duration.
 *
 * Revision 1.18  2004/04/07 19:02:07  gadde
 * Add NIfTI-1 support.
 *
 * Revision 1.17  2004/02/24 18:01:06  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.16  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.15  2003/11/18 22:10:25  gadde
 * Make sure user-specified orientation trumps all.
 *
 * Revision 1.14  2003/11/17 21:43:27  gadde
 * Fix last-dimension size calculation.
 *
 * Revision 1.13  2003/11/07 16:05:28  gadde
 * Add support for AVW_ImageFile/AVW_VolumeFile.
 *
 * Revision 1.12  2003/11/05 18:38:21  gadde
 * One more byte order fix.
 *
 * Revision 1.11  2003/11/05 18:34:44  gadde
 * Fix some byte-swapping issues.
 *
 * Revision 1.10  2003/11/05 17:23:59  gadde
 * Use .mat files for orientation if they exist.
 * Add option to ignore them and use Analyze .orient field (--strictanalyze).
 *
 * Revision 1.9  2003/09/08 20:22:54  gadde
 * Populate nth dimension if numinfiles > 1
 *
 * Revision 1.8  2003/08/19 19:35:41  gadde
 * New options scheme, so that options are available using whichever form of bxhabsorb you use.
 *
 * Revision 1.7  2003/07/29 17:03:09  gadde
 * Get rid of singleton dimensions (if dim > 3)
 *
 * Revision 1.6  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.5  2003/06/24 14:40:13  gadde
 * Support multiple input files.
 *
 * Revision 1.4  2003/06/18 16:14:15  gadde
 * win32 fixes
 *
 * Revision 1.3  2003/06/18 16:08:17  gadde
 * fopen in binary mode
 *
 * Revision 1.2  2003/06/06 20:07:33  gadde
 * Update documentation.
 *
 * Revision 1.1  2003/06/06 19:41:43  gadde
 * Initial commit.
 *
 */
