static const char rcsid[] = "$Id: bxh_convert_iowasignafive.c,v 1.1 2009-01-15 20:55:18 gadde Exp $";

/*
 * signafive2bxh.cc --
 * 
 * Creates a BXH file based on given Signa 5 files.
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include "bxh_signamaps.h"
#include "bxh_utils.h"

extern BXHDocPtr createDocFromSigna5(const char **ifnames, int numfiles, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient, size_t dimzsize, size_t dimtsize);

static struct signamaps * signamap3list[3] = { &signamaps3align0, &signamaps3align2, NULL };

extern char * domutil_errorbuf;

#define ISIZE_FILEDATA 2304
#define ISIZE_SUITEDATA 114
#define ISIZE_EXAMDATA 1024
#define ISIZE_SERIESDATA 1020
#define ISIZE_MRIMAGEDATA 1022
#define ISIZE_USERDATA 2420
#define IOFFSET_FILEDATA 0
#define IOFFSET_SUITEDATA ((IOFFSET_FILEDATA) + (ISIZE_FILEDATA))
#define IOFFSET_EXAMDATA ((IOFFSET_SUITEDATA) + (ISIZE_SUITEDATA))
#define IOFFSET_SERIESDATA ((IOFFSET_EXAMDATA) + (ISIZE_EXAMDATA))
#define IOFFSET_MRIMAGEDATA ((IOFFSET_SERIESDATA) + (ISIZE_SERIESDATA))
#define IOFFSET_USERDATA ((IOFFSET_MRIMAGEDATA) + (ISIZE_MRIMAGEDATA))

static int
grabAndSwapVal(FILE * fp, off_t off, size_t size, void * data)
{
    unsigned int tmpint;
    if (fseek(fp, off, SEEK_SET) == -1) {
        perror("fseek");
        return -1;
    }
    if (fread(data, size, 1, fp) != 1) {
        fprintf(stderr, "fread: error");
        return -1;
    }
    tmpint = 1;
    if (((char *)&tmpint)[0] != 0) {
        char * chrdata = (char *)data;
        if (size == 2) {
            char swap = chrdata[0];
            chrdata[0] = chrdata[1];
            chrdata[1] = swap;
        } else if (size == 4) {
            char swap = chrdata[0];
            chrdata[0] = chrdata[3];
            chrdata[3] = swap;
            swap = chrdata[1];
            chrdata[1] = chrdata[2];
            chrdata[2] = swap;
        } else if (size != 1) {
            fprintf(stderr, "grabAndSwapValue: Can't swap sizes other than 1, 2 or 4");
            return -1;
        }
    }
    return 0;
}

BXHDocPtr
createDocFromIowaSigna5(const char *imagedir, const char *ofname)
{
    BXHDocPtr docp = NULL;
    struct stat statbuf;
    DIR * dirp = NULL;
    struct dirent * direntp = NULL;
    char ** infiles = NULL;
    int numinfiles = 0;
    struct signamrmap * mrmap = NULL;
    struct signamaps ** curmapptr = NULL;
    double * uservar39 = NULL;
    double * uservar43 = NULL;
    int * sliceorder = NULL;
    int filenum;

    size_t hintsizes[4] = { (size_t)-1, (size_t)-1, (size_t)-1, (size_t)-1 };
    double hintorigin[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintspacing[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintgap[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };

    for (curmapptr = &signamap3list[0]; *curmapptr; curmapptr++) {
	if ((*curmapptr)->alignsize == 2)
	    break;
    }
    if (*curmapptr == NULL) {
	fprintf(stderr, "iowa-signafive2bxh: don't have a map supporting alignsize of 2");
	goto FAIL;
    }
    mrmap = (*curmapptr)->mrmap;

    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "iowa-signafive2bxh: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((dirp = opendir(imagedir)) == NULL) {
	fprintf(stderr, "iowa-signafive2bxh: error opening directory ('%s')\n", imagedir);
	perror("opendir");
	goto FAIL;
    }

    while ((direntp = readdir(dirp)) != NULL) {
	char * filename = NULL;
	if (strcmp(direntp->d_name, ".") == 0 ||
	    strcmp(direntp->d_name, "..") == 0) {
	    continue;
	}
	if (strncmp(direntp->d_name, "I.", 2) != 0) {
	    continue;
	}
	filename = strdup(imagedir);
	filename = (char *)realloc(filename, strlen(imagedir) + 1 + strlen(direntp->d_name) + 1);
#ifdef WIN32
	strcat(filename, "\\");
#else
	strcat(filename, "/");
#endif
	strcat(filename, direntp->d_name);
	infiles = (char **)realloc(infiles, sizeof(char *) * (numinfiles + 1));
	infiles[numinfiles] = filename;
	numinfiles++;
    }

    uservar39 = (double *)malloc(sizeof(double)*numinfiles);
    uservar43 = (double *)malloc(sizeof(double)*numinfiles);
    for (filenum = 0; filenum < numinfiles; filenum++) {
	FILE * gefp = NULL;
	float tmpfloat;
	if ((gefp = fopen(infiles[filenum], "rb")) == NULL) {
	    fprintf(stderr, "Error opening signa file '%s'\n", infiles[filenum]);
	    perror("open");
	    goto FAIL;
	}
	grabAndSwapVal(gefp, IOFFSET_MRIMAGEDATA+mrmap->user39, 4, &tmpfloat);
	uservar39[filenum] = tmpfloat;
	grabAndSwapVal(gefp, IOFFSET_MRIMAGEDATA+mrmap->user43, 4, &tmpfloat);
	uservar43[filenum] = tmpfloat;
	fclose(gefp);
    }

    sliceorder = (int *)malloc(sizeof(int) * numinfiles);
    for (filenum = 0; filenum < numinfiles; filenum++) {
	sliceorder[filenum] = filenum;
    }

    /* mergesort the slices to scan order using filename */
    {
	int * tempsliceorder = NULL;
	int setsize = 0;
	tempsliceorder = (int *)malloc(sizeof(int) * numinfiles);
	setsize = 1;
	while (setsize < numinfiles) {
	    /* precondition: all subsets of size setsize are sorted */
	    /* postcondition: by merging neighboring subsets of size
	     *   setsize, all subsets of size setsize*2 will be sorted */
	    int tempind = 0;
	    int setstart = 0;
	    for (setstart = 0;
		 setstart < numinfiles;
		 setstart += (2 * setsize)) {
		int start1, start2, end1, end2;
		start1 = setstart;
		start2 = end1 = start1 + setsize;
		end2 = start2 + setsize;
		if (end2 > numinfiles) { end2 = numinfiles; }
		if (start2 >= numinfiles) { start2 = numinfiles; }
		if (end1 > numinfiles) { end1 = numinfiles; }
		while (start1 < end1 && start2 < end2) {
/* 		    if (uservar39[sliceorder[start1]] < */
/* 			uservar39[sliceorder[start2]]) { */
		    if (strcmp(infiles[sliceorder[start1]],
			       infiles[sliceorder[start2]]) < 0) {
			tempsliceorder[tempind++] = sliceorder[start1++];
		    } else {
			tempsliceorder[tempind++] = sliceorder[start2++];
		    }
		}
		while (start1 < end1) {
		    tempsliceorder[tempind++] = sliceorder[start1++];
		}
		while (start2 < end2) {
		    tempsliceorder[tempind++] = sliceorder[start2++];
		}
	    }
	    {
		int * swap = tempsliceorder;
		tempsliceorder = sliceorder;
		sliceorder = swap;
	    }
	    setsize *= 2;
	}
	free(tempsliceorder);
    }

    /* now go through the first few slices to find out where slice numbers
     * (in User Variable 43) wrap around */
    {
	int lastslicenum = -1;
	int firstslicenum = -1;
	int skipind = 0;
	int numslices = 0;
	int numtimepoints = 0;
	int sliceind = 0;
	for (sliceind = 0; sliceind < numinfiles; sliceind++) {
	    int curslicenum = uservar43[sliceorder[sliceind]];
	    if (firstslicenum == -1)
		firstslicenum = curslicenum;
	    if (lastslicenum >= curslicenum)
		break;
	    if (curslicenum - lastslicenum > 1) {
		skipind = sliceind;
		firstslicenum = curslicenum;
	    }
	    lastslicenum = curslicenum;
	}
	
	if (firstslicenum == -1 || lastslicenum == -1) {
	    fprintf(stderr, "iowa-signafive2bxh: Error finding first and last slice numbers\n");
	    goto FAIL;
	}

	if (skipind > 0) {
	    fprintf(stderr, "Skipping slices %s to %s\n", infiles[sliceorder[0]], infiles[sliceorder[skipind-1]]);
	    numinfiles -= skipind;
	    memmove(sliceorder, &sliceorder[skipind], numinfiles * sizeof(sliceorder[0]));
	}

	numslices = lastslicenum + 1 - firstslicenum;
	numtimepoints = numinfiles / numslices;
	
	if (numslices * numtimepoints != numinfiles) {
	    fprintf(stderr, "iowa-signafive2bxh: Total slices (%d) does not equal slices (%d) * timepoints (%d)\n", numinfiles, numslices, numtimepoints);
	    goto FAIL;
	}

	fprintf(stderr, "numslices=%d, numtimepoints=%d\n", numslices, numtimepoints);

	docp = createDocFromSigna5((const char **)infiles, numinfiles, ofname, hintsizes, hintorigin, hintspacing, hintgap, NULL, (size_t)-1, (size_t)-1);
    }

    goto EXIT;

FAIL:
    fprintf(stderr, "iowa-signafive2bxh: conversion failed.\n");
    if (docp) {
	bxh_freeDocument(docp);
    }
    docp = NULL;

EXIT:
    return docp;
}

int
createBXHFromIowaSigna5(const char *imagedir, const char *ofname)
{
    int result = 0;
    BXHDocPtr docp = NULL;
    struct stat statbuf;

    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "analyze2bxh: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromIowaSigna5(imagedir, ofname)) == NULL)
	goto FAIL;
    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

  FAIL:
    result = -1;

  EXIT:
    if (docp) {
	bxh_freeDocument(docp);
    }
    return result;
}

BXHDocPtr
bxh_iowasignafive2doc(const char * dirname)
{
    return createDocFromIowaSigna5(dirname, "dummy.bxh");
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.12  2005/11/10 21:11:08  gadde
 * Use file name rather than user39 to sort files.
 *
 * Revision 1.11  2005/09/19 16:31:58  gadde
 * Documentation and help message updates.
 *
 * Revision 1.10  2005/07/07 15:09:22  gadde
 * Updates for ver11 pfiles.
 *
 * Revision 1.9  2005/03/28 17:45:22  gadde
 * Version bump
 *
 * Revision 1.8  2004/03/19 15:13:32  gadde
 * Major changes in datarec support, include a new 'prepare' stage
 * that initiates filtering, permutation.
 * bxh_datarec_readData has been changed to bxh_datarec_readRawData
 * to reflect this.
 * Added some fMRI QA measures.
 *
 * Revision 1.7  2004/03/12 16:29:04  gadde
 * Minor updates
 *
 * Revision 1.6  2004/02/26 22:32:28  gadde
 * Windows fix?
 *
 * Revision 1.5  2004/02/26 22:25:40  gadde
 * Windows fix.
 *
 * Revision 1.4  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.3  2004/01/08 21:50:14  gadde
 * Fix comment.
 *
 * Revision 1.2  2004/01/08 17:47:18  gadde
 * Update docs
 *
 * Revision 1.1  2004/01/08 16:33:27  gadde
 * Add iowa GE support and other minor fixes
 *
 */
