static const char rcsid[] = "$Id: bxh_convert_pfile.c,v 1.1 2009-01-15 20:56:16 gadde Exp $";

/*
 * pfile2bxh.cc --
 * 
 * Creates a BXH file based on given Signa p-file.
 */

#include "bxh_utils.h"
#include "bxh_signa.h"
#include "bxh_signamaps.h"

#include <ctype.h>
#include <math.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#if defined(WIN32) && !defined(__MINGW32__)
#include <pstdint.h>
#include <float.h>
#ifndef NAN
static const uint32_t nan[1] = {0x7fffffff};
#define NAN (*(const float *) &nan[0])
#endif
#define MYNAN NAN
#ifndef isnan
#define isnan(x) _isnan(x)
#endif
#else /* #ifdef WIN32 */
/* this works on Linux */
#define MYNAN (0.0/0.0)
#endif

extern char * domutil_errorbuf;

static struct signamaps * signamap3list[3] = { &signamaps3align0, &signamaps3align2, NULL };
static struct signamaps * signamap11list[3] = { &signamaps11align0, &signamaps11align2, NULL };
static struct signamaps * signamap20list[3] = { &signamaps20align0, &signamaps20align2, NULL };
static struct signamaps * signamap27list[4] = { &signamaps27align0, &signamaps27align2, &signamaps27align4, NULL };

enum {
    PSDSPEC_NONE,
    PSDSPEC_SPIRAL,
    PSDSPEC_INVSPIRAL,
    PSDSPEC_EPIDUKE,
    PSDSPEC_NEWPF,
    PSDSPEC_SPEPI,
    PSDSPEC_SENSESPIRAL,
    PSDSPEC_EPIRT,
};

static struct {
    const char * name;
    unsigned int num;
} psdmap[] = {
    { "sss",		PSDSPEC_SPIRAL },
    { "epiduke",	PSDSPEC_EPIDUKE },
    { "invspiral",	PSDSPEC_INVSPIRAL },
    { "spiral_in",	PSDSPEC_INVSPIRAL },
    { "crm/ssscrm_v1",	PSDSPEC_SPIRAL },
    { "crm/ssscrm_v1b",	PSDSPEC_SPIRAL },
    { "crm/ssscrm_v1c",	PSDSPEC_SPIRAL },
    { "crm/ssscrm_v2",	PSDSPEC_SPIRAL },
    { "crm/ssscrm_file", PSDSPEC_SPIRAL },
    { "irsss2",		PSDSPEC_SPIRAL },
    { "sss_biac1",      PSDSPEC_SPIRAL },
    { "sss_biac2",      PSDSPEC_SPIRAL },
    { "invspiral_biac1", PSDSPEC_INVSPIRAL },
    { "invspiral_biac2", PSDSPEC_INVSPIRAL },
    { "newpf",		PSDSPEC_NEWPF },
    { "spepi",		PSDSPEC_SPEPI },
    { "spepi_11",	PSDSPEC_SPEPI },
    { "spepi2",		PSDSPEC_SPEPI },
    { "spstime1",       PSDSPEC_SPEPI },
    { "spstime2",       PSDSPEC_SPEPI },
    { "spstime3",       PSDSPEC_SPEPI },
    { "spio",           PSDSPEC_SPEPI },
    { "sensespiral",    PSDSPEC_SENSESPIRAL },
    { "epiRT",          PSDSPEC_EPIRT },
    { NULL,		PSDSPEC_NONE }
};

#ifndef TOPLEVELELEM
#define TOPLEVELELEM "bxh"
#endif

static int
grabAndSwapVal(FILE * fp, off_t off, size_t size, void * data, int valismsbfirst)
{
    unsigned int tmpint;
    if (fseek(fp, off, SEEK_SET) == -1) {
        perror("fseek");
        return -1;
    }
    if (fread(data, size, 1, fp) != 1) {
        fprintf(stderr, "fread: error");
        return -1;
    }
    tmpint = 1;
    if ((((char *)&tmpint)[0] != 0 && valismsbfirst) ||
	(((char *)&tmpint)[0] == 0 && !valismsbfirst)) {
        char * chrdata = (char *)data;
        if (size == 2) {
            char swap = chrdata[0];
            chrdata[0] = chrdata[1];
            chrdata[1] = swap;
        } else if (size == 4) {
            char swap = chrdata[0];
            chrdata[0] = chrdata[3];
            chrdata[3] = swap;
            swap = chrdata[1];
            chrdata[1] = chrdata[2];
            chrdata[2] = swap;
        } else if (size != 1) {
            fprintf(stderr, "grabAndSwapValue: Can't swap sizes other than 1, 2 or 4");
            return -1;
        }
    }
    return 0;
}

BXHDocPtr
createDocFromPFile(const char *pfname, const char **ifnames, int numfiles, const char *ofname, size_t *hintsize, double *hintorigin, double *hintspacing, double *hintgap, const char *forceorient, float forceversion, int msbfirst, const char * dimorder, const char * elemtype, int usemrorigin)
{
    BXHDocPtr docp = NULL;
    bxhrawdatarec * datarec = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    bxhdimension * dimx = NULL;
    bxhdimension * dimy = NULL;
    bxhdimension * dimz = NULL;
    bxhdimension * dimt = NULL;
    bxhdimension * dimr = NULL;
    bxhdimension * dima = NULL;
    bxhdimension * dims = NULL;
    BXHElementPtr tmpelem = NULL;
    BXHElementPtr tmpelem2 = NULL;
    char psdname[34] = "";
    char psdiname[34] = "";
    unsigned int psdspec = PSDSPEC_NONE;
    unsigned int xsize = (unsigned int)-1;
    unsigned int ysize = (unsigned int)-1;
    float dfov;
    unsigned int numslices = 0;
    unsigned int numtimepoints = 0;
    int hdrbigend = 1;
    short tmpshort;
    float tmpfloat;
    char numbuf[32];
    size_t totaldatasize = 0;
    int i;
    time_t tmptime;
    struct tm *tmp;
    FILE * pffp = NULL;
    char tmpbuf[64];
    int32_t tmpint;
    uint32_t tmpuint;

    float hdrvers = 0;

    struct signardbmap * rdbmap = NULL;
    struct signamrmap * mrmap = NULL;
    struct signasemap * semap = NULL;

    /* these are version 5 sizes, adjusted below for later versions */
    size_t HSIZE_RDB_HEADER_REC = 2048;
    size_t HSIZE_PER_PASS = 4096;
    size_t HSIZE_UNLOCK = 4096;
    size_t HSIZE_DATA_ACQ = 20480;
    size_t HSIZE_NEX = 2052;
    size_t HSIZE_NEX_ABORT = 2052;
    size_t HSIZE_TOOLSDATA = 2048;
    size_t HSIZE_PRESCAN = 0;
    size_t HSIZE_EXAMDATA = 1024;
    size_t HSIZE_SERIESDATA = 1020;
    size_t HSIZE_MRIMAGEDATA = 1044;

    off_t HOFFSET_RDB_HEADER_REC = 0;
    off_t HOFFSET_PER_PASS = 0;
    off_t HOFFSET_UNLOCK = 0;
    off_t HOFFSET_DATA_ACQ = 0;
    off_t HOFFSET_NEX = 0;
    off_t HOFFSET_NEX_ABORT = 0;
    off_t HOFFSET_TOOLSDATA = 0;
    off_t HOFFSET_PRESCAN = 0;
    off_t HOFFSET_EXAMDATA = 0;
    off_t HOFFSET_SERIESDATA = 0;
    off_t HOFFSET_MRIMAGEDATA = 0;
    off_t HOFFSET_USERDATA = 0;

    if ((pffp = fopen(pfname, "rb")) == NULL) {
	fprintf(stderr, "Error opening pfile '%s'\n", pfname);
	perror("open");
	goto FAIL;
    }

    /*** figure out version of pfile ***/
    {
	struct signamaps ** curmapptr = NULL;
	int alignsize = 0;

	grabAndSwapVal(pffp, 0, 4, &hdrvers, 0);
	hdrvers = (float)floor((double)hdrvers);
	if (hdrvers != (float)(int)hdrvers || hdrvers <= 0 || hdrvers >= 100) {
	    grabAndSwapVal(pffp, 0, 4, &hdrvers, 1);
	}
	if (forceversion != -1) {
	    hdrvers = forceversion;
	}
	if (hdrvers >= 7.0) {
	    HSIZE_EXAMDATA = 1040;
	    HSIZE_SERIESDATA = 1028;
	}
	if (hdrvers >= 8.0) {
	    HSIZE_DATA_ACQ = 40960;
	}
	if (hdrvers >= 9.0) {
	    HSIZE_SERIESDATA = 1536;
	    HSIZE_MRIMAGEDATA = 1536;
	    hdrbigend = 0;
	}
	if (hdrvers >= 11.0) {
	    HSIZE_DATA_ACQ = 45056;
	    HSIZE_SERIESDATA = 2048;
	}
	if (hdrvers >= 20.0) {
	    HSIZE_RDB_HEADER_REC = 4096;
	    HSIZE_PER_PASS = 16384;
	    HSIZE_UNLOCK = 16384;
	    HSIZE_DATA_ACQ = 98304;
	    HSIZE_NEX = 2052;
	    HSIZE_NEX_ABORT = 2052;
	    HSIZE_TOOLSDATA = 3548;
	    HSIZE_EXAMDATA = 1960;
	    HSIZE_SERIESDATA = 2560;
	    HSIZE_MRIMAGEDATA = 2448;
	}
	if (hdrvers >= 27.0) {
	    HSIZE_DATA_ACQ = 147456;
	    HSIZE_TOOLSDATA = 2048;
	    HSIZE_PRESCAN = 3188;
	    alignsize = 4;
	}

	if (hdrvers >= 27.0) {
	    curmapptr = &signamap27list[0];
	} else if (hdrvers >= 20.0) {
	    curmapptr = &signamap20list[0];
	} else if (hdrvers >= 11.0) {
	    curmapptr = &signamap11list[0];
	} else {
	    curmapptr = &signamap3list[0];
	}

	for (/*null*/; *curmapptr; curmapptr++) {
	    if ((*curmapptr)->alignsize == alignsize)
		break;
	}
	if (*curmapptr == NULL) {
	    fprintf(stderr, "pfile2bxh: don't have a map supporting alignsize of 0");
	    goto FAIL;
	}
	rdbmap = (*curmapptr)->rdbmap;
	mrmap = (*curmapptr)->mrmap;
	semap = (*curmapptr)->semap;
    }

    /* now that we have the correct sizes of each section, calculate offsets */
    HOFFSET_RDB_HEADER_REC = 0;
    HOFFSET_PER_PASS = ((HOFFSET_RDB_HEADER_REC) + (HSIZE_RDB_HEADER_REC));
    HOFFSET_UNLOCK = ((HOFFSET_PER_PASS) + (HSIZE_PER_PASS));
    HOFFSET_DATA_ACQ = ((HOFFSET_UNLOCK) + (HSIZE_UNLOCK));
    HOFFSET_NEX = ((HOFFSET_DATA_ACQ) + (HSIZE_DATA_ACQ));
    HOFFSET_NEX_ABORT = ((HOFFSET_NEX) + (HSIZE_NEX));
    HOFFSET_TOOLSDATA = ((HOFFSET_NEX_ABORT) + (HSIZE_NEX_ABORT));
    HOFFSET_PRESCAN = ((HOFFSET_TOOLSDATA) + (HSIZE_TOOLSDATA));
    HOFFSET_EXAMDATA = ((HOFFSET_PRESCAN) + (HSIZE_PRESCAN));
    HOFFSET_SERIESDATA = ((HOFFSET_EXAMDATA) + (HSIZE_EXAMDATA));
    HOFFSET_MRIMAGEDATA = ((HOFFSET_SERIESDATA) + (HSIZE_SERIESDATA));
    HOFFSET_USERDATA = ((HOFFSET_MRIMAGEDATA) + (HSIZE_MRIMAGEDATA));

    /*** grab PSD ***/
    {
	int psdmapind = 0;
	if (fseek(pffp, HOFFSET_MRIMAGEDATA + mrmap->psdname, SEEK_SET) == -1) {
	    perror("fseek");
	    goto FAIL;
	}
	if (fread(&psdname[0], sizeof(psdname), 1, pffp) != 1) {
	    perror("fread");
	    goto FAIL;
	}
	if (fseek(pffp, HOFFSET_MRIMAGEDATA + mrmap->psd_iname, SEEK_SET) == -1) {
	    perror("fseek");
	    goto FAIL;
	}
	if (fread(&psdiname[0], sizeof(psdiname), 1, pffp) != 1) {
	    perror("fread");
	    goto FAIL;
	}
	while (psdmap[psdmapind].name != NULL) {
	    if (strcmp(psdname, psdmap[psdmapind].name) == 0) {
		psdspec = psdmap[psdmapind].num;
		break;
	    }
	    psdmapind++;
	}
#ifdef BIAC
	if (psdspec == PSDSPEC_NONE) {
	    fprintf(stderr, "Warning: I don't recognize PSD %s -- some values may be incorrect.\n", &psdname[0]);
	}
#endif
    }

    if (dimorder == NULL) {
	dimorder = "x,y,z,t";
    }

    totaldatasize = 0;
    for (i = 0; i < numfiles; i++) {
	struct stat statbuf;
	if (stat(ifnames[i], &statbuf) == -1) {
	    fprintf(stderr, "Error opening file %s!\n", ifnames[i]);
	    perror("stat");
	    goto FAIL;
	}
	totaldatasize += statbuf.st_size;
    }

#ifdef BIAC
    if (psdspec == PSDSPEC_EPIDUKE) {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user2, 4, &tmpfloat, hdrbigend);
	xsize = (unsigned int)tmpfloat;
	ysize = (unsigned int)tmpfloat;
    } else if (psdspec != PSDSPEC_NONE && psdspec != PSDSPEC_EPIRT) {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user3, 4, &tmpfloat, hdrbigend);
	xsize = (unsigned int)tmpfloat;
	ysize = (unsigned int)tmpfloat;
    } else
#endif
    {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_im_size, 2, &tmpshort, hdrbigend);
	xsize = (unsigned int)tmpshort;
	ysize = (unsigned int)tmpshort;
    }

    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->dfov, 4, &tmpfloat, hdrbigend); /* dfov */
    dfov = tmpfloat;
    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_nslices, 2, &tmpshort, hdrbigend); /* nslices -> z */
    numslices = tmpshort;
    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->slquant, 2, &tmpshort, hdrbigend); /* nslices -> z */
    if (tmpshort != 0 && tmpshort < numslices) {
	numslices = tmpshort;
    }
#ifdef BIAC
    if (psdspec == PSDSPEC_EPIDUKE) {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user6, 4, &tmpfloat, hdrbigend);
	numtimepoints = (unsigned int)tmpfloat;
    } else if (psdspec != PSDSPEC_NONE && psdspec != PSDSPEC_EPIRT) {
	/* is this where all the other pulse sequences leave the number of volumes? */
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user1, 4, &tmpfloat, hdrbigend);
	numtimepoints = (unsigned int)tmpfloat;
    } else
#endif
    {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_nframes, 2, &tmpshort, hdrbigend);
	if (tmpshort != 0) {
	    numtimepoints = (unsigned int)tmpshort;
	} else {
	    numtimepoints = numfiles;
	}
    }

    /* use hints if provided */
    if (hintsize[0] != (size_t)-1) {
	xsize = hintsize[0];
    }
    if (hintsize[1] != (size_t)-1) {
	ysize = hintsize[1];
    }
    if (hintsize[2] != (size_t)-1) {
	numslices = hintsize[2];
    }
    if (hintsize[3] != (size_t)-1) {
	numtimepoints = hintsize[3];
    }

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    if (bxh_SIGNA_readPFileFromFile(docp, pfname, numslices, numtimepoints, dimorder, msbfirst, hdrvers) != 0)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;
    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;

#ifdef BIAC
    if (psdspec == PSDSPEC_INVSPIRAL || psdspec == PSDSPEC_SPIRAL) {
	if ((tmpelem = bxh_getChildElement(acquisitiondata, "frequencydirection")) != NULL) {
	    if ((tmpelem2 = bxh_removeChildElement(acquisitiondata, tmpelem)) == NULL)
		goto FAIL;
	    bxh_element_unref(tmpelem); tmpelem = NULL;
	    bxh_element_unref(tmpelem2); tmpelem2 = NULL;
	}
    }

    {    
	char tmpbuf[34];
	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->dim_X, 4, &tmpfloat, hdrbigend);
	sprintf(&numbuf[0], "%g", (double)tmpfloat);
	strcat(&tmpbuf[0], &numbuf[0]);
	if (psdspec != PSDSPEC_INVSPIRAL && psdspec != PSDSPEC_SPIRAL) {
	    strcat(&tmpbuf[0], " ");
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->dim_Y, 4, &tmpfloat, hdrbigend);
	    sprintf(&numbuf[0], "%g", (double)tmpfloat);
	    strcat(&tmpbuf[0], &numbuf[0]);
	}
	if (bxh_appendChildElement(acquisitiondata, "acquisitionmatrix", &tmpbuf[0]) != 0)
	    goto FAIL;
    }
#endif

    {
	float tlhc_1[3]; /* top-left corner (RAS), slice 1 */
	float blhc_1[3]; /* bottom-left corner (RAS), slice 1 */
	float trhc_1[3]; /* top-right corner (RAS), slice 1 */
	float tlhc_N[3]; /* top-left corner (RAS), slice N */
	float blhc_N[3]; /* bottom-left corner (RAS), slice N */
	float trhc_N[3]; /* top-right corner (RAS), slice N */
	float Xr, Xa, Xs; /* direction cosines */
	float Yr, Ya, Ys;
	float Zr, Za, Zs;
	float Xlen, Ylen, Zlen;
	int i;
	int slicestructsize = 40;
	
	if (hdrvers >= 11.0) {
	    slicestructsize = 44;
	}
	if (hdrvers >= 20.0) {
	    slicestructsize = 48;
	}
	if (hdrvers >= 27.0) {
	    slicestructsize = 72;
	}

	datarec = bxh_datarec_createFromElement(imagedata, NULL);

	for (i = 0; i < datarec->numdims; i++) {
	    if (strcmp(datarec->dimensions[i].type, "x") == 0) {
		dimx = &datarec->dimensions[i];
	    }
	    if (strcmp(datarec->dimensions[i].type, "y") == 0) {
		dimy = &datarec->dimensions[i];
	    }
	    if (strcmp(datarec->dimensions[i].type, "z") == 0) {
		dimz = &datarec->dimensions[i];
	    }
	    if (strcmp(datarec->dimensions[i].type, "t") == 0) {
		dimt = &datarec->dimensions[i];
		if (dimt->size == 0 || dimt->size == 1) {
		    memmove(dimt, dimt+1, (datarec->numdims - i - 1) * sizeof(bxhdimension));
		    datarec->numdims--;
		    i--;
		    dimt = NULL;
		}
	    }
	}

	if (dimx) {
	    if (xsize != (unsigned int)-1)
		dimx->size = xsize;
	    if (dimx->size != 0 && dimx->spacing == 0)
		dimx->spacing = (double)dfov / (double)dimx->size;
#ifdef BIAC
	    if (psdspec == PSDSPEC_EPIDUKE) {
		/* make sure to fix for epiduke */
		dimx->spacing = (double)dfov / (double)xsize;
	    }
#endif
	}
	if (dimy) {
	    if (ysize != (unsigned int)-1)
		dimy->size = ysize;
	    if (dimy->size != 0 && dimy->spacing == 0)
		dimy->spacing = (double)dfov / (double)dimy->size;
#ifdef BIAC
	    if (psdspec == PSDSPEC_EPIDUKE) {
		/* make sure to fix for epiduke */
		dimy->spacing = (double)dfov / (double)ysize;
	    }
#endif
	}
	if (dimz && dimz->size > 1) {
	    double calcslicespacing = 0;
	    double epsilon = 0.005;
	    float firsttlr, firsttla, firsttls;
	    float lasttlr, lasttla, lasttls;
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+4, 4, &firsttlr, hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+8, 4, &firsttla, hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+12, 4, &firsttls, hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+4, 4, &lasttlr, hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+8, 4, &lasttla, hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+12, 4, &lasttls, hdrbigend);
	    calcslicespacing =
		sqrt((lasttlr - firsttlr) * (lasttlr - firsttlr) +
		     (lasttla - firsttla) * (lasttla - firsttla) +
		     (lasttls - firsttls) * (lasttls - firsttls))
		/ (numslices - 1);
	    if (fabs(calcslicespacing - dimz->spacing) > epsilon) {
		fprintf(stderr, "Warning: reported slice spacing (%g) is different from calculated spacing (%g) by more than %g.  Using calculated number.\n", dimz->spacing, calcslicespacing, epsilon);
	    }
	    sprintf(&numbuf[0], "%g", dimz->spacing);
	    if (bxh_appendChildElement(acquisitiondata, "prescribedslicespacing", &numbuf[0]) != 0)
		goto FAIL;
	    dimz->spacing = calcslicespacing;
	}

	/* find per-slice info */
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+4,  4, &tlhc_1[0], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+8,  4, &tlhc_1[1], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+12, 4, &tlhc_1[2], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+16, 4, &blhc_1[0], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+20, 4, &blhc_1[1], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+24, 4, &blhc_1[2], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+28, 4, &trhc_1[0], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+32, 4, &trhc_1[1], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+36, 4, &trhc_1[2], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+4,  4, &tlhc_N[0], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+8,  4, &tlhc_N[1], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+12, 4, &tlhc_N[2], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+16, 4, &blhc_N[0], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+20, 4, &blhc_N[1], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+24, 4, &blhc_N[2], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+28, 4, &trhc_N[0], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+32, 4, &trhc_N[1], hdrbigend);
	grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*(numslices-1))+36, 4, &trhc_N[2], hdrbigend);

	if (hdrvers >= 11.0) {
	    float brhc_1[3]; /* bottom-right corner (RAS), slice 1 */
	    float brhc_N[3]; /* bottom-right corner (RAS), slice N */
	    float swappt[3];
	    int transpose, rotate;

	    /* assume transpose and rotate values don't change per slice */
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+40, 2, &tmpshort, hdrbigend);
	    transpose = tmpshort;
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*0)+42, 2, &tmpshort, hdrbigend);
	    rotate = tmpshort;
	    /* fprintf(stderr, "rotate=%d, transpose=%d\n", rotate, transpose); */

	    /* calculate fourth corner */
	    brhc_1[0] = blhc_1[0] + trhc_1[0] - tlhc_1[0];
	    brhc_1[1] = blhc_1[1] + trhc_1[1] - tlhc_1[1];
	    brhc_1[2] = blhc_1[2] + trhc_1[2] - tlhc_1[2];
	    brhc_N[0] = blhc_N[0] + trhc_N[0] - tlhc_N[0];
	    brhc_N[1] = blhc_N[1] + trhc_N[1] - tlhc_N[1];
	    brhc_N[2] = blhc_N[2] + trhc_N[2] - tlhc_N[2];

	    /* Not sure if this is right -- XXX STAB IN THE DARK XXX */
	    if (transpose == 3) {
		memcpy(swappt, trhc_1, sizeof(float)*3);
		memcpy(trhc_1, blhc_1, sizeof(float)*3);
		memcpy(blhc_1, swappt, sizeof(float)*3);
		memcpy(swappt, trhc_N, sizeof(float)*3);
		memcpy(trhc_N, blhc_N, sizeof(float)*3);
		memcpy(blhc_N, swappt, sizeof(float)*3);
	    }

	    switch (rotate) {
	    case 0: /* do nothing */
		break;
	    case 1: /* 90deg to the right */
		memcpy(swappt, tlhc_1, sizeof(float)*3);
		memcpy(tlhc_1, trhc_1, sizeof(float)*3);
		memcpy(trhc_1, brhc_1, sizeof(float)*3);
		memcpy(brhc_1, blhc_1, sizeof(float)*3);
		memcpy(blhc_1, swappt, sizeof(float)*3);
		memcpy(swappt, tlhc_N, sizeof(float)*3);
		memcpy(tlhc_N, trhc_N, sizeof(float)*3);
		memcpy(trhc_N, brhc_N, sizeof(float)*3);
		memcpy(brhc_N, blhc_N, sizeof(float)*3);
		memcpy(blhc_N, swappt, sizeof(float)*3);
		break;
	    case 2: /* 180deg to the right */
		memcpy(swappt, tlhc_1, sizeof(float)*3);
		memcpy(tlhc_1, brhc_1, sizeof(float)*3);
		memcpy(brhc_1, swappt, sizeof(float)*3);
		memcpy(swappt, trhc_1, sizeof(float)*3);
		memcpy(trhc_1, blhc_1, sizeof(float)*3);
		memcpy(blhc_1, swappt, sizeof(float)*3);
		memcpy(swappt, tlhc_N, sizeof(float)*3);
		memcpy(tlhc_N, brhc_N, sizeof(float)*3);
		memcpy(brhc_N, swappt, sizeof(float)*3);
		memcpy(swappt, trhc_N, sizeof(float)*3);
		memcpy(trhc_N, blhc_N, sizeof(float)*3);
		memcpy(blhc_N, swappt, sizeof(float)*3);
		break;
	    case 3: /* 270deg to the right */
		memcpy(swappt, tlhc_1, sizeof(float)*3);
		memcpy(tlhc_1, blhc_1, sizeof(float)*3);
		memcpy(blhc_1, brhc_1, sizeof(float)*3);
		memcpy(brhc_1, trhc_1, sizeof(float)*3);
		memcpy(trhc_1, swappt, sizeof(float)*3);
		memcpy(swappt, tlhc_N, sizeof(float)*3);
		memcpy(tlhc_N, blhc_N, sizeof(float)*3);
		memcpy(blhc_N, brhc_N, sizeof(float)*3);
		memcpy(brhc_N, trhc_N, sizeof(float)*3);
		memcpy(trhc_N, swappt, sizeof(float)*3);
		break;
	    }

	    /* Not sure if this is right -- XXX STAB IN THE DARK XXX */
	    if (transpose == -3) {
		memcpy(swappt, trhc_1, sizeof(float)*3);
		memcpy(trhc_1, blhc_1, sizeof(float)*3);
		memcpy(blhc_1, swappt, sizeof(float)*3);
		memcpy(swappt, trhc_N, sizeof(float)*3);
		memcpy(trhc_N, blhc_N, sizeof(float)*3);
		memcpy(blhc_N, swappt, sizeof(float)*3);
	    }
#ifdef BIAC
	} if (psdspec == PSDSPEC_EPIDUKE && hdrvers <= 7.0) {
	    /* we don't believe the slice table on epiduke on early P-file versions */
	    float mr_tlhc_1[3]; /* top-left corner (RAS), slice 1 from MR struct */
	    float mr_trhc_1[3]; /* top-right corner (RAS), slice 1 from MR struct */
	    float mr_brhc_1[3]; /* bottom-right corner (RAS), slice 1 from MR struct */
	    float mr_blhc_1[3]; /* bottom-left corner (RAS), slice 1 from MR struct */
	    float zextent[3]; /* RAS vector from first to last slice using slice table */
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->tlhc_R, 4, &mr_tlhc_1[0], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->tlhc_A, 4, &mr_tlhc_1[1], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->tlhc_S, 4, &mr_tlhc_1[2], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->trhc_R, 4, &mr_trhc_1[0], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->trhc_A, 4, &mr_trhc_1[1], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->trhc_S, 4, &mr_trhc_1[2], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->brhc_R, 4, &mr_brhc_1[0], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->brhc_A, 4, &mr_brhc_1[1], hdrbigend);
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->brhc_S, 4, &mr_brhc_1[2], hdrbigend);
	    /* calculate fourth corner */
	    mr_blhc_1[0] = mr_brhc_1[0] + mr_tlhc_1[0] - mr_trhc_1[0];
	    mr_blhc_1[1] = mr_brhc_1[1] + mr_tlhc_1[1] - mr_trhc_1[1];
	    mr_blhc_1[2] = mr_brhc_1[2] + mr_tlhc_1[2] - mr_trhc_1[2];

	    /* OK, now we know what the MR struct thinks of the corners,
	     * and we know what the slice table thinks of the corners.
	     * So, use the MR struct corners for the first slice, and
	     * use the offset between first and last slice in the slice
	     * table to adjust the MR struct first slice corners to get
	     * last slice corners.
	     */
	    zextent[0] = tlhc_N[0] - tlhc_1[0];
	    zextent[1] = tlhc_N[1] - tlhc_1[1];
	    zextent[2] = tlhc_N[2] - tlhc_1[2];

	    tlhc_1[0] = mr_tlhc_1[0];
	    tlhc_1[1] = mr_tlhc_1[1];
	    tlhc_1[2] = mr_tlhc_1[2];
	    trhc_1[0] = mr_trhc_1[0];
	    trhc_1[1] = mr_trhc_1[1];
	    trhc_1[2] = mr_trhc_1[2];
	    blhc_1[0] = mr_blhc_1[0];
	    blhc_1[1] = mr_blhc_1[1];
	    blhc_1[2] = mr_blhc_1[2];
	    tlhc_N[0] = mr_tlhc_1[0] + zextent[0];
	    tlhc_N[1] = mr_tlhc_1[1] + zextent[1];
	    tlhc_N[2] = mr_tlhc_1[2] + zextent[2];
	    trhc_N[0] = mr_trhc_1[0] + zextent[0];
	    trhc_N[1] = mr_trhc_1[1] + zextent[1];
	    trhc_N[2] = mr_trhc_1[2] + zextent[2];
	    blhc_N[0] = mr_blhc_1[0] + zextent[0];
	    blhc_N[1] = mr_blhc_1[1] + zextent[1];
	    blhc_N[2] = mr_blhc_1[2] + zextent[2];
#endif
	}

	/* calculate direction vectors */
	Xr = trhc_1[0] - tlhc_1[0];
	Xa = trhc_1[1] - tlhc_1[1];
	Xs = trhc_1[2] - tlhc_1[2];
	Yr = blhc_1[0] - tlhc_1[0];
	Ya = blhc_1[1] - tlhc_1[1];
	Ys = blhc_1[2] - tlhc_1[2];
	Zr = tlhc_N[0] - tlhc_1[0];
	Za = tlhc_N[1] - tlhc_1[1];
	Zs = tlhc_N[2] - tlhc_1[2];
	/* normalize vectors */
	Xlen = (float)sqrt(Xr*Xr + Xa*Xa + Xs*Xs);
	Ylen = (float)sqrt(Yr*Yr + Ya*Ya + Ys*Ys);
	Zlen = (float)sqrt(Zr*Zr + Za*Za + Zs*Zs);
	if (Xlen > 0) {
	    Xr /= Xlen; Xa /= Xlen; Xs /= Xlen;
	}
	if (Ylen > 0) {
	    Yr /= Ylen; Ya /= Ylen; Ys /= Ylen;
	}
	if (Zlen > 0) {
	    Zr /= Zlen; Za /= Zlen; Zs /= Zlen;
	}
	/* determine X/Y/Z to R/A/S mapping */
	if (fabs(Xr) > fabs(Yr) && fabs(Xr) > fabs(Zr)) {
	    dimr = dimx;
	    if (fabs(Ya) > fabs(Xa) && fabs(Ya) > fabs(Za)) {
		dima = dimy; dims = dimz;
	    } else {
		dima = dimz; dims = dimy;
	    }
	} else if (fabs(Yr) > fabs(Xr) && fabs(Yr) > fabs(Zr)) {
	    dimr = dimy;
	    if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
		dima = dimx; dims = dimz;
	    } else {
		dima = dimz; dims = dimx;
	    }
	} else {
	    dimr = dimz;
	    if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
		dima = dimx; dims = dimy;
	    } else {
		dima = dimy; dims = dimx;
	    }
	}
	if (dimx) {
	    if (dimx->direction == NULL)
		dimx->direction = (double *)malloc(sizeof(double) * 3);
	    if (Xlen == 0) {
		/* just make a fake normal direction vector */
		dimx->direction[0] = Ya*Zs - Ys*Za;
		dimx->direction[1] = Ys*Zr - Yr*Zs;
		dimx->direction[2] = Yr*Za - Ya*Zr;
	    } else {
		dimx->direction[0] = Xr;
		dimx->direction[1] = Xa;
		dimx->direction[2] = Xs;
	    }
	}
	if (dimy) {
	    if (dimy->direction == NULL)
		dimy->direction = (double *)malloc(sizeof(double) * 3);
	    if (Ylen == 0) {
		/* just make a fake normal direction vector */
		dimy->direction[0] = Xa*Zs - Xs*Za;
		dimy->direction[1] = Xs*Zr - Xr*Zs;
		dimy->direction[2] = Xr*Za - Xa*Zr;
	    } else {
		dimy->direction[0] = Yr;
		dimy->direction[1] = Ya;
		dimy->direction[2] = Ys;
	    }
	}
	if (dimz) {
	    if (dimz->direction == NULL)
		dimz->direction = (double *)malloc(sizeof(double) * 3);
	    if (Zlen == 0) {
	      /* just make a fake normal direction vector */
		dimz->direction[0] = Ya*Xs - Ys*Xa;
		dimz->direction[1] = Ys*Xr - Yr*Xs;
		dimz->direction[2] = Yr*Xa - Ya*Xr;
	    } else {
		dimz->direction[0] = Zr;
		dimz->direction[1] = Za;
		dimz->direction[2] = Zs;
	    }
	}
	if (usemrorigin) {
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->tlhc_R, 4, &tmpfloat, hdrbigend);
	    if (dimr) dimr->origin = tmpfloat;
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->tlhc_A, 4, &tmpfloat, hdrbigend);
	    if (dima) dima->origin = tmpfloat;
	    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->tlhc_S, 4, &tmpfloat, hdrbigend);
	    if (dims) dims->origin = tmpfloat;
	} else {
	    if (dimr) dimr->origin = tlhc_1[0];
	    if (dima) dima->origin = tlhc_1[1];
	    if (dims) dims->origin = tlhc_1[2];
	}
	/* fix origin to point to center of voxels
	 * (x and y dimensions use edge of bounding box in GE) */
	dimr->origin +=
		(dimx->spacing / 2.0) * dimx->direction[0] +
		(dimy->spacing / 2.0) * dimy->direction[0];
	dima->origin +=
		(dimx->spacing / 2.0) * dimx->direction[1] +
		(dimy->spacing / 2.0) * dimy->direction[1];
	dims->origin +=
		(dimx->spacing / 2.0) * dimx->direction[2] +
		(dimy->spacing / 2.0) * dimy->direction[2];

	bxh_datarec_frags_free(datarec);
	for (i = 0; i < numfiles; i++) {
	    struct stat statbuf;
	    if (stat(ifnames[i], &statbuf) == -1) {
		perror("stat");
		goto FAIL;
	    }
	    bxh_datarec_addfrag(datarec, ifnames[i], 0, statbuf.st_size, ofname, 1);
	}

	{
	    char * hintnames[4] = { "x", "y", "z", "t" };
	    int dimnum;
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		bxhdimension * dimp = &datarec->dimensions[dimnum];
		int hintnum;
		for (hintnum = 0; hintnum < 4; hintnum++) {
		    if (strcmp(dimp->type, hintnames[hintnum]) == 0)
			break;
		}
		if (hintnum == 4) continue;
		if (hintorigin[hintnum] != HUGE_VAL)
		    dimp->origin = hintorigin[hintnum];
		if (hintspacing[hintnum] != HUGE_VAL)
		    dimp->spacing = hintspacing[hintnum];
		if (hintgap[hintnum] != HUGE_VAL)
		    dimp->gap = hintgap[hintnum];
		if (forceorient && hintnum < 3) {
		    if (dimp->direction == NULL)
			dimp->direction = (double *)malloc(sizeof(double)*3);
		    memset(dimp->direction, '\0', sizeof(double)*3);
		    if (tolower(forceorient[hintnum]) == 'r') {
			dimp->direction[0] = 1;
		    } else if (tolower(forceorient[hintnum]) == 'a') {
			dimp->direction[1] = 1;
		    } else if (tolower(forceorient[hintnum]) == 's') {
			dimp->direction[2] = 1;
		    } else if (tolower(forceorient[hintnum]) == 'l') {
			dimp->direction[0] = -1;
		    } else if (tolower(forceorient[hintnum]) == 'p') {
			dimp->direction[1] = -1;
		    } else if (tolower(forceorient[hintnum]) == 'i') {
			dimp->direction[2] = -1;
		    }
		}
	    }
	}

	if (datarec->elemtype) {
	    free(datarec->elemtype);
	}
	datarec->elemtype = strdup(elemtype);

    }

#ifdef BIAC
    if (psdspec == PSDSPEC_EPIDUKE) {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user15, 4, &tmpfloat, hdrbigend);
	sprintf(&numbuf[0], "%g", (double)tmpfloat);
	if (bxh_appendChildElement(acquisitiondata, "disdaqs", &numbuf[0]) != 0)
	    goto FAIL;
    } else if (psdspec == PSDSPEC_SENSESPIRAL) {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user20, 4, &tmpfloat, hdrbigend);
	sprintf(&numbuf[0], "%g", (double)tmpfloat);
	if (bxh_appendChildElement(acquisitiondata, "disdaqs", &numbuf[0]) != 0)
	    goto FAIL;
    } else if (psdspec != PSDSPEC_NONE && psdspec != PSDSPEC_EPIRT) {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user19, 4, &tmpfloat, hdrbigend);
	sprintf(&numbuf[0], "%g", (double)tmpfloat);
	if (bxh_appendChildElement(acquisitiondata, "disdaqs", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user16, 4, &tmpfloat, hdrbigend);
	if (tmpfloat == 1) {
	    if (bxh_appendChildElement(acquisitiondata, "diffusion", "isotropic") != 0)
		goto FAIL;
	    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user17, 4, &tmpfloat, hdrbigend);
	    sprintf(&numbuf[0], "%g", (double)tmpfloat);
	    if (bxh_appendChildElement(acquisitiondata, "diffusionweighting", &numbuf[0]) != 0)
		goto FAIL;
	} else if (tmpfloat == 2) {
	    if (bxh_appendChildElement(acquisitiondata, "diffusion", "anisotropic") != 0)
		goto FAIL;
	    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user18, 4, &tmpfloat, hdrbigend);
	    sprintf(&numbuf[0], "%g", (double)tmpfloat);
	    if (bxh_appendChildElement(acquisitiondata, "diffusionweighting", &numbuf[0]) != 0)
		goto FAIL;
	}

	if ((tmpelem = bxh_getChildElement(acquisitiondata, "fieldofview")) != NULL) {
	    if ((tmpelem2 = bxh_removeChildElement(acquisitiondata, tmpelem)) == NULL)
		goto FAIL;
	    bxh_element_unref(tmpelem); tmpelem = NULL;
	    bxh_element_unref(tmpelem2); tmpelem2 = NULL;
	    tmpelem = NULL;
	}
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_user0, 4, &tmpfloat, hdrbigend);
	tmpfloat *= 10.0; /* cm => mm */
	sprintf(&numbuf[0], "%g %g", (double)tmpfloat, (double)tmpfloat);
	if (bxh_appendChildElement(acquisitiondata, "fieldofview", &numbuf[0]) != 0)
	    goto FAIL;
    }
#endif

    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_ps_mps_freq, 4, &tmpuint, hdrbigend);
    sprintf(&numbuf[0], "%u", (unsigned int)tmpuint);
    if (bxh_appendChildElement(acquisitiondata, "manualprescancenterfrequency", &numbuf[0]) != 0)
	goto FAIL;
    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_ps_mps_tg, 4, &tmpint, hdrbigend);
    sprintf(&numbuf[0], "%d", (int)tmpint);
    if (bxh_appendChildElement(acquisitiondata, "manualprescantransmitgain", &numbuf[0]) != 0)
	goto FAIL;

    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_ps_aps_freq, 4, &tmpuint, hdrbigend);
    sprintf(&numbuf[0], "%u", (unsigned int)tmpuint);
    if (bxh_appendChildElement(acquisitiondata, "autoprescancenterfrequency", &numbuf[0]) != 0)
	goto FAIL;
    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_ps_aps_tg, 4, &tmpint, hdrbigend);
    sprintf(&numbuf[0], "%d", (int)tmpint);
    if (bxh_appendChildElement(acquisitiondata, "autoprescantransmitgain", &numbuf[0]) != 0)
	goto FAIL;

    grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_dacq_ctrl, 2, &tmpshort, hdrbigend);
    sprintf(&numbuf[0], "%u", (unsigned int)tmpshort);
    if (bxh_appendChildElement(acquisitiondata, "dacq_ctrl", &numbuf[0]) != 0)
	goto FAIL;

    if (hdrvers >= 27.0) {
	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_mb_slice_order_sign, 4, &tmpint, hdrbigend);
	sprintf(&numbuf[0], "%d", (unsigned int)tmpint);
	if (bxh_appendChildElement(acquisitiondata, "mb_slice_order_sign", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_mb_factor, 4, &tmpuint, hdrbigend);
	sprintf(&numbuf[0], "%u", (unsigned int)tmpuint);
	if (bxh_appendChildElement(acquisitiondata, "mb_factor", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_mb_slice_fov_shift_factor, 4, &tmpuint, hdrbigend);
	sprintf(&numbuf[0], "%u", (unsigned int)tmpuint);
	if (bxh_appendChildElement(acquisitiondata, "mb_slice_fov_shift_factor", &numbuf[0]) != 0)
	    goto FAIL;
    }


    {
	char numbuf[128];
	static char sliceorder[1024];
	int * sliceorderlist = NULL; /* stores slice indices starting at 1 */
	char ** datapoints = NULL;
	int slicenum;
	int sliceorderind;
	char * curpos = &sliceorder[0];
	int slicestructsize = 40;
	int skipsliceorder = 0;

	if (hdrvers >= 11.0) {
	    slicestructsize = 44;
	}
	if (hdrvers >= 20.0) {
	    slicestructsize = 48;
	}
	if (hdrvers >= 27.0) {
	    slicestructsize = 72;
	}
	sliceorderlist = (int *)malloc(sizeof(int)*numslices);
	datapoints = (char **)malloc(sizeof(char *)*numslices);
	sliceorder[0] = '\0';
	for (slicenum = 0; slicenum < (int)numslices; slicenum++) {
	    int passnumber = 0;
	    int sliceinpass = 0;
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*slicenum),  2, &tmpshort, hdrbigend);
	    passnumber = tmpshort;
	    if (passnumber > 0) {
		int slicenum2;
		fprintf(stderr, "pfile2bxh: more than one pass listed -- skipping slice order\n");
		for (slicenum2 = 0; slicenum2 < slicenum; slicenum2++) {
		    free(datapoints[slicenum2]);
		}
		free(datapoints);
		skipsliceorder = 1;
		break;
	    }
	    grabAndSwapVal(pffp, HOFFSET_DATA_ACQ+(slicestructsize*slicenum)+2,  2, &tmpshort, hdrbigend);
	    sliceinpass = tmpshort;
	    /* slice in pass is indexed starting at 1 */
	    if (sliceinpass > numslices) {
		int slicenum2;
		fprintf(stderr, "pfile2bxh: Got a slice_in_pass number (%d) greater than the number of slices (%d)!\n", (int)sliceinpass, (int)numslices);
		for (slicenum2 = 0; slicenum2 < slicenum; slicenum2++) {
		    free(datapoints[slicenum2]);
		}
		free(datapoints);
		free(sliceorderlist);
		goto FAIL;
	    }
	    sliceorderlist[sliceinpass-1] = slicenum + 1;
	    sprintf(&numbuf[0], "%d", sliceinpass);
	    datapoints[slicenum] = strdup(&numbuf[0]);
	}
	if (!skipsliceorder) {
	    unsigned int mb_factor = 0;
	    int mb_slice_order_sign = 0;
	    if (hdrvers >= 27.0) {
		grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_mb_factor, 4, &mb_factor, hdrbigend);
		grabAndSwapVal(pffp, HOFFSET_RDB_HEADER_REC+rdbmap->rdb_hdr_mb_slice_order_sign, 4, &mb_slice_order_sign, hdrbigend);
		if (mb_factor > 1) {
		    int slices_per_band = (numslices + mb_factor - 1) / mb_factor;
		    int sliceind;
		    int slice_offset = 0;
		    // GE Rev 27 apparently always acquires an odd
		    // number of slices per band, even if
		    // numslices / numbands is even.  It seems to
		    // acquire extra slices beyond the end of the
		    // prescription and throws them away.
		    // Good news is that slice order in slice table
		    // at end of pfile header seems to have the
		    // correct slice acquisition order for the first
		    // slices_per_band slices (or slices_per_band + 1
		    // if slices_per_band is even).  So we'll just
		    // repeat the first slices_per_band (or
		    // slices_per_band + 1) sliceinpass values until
		    // we reach the right number of slices.
		    if (slices_per_band % 2 == 0) {
			slices_per_band++;
		    }
		    char ** newdatapoints = (char **)malloc(sizeof(char *)*numslices);
		    for (sliceind = 0; sliceind < numslices; sliceind++) {
			int sliceorderind = (slice_offset + sliceind) % slices_per_band;
			newdatapoints[sliceind] = strdup(datapoints[sliceorderind]);
		    }
		    for (sliceind = 0; sliceind < numslices; sliceind++) {
			free(datapoints[sliceind]); datapoints[sliceind] = NULL;
		    }
		    free(datapoints);
		    datapoints = newdatapoints;
		    newdatapoints = NULL;
		}
	    }
	    dimz->dpstructs = (bxhdatapoints *)realloc(dimz->dpstructs, sizeof(bxhdatapoints) * (dimz->numdpstructs + 1));
	    {
		bxhdatapoints * dpstructp = &dimz->dpstructs[dimz->numdpstructs];
		dpstructp->label = strdup("acquisitiontimeindex");
		dpstructp->values = datapoints;
		dpstructp->numvalues = numslices;
	    }
	    dimz->numdpstructs++;
	    /* NOTE: slice numbers are indexed with first slice==1 (not zero!) */
	    for (sliceorderind = 0; sliceorderind < (int)numslices; sliceorderind++) {
		if (curpos != &sliceorder[0])
		    curpos += sprintf(curpos, ",");
		curpos += sprintf(curpos, "%d", sliceorderlist[sliceorderind]);
	    }
	    if (sliceorder[0] != '\0') {
		if (mb_factor > 1) {
		    fprintf(stderr, "Won't write sliceorder element for multi-band acquisition (acquisitiontimeindex is still written)\n");
		} else if (bxh_appendChildElement(acquisitiondata, "sliceorder", &sliceorder[0]) != 0) {
		    fprintf(stderr, "pfile2bxh: Couldn't append sliceorder element!\n");
		    free(sliceorderlist);
		    goto FAIL;
		}
	    }
	}
	free(sliceorderlist);
    }

    if (bxh_appendChildElement(acquisitiondata, "psdname", &psdname[0]) != 0)
	goto FAIL;
    if (bxh_appendChildElement(acquisitiondata, "psdinternalname", &psdiname[0]) != 0)
	goto FAIL;
    grabAndSwapVal(pffp, HOFFSET_MRIMAGEDATA+mrmap->psd_datetime, 4, &tmpint, hdrbigend); /* psd_datetime */
    tmptime = tmpint;
    tmp = gmtime(&tmptime);
    sprintf(&tmpbuf[0], "%d-%02d-%02d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday);
    if (bxh_appendChildElement(acquisitiondata, "psddate", &tmpbuf[0]) != 0)
	goto FAIL;
    sprintf(&tmpbuf[0], "%02d:%02d:%02d", tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
    if (bxh_appendChildElement(acquisitiondata, "psdtime", &tmpbuf[0]) != 0)
	goto FAIL;

    bxh_datarec_writeToElement(imagedata, datarec);
    bxh_datarec_free(datarec); datarec = NULL;

    goto EXIT;

FAIL:
    fprintf(stderr, "pfile2bxh: conversion failed.\n");
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

EXIT:
    if (pffp)
	fclose(pffp);
    if (imagedata)
	bxh_element_unref(imagedata);
    if (acquisitiondata)
	bxh_element_unref(acquisitiondata);

    return docp;
}

int
createBXHFromPFile(const char *pfname, const char **ifnames, int numfiles, const char *ofname, size_t *hintsize, double *hintorigin, double *hintspacing, double *hintgap, const char *forceorient, float forceversion, int msbfirst, const char * dimorder, const char * elemtype, int usemrorigin)
{
    int result = 0;
    BXHDocPtr docp = NULL;
    struct stat statbuf;

    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "createBXHFromPfile: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromPFile(pfname, ifnames, numfiles, ofname, hintsize, hintorigin, hintspacing, hintgap, forceorient, forceversion, msbfirst, dimorder, elemtype, usemrorigin)) == NULL)
	goto FAIL;

    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    result = -1;

EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

    return result;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.84  2008/04/04 14:08:26  gadde
 * Use nframes for number of timepoints if no sequence-specific hints.
 *
 * Revision 1.83  2008/01/28 17:01:40  gadde
 * Fix memory leak.
 *
 * Revision 1.82  2008/01/22 18:02:02  gadde
 * Update struct sizes for < 7.0 versions of Pfile (but don't know
 * how the structs themselves changed).
 *
 * Revision 1.81  2007/11/08 16:11:20  gadde
 * Add sensespiral CVs.
 *
 * Revision 1.80  2007/09/17 18:58:23  gadde
 * Add "transpose" code.
 *
 * Revision 1.79  2007/01/16 19:34:42  gadde
 * Memory fixes.
 *
 * Revision 1.78  2006/11/15 22:09:51  gadde
 * Put in some code to deal with strange coordinates in slice table
 * on 4T epiduke data.
 *
 * Revision 1.77  2006/11/01 19:44:04  gadde
 * Add --usemrorigin option.
 *
 * Revision 1.76  2006/08/14 17:28:58  gadde
 * 64-bit fix (time_t is not 32-bit on 64-bit machines)
 *
 * Revision 1.75  2006/07/31 17:56:14  gadde
 * Fix X/Y spacing for epiduke.
 *
 * Revision 1.74  2006/07/28 14:08:51  gadde
 * Add more spepi-like sequences.
 *
 * Revision 1.73  2006/07/11 14:08:55  gadde
 * Fix to use correct user CVs for EPIDUKE.
 *
 * Revision 1.72  2006/06/06 18:56:23  gadde
 * Comment out diagnostic message.
 *
 * Revision 1.71  2006/03/24 14:55:37  gadde
 * Use the rotate field in the per-slice table.
 *
 * Revision 1.70  2006/02/08 19:05:56  gadde
 * Go back to using tlhc etc. for BIAC data until we can interpret 'rotate' field
 *
 * Revision 1.69  2005/12/06 19:12:02  gadde
 * Add some more sequence names.
 *
 * Revision 1.68  2005/09/19 16:31:59  gadde
 * Documentation and help message updates.
 *
 * Revision 1.67  2005/09/14 14:49:32  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.66  2005/08/03 16:27:52  gadde
 * Surround some BIAC-specific stuff with if (psdspec != PSDSPEC_NONE)
 * so it can be used outside BIAC too.
 *
 * Revision 1.65  2005/07/12 21:18:10  gadde
 * Update slice structure for ver11 pfiles.
 *
 * Revision 1.64  2005/07/12 18:37:08  gadde
 * Add "spepi" to recognized PSDs.
 *
 * Revision 1.63  2005/07/07 19:26:21  gadde
 * Remove t dimension if there is only one timepoint.
 *
 * Revision 1.62  2005/07/07 15:05:03  gadde
 * Updates for ver11 pfiles.
 *
 * Revision 1.61  2005/05/10 19:49:11  gadde
 * Add slice order info from user CVs.
 *
 * Revision 1.60  2005/03/08 18:09:58  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.59  2005/02/02 17:34:05  gadde
 * Make section size calculation a little more clear.
 *
 * Revision 1.58  2005/02/01 21:45:27  gadde
 * Add support for little-endian (i.e. vers>9.0) image headers.
 *
 * Revision 1.57  2005/01/11 21:29:35  gadde
 * Add --forceversion option.
 *
 * Revision 1.56  2004/11/02 18:24:35  gadde
 * Get location/orientation info from per-slice table rather than
 * from MR corner fields (MR_tlhc_R, MR_tlhc_A, ...) which are not
 * necessarily reliable.
 *
 * Revision 1.55  2004/07/07 17:00:49  gadde
 * Memory fixes.
 *
 * Revision 1.54  2004/06/18 15:21:51  gadde
 * Standardize frag creation (redux)
 *
 * Revision 1.53  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.52  2004/05/20 18:03:49  gadde
 * Don't fail on unrecognized PSD.
 *
 * Revision 1.51  2004/05/20 17:21:38  gadde
 * Fix BIAC #ifdefs.
 *
 * Revision 1.50  2004/05/20 14:44:47  gadde
 * BIAC: Don't fail on unrecognized PSDs.
 *
 * Revision 1.49  2004/04/29 20:40:11  gadde
 * Add some more pulse sequences.
 *
 * Revision 1.48  2004/02/24 18:01:07  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.47  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.46  2004/02/12 22:01:05  gadde
 * Use number of files as number of timepoints in non-BIAC case.
 *
 * Revision 1.45  2004/01/02 18:41:43  gadde
 * -Wall fixes
 *
 * Revision 1.44  2003/11/14 19:03:51  gadde
 * Add internal PSD name and PSD compile date/time.
 *
 * Revision 1.43  2003/10/22 16:08:48  gadde
 * Fix some compiler warnings
 *
 * Revision 1.42  2003/09/26 16:26:27  gadde
 *  * Don't accumulate slice spacings, just use total volume spacing and
 *  * divide.
 *
 * Revision 1.41  2003/09/26 15:06:39  gadde
 * Use calculated slice spacing.
 *
 * Revision 1.40  2003/09/08 16:40:41  gadde
 * Fix x/y typo.
 *
 * Revision 1.39  2003/08/19 19:35:41  gadde
 * New options scheme, so that options are available using whichever form of bxhabsorb you use.
 *
 * Revision 1.38  2003/08/19 14:35:04  gadde
 * Use the correct "dif" flag (user16, not user19).
 *
 * Revision 1.37  2003/08/05 15:53:39  gadde
 * Use display field of view to calculate spacing if we haven't gotten it
 * by other means.
 *
 * Revision 1.36  2003/07/31 21:12:50  gadde
 * Check for newer version of Pfile header.
 *
 * Revision 1.35  2003/07/31 15:18:46  gadde
 * #ifdef out BIAC-specific things
 *
 * Revision 1.34  2003/07/25 20:43:53  gadde
 * Windows fixes for the C++ conversion of some files.
 * Also reordered some headers so "interface" is not
 * defined before including gdome headers.
 *
 * Revision 1.33  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.32  2003/06/18 16:11:49  gadde
 * win32 fixes
 *
 * Revision 1.31  2003/06/18 16:07:14  gadde
 * fopen in binary mode
 *
 * Revision 1.30  2003/05/16 19:57:09  gadde
 * Require a basepath for datarec.
 *
 * Revision 1.29  2003/05/10 18:20:37  gadde
 * Don't null-terminate frags.
 *
 * Revision 1.28  2003/04/28 19:47:14  gadde
 * Move to catch-all BXH encapsulation program bxhabsorb -- dicom2bxh, pfile2bxh,
 * etc. will all be symbolic links to bxhabsorb.
 *
 * Revision 1.27  2003/04/11 16:17:44  gadde
 * Change default dimorder to x,y,z,t
 *
 * Revision 1.26  2003/04/11 15:17:28  gadde
 * Move some PSD-specific stuff into pfile2bxh, like acquisitionmatrix
 * and frequencydirection for spirals.
 *
 * Revision 1.25  2003/04/03 16:47:43  gadde
 * Add some more psds.
 *
 * Revision 1.24  2003/04/02 20:22:58  gadde
 * Get rid of warning.
 *
 * Revision 1.23  2003/03/24 16:49:24  gadde
 * Add support for ximg format
 *
 * Revision 1.22  2003/02/13 16:58:29  gadde
 * Multi-line literal snuck in -- purged.
 *
 * Revision 1.21  2003/01/17 16:16:17  gadde
 * Add version string to history
 *
 * Revision 1.20  2003/01/16 22:06:58  gadde
 * Fixed phraseology.
 *
 * Revision 1.19  2003/01/16 21:13:38  gadde
 * Add field of view from psd
 *
 * Revision 1.18  2003/01/16 20:55:43  gadde
 * Add psd-specific info
 *
 * Revision 1.17  2002/12/03 20:41:05  gadde
 * Big update --
 *  add new datarec module, convert more programs from domutil_ to bxh_,
 *  add new bxh_getElement* functions, and domutil_prettify.
 *
 * Revision 1.16  2002/11/25 16:21:31  gadde
 * Mega-update to merge in new library functions.
 *
 * Revision 1.15  2002/11/15 21:16:51  gadde
 * More uncommitted namespace changes
 *
 * Revision 1.14  2002/11/15 20:52:33  gadde
 * Now BXH files have the correct namespace declarations (at the expense
 * of some fragile code in dom_utils.c that breaks the GDOME abstraction
 * layer).
 *
 * Revision 1.13  2002/10/25 20:31:01  gadde
 * Added infrastructure to support different PSDs.  Nothing useful happens yet.
 *
 * Revision 1.12  2002/09/18 21:47:32  gadde
 * Added nshots, nechoes, bandwidth.
 * R,A,S dimensions calculated differently.
 *
 * Revision 1.11  2002/09/13 19:52:43  gadde
 * Fixed tr, te, ti units (us -> ms)
 * Added units for 't' dimension, and spacing for 'z' dimension.
 *
 * Revision 1.10  2002/09/10 04:44:53  gadde
 * Comment positioning.
 *
 * Revision 1.9  2002/09/10 04:27:34  gadde
 * /mriheader -> /bxh
 *
 * Revision 1.8  2002/09/09 19:49:37  gadde
 * Fixed integer division of MagneticFieldStrength -- oops.
 * Also, /mriheader -> /bxh
 *
 * Revision 1.7  2002/09/09 17:05:29  gadde
 * Now outputs orientation vectors (dimension/direction).
 *
 * Revision 1.6  2002/08/30 14:32:13  gadde
 * Now a C file.
 *
 * Revision 1.5  2002/08/21 19:45:18  gadde
 * mxh -> bxh
 *
 * Revision 1.4  2002/08/20 21:03:35  gadde
 * scantime was mislabeled as scandate.
 *
 * Revision 1.3  2002/08/20 20:08:54  gadde
 * Added const to rcsid to avoid compiler warning.
 *
 * Revision 1.2  2002/08/20 15:32:33  gadde
 * Everything moved from libxml to libgdome (DOM).  All utility functions
 * prefixed with domutil_.  XPath support still shaky in gdome, but dom_utils.c
 * accounts for this, and extra code can be removed when gdome fully supports
 * the DOM level 3 XPath specification.
 *
 */
