static const char rcsid[] = "$Id: bxh_convert_signafive.c,v 1.1 2009-01-15 20:55:18 gadde Exp $";

/*
 * signafive2bxh.cc --
 * 
 * Creates a BXH file based on given Signa 5 files.
 */

#include "bxh_utils.h"
#include "bxh_signa.h"

#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

extern char * domutil_errorbuf;

#ifndef TOPLEVELELEM
#define TOPLEVELELEM "bxh"
#endif

BXHDocPtr
createDocFromSigna5(const char **ifnames, int numfiles, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient, size_t dimzsize, size_t dimtsize)
{
    BXHDocPtr docp = NULL;
    BXHElementPtr imagedata = NULL;
    struct stat statbuf;
    char * dimorder = NULL;
    
    if (dimzsize == (size_t)-1 && hintsize[2] != (size_t)-1)
	dimzsize = hintsize[2];

    if (dimtsize == (size_t)-1 && hintsize[3] != (size_t)-1)
	dimtsize = hintsize[3];

    if (dimzsize == (size_t)-1)
	dimzsize = numfiles;

    if (dimtsize == (size_t)-1)
	dimtsize = numfiles / dimzsize;

    if (dimzsize * dimtsize != numfiles) {
	fprintf(stderr, "signafive2bxh: number of slices (%d) multiplied by number of timepoints (%d) does not equal number of input files (%d)\n", (int)dimzsize, (int)dimtsize, numfiles);
	goto FAIL;
    }

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;


    if (dimtsize == 1)
	dimorder = "x,y,z";
    else
	dimorder = "x,y,z,t";
    if (bxh_SIGNA_readIMGFromFile(docp, ifnames[0], dimzsize, dimtsize, dimorder) != 0)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;
    
    {
	bxhrawdatarec * datarec = NULL;
	int i;
	
	datarec = bxh_datarec_createFromElement(imagedata, NULL);
	bxh_datarec_frags_free(datarec);
	for (i = 0; i < numfiles; i++) {
	    struct stat statbuf;
	    if (stat(ifnames[i], &statbuf) == -1) {
		perror("stat");
		goto FAIL;
	    }
	    bxh_datarec_addfrag(datarec, ifnames[i], 7904, statbuf.st_size - 7904, ofname, 1);
	}

	{
	    char * hintnames[4] = { "x", "y", "z", "t" };
	    int dimnum;
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		bxhdimension * dimp = &datarec->dimensions[dimnum];
		int hintnum;
		for (hintnum = 0; hintnum < 4; hintnum++) {
		    if (strcmp(dimp->type, hintnames[hintnum]) == 0)
			break;
		}
		if (hintnum == 4) continue;
		if (hintorigin[hintnum] != HUGE_VAL)
		    dimp->origin = hintorigin[hintnum];
		if (hintspacing[hintnum] != HUGE_VAL)
		    dimp->spacing = hintspacing[hintnum];
		if (hintgap[hintnum] != HUGE_VAL)
		    dimp->gap = hintgap[hintnum];
		if (forceorient && hintnum < 3) {
		    if (dimp->direction == NULL)
			dimp->direction = (double *)malloc(sizeof(double)*3);
		    memset(dimp->direction, '\0', sizeof(double)*3);
		    if (tolower(forceorient[hintnum]) == 'r') {
			dimp->direction[0] = 1;
		    } else if (tolower(forceorient[hintnum]) == 'a') {
			dimp->direction[1] = 1;
		    } else if (tolower(forceorient[hintnum]) == 's') {
			dimp->direction[2] = 1;
		    } else if (tolower(forceorient[hintnum]) == 'l') {
			dimp->direction[0] = -1;
		    } else if (tolower(forceorient[hintnum]) == 'p') {
			dimp->direction[1] = -1;
		    } else if (tolower(forceorient[hintnum]) == 'i') {
			dimp->direction[2] = -1;
		    }
		}
	    }
	}

	bxh_datarec_writeToElement(imagedata, datarec);
	bxh_datarec_free(datarec); datarec = NULL;
    }

    goto EXIT;

FAIL:
    fprintf(stderr, "signafive2bxh: conversion failed.\n");
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

EXIT:
    if (imagedata)
	bxh_element_unref(imagedata);

    return docp;
}

int
createBXHFromSigna5(const char **ifnames, int numfiles, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient, size_t dimzsize, size_t dimtsize)
{
    int result = 0;
    BXHDocPtr docp = NULL;
    struct stat statbuf;
    
    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "createBXHFromSigna5: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromSigna5(ifnames, numfiles, ofname, hintsize, hintorigin, hintspacing, hintgap, forceorient, dimzsize, dimtsize)) == NULL)
	goto FAIL;

    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    result = -1;

EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

    return result;
}

BXHDocPtr
bxh_signafive2doc(const char * filename)
{
    size_t hintsizes[4] = { (size_t)-1, (size_t)-1, (size_t)-1, (size_t)-1 };
    double hintorigin[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintspacing[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintgap[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    return createDocFromSigna5(&filename, 1, "dummy.bxh", &hintsizes[0], &hintorigin[0], &hintspacing[0], &hintgap[0], NULL, 0, 0);
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.19  2008/01/28 17:01:53  gadde
 * Fix memory leak.
 *
 * Revision 1.18  2005/09/19 16:31:59  gadde
 * Documentation and help message updates.
 *
 * Revision 1.17  2005/07/07 15:09:22  gadde
 * Updates for ver11 pfiles.
 *
 * Revision 1.16  2005/03/08 18:09:58  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.15  2004/06/15 16:16:12  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.14  2004/02/24 18:01:07  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.13  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.12  2004/01/07 19:41:52  gadde
 * Fix error message.
 *
 * Revision 1.11  2004/01/02 18:41:20  gadde
 * Allow more than one timepoint for Signa files
 *
 * Revision 1.10  2003/08/19 19:35:41  gadde
 * New options scheme, so that options are available using whichever form of bxhabsorb you use.
 *
 * Revision 1.9  2003/07/25 20:43:54  gadde
 * Windows fixes for the C++ conversion of some files.
 * Also reordered some headers so "interface" is not
 * defined before including gdome headers.
 *
 * Revision 1.8  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.7  2003/06/18 16:11:36  gadde
 * win32 fixes
 *
 * Revision 1.6  2003/05/16 19:56:57  gadde
 * Require a basepath for datarec.
 *
 * Revision 1.5  2003/05/10 18:20:06  gadde
 * *** empty log message ***
 *
 * Revision 1.4  2003/04/28 20:51:09  gadde
 * Fix documentation
 *
 * Revision 1.3  2003/04/28 19:48:12  gadde
 * Move to catch-all BXH encapsulation program bxhabsorb -- dicom2bxh, pfile2bxh,
 * etc. will all be symbolic links to bxhabsorb.
 *
 * Revision 1.2  2003/04/18 19:10:44  gadde
 * Fix comments
 *
 * Revision 1.1  2003/04/18 19:10:19  gadde
 * *** empty log message ***
 */
