static const char rcsid[] = "$Id: bxh_utils-xerces.cpp,v 1.12 2007-01-16 19:34:27 gadde Exp $";

/* bxh_utils.cpp --
 *
 * Create and modify BXH files using these functions.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu), Nov. 2002.
 */

#include <bxh_config.h>

#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#ifndef WIN32
#include <unistd.h>
#endif

/* DOM includes */
#include <xercesc/dom/DOM.hpp>
#include <xercesc/sax/SAXException.hpp>
#include <xercesc/util/XMLUniDefs.hpp>
#include <xercesc/framework/LocalFileFormatTarget.hpp>
#include <xercesc/framework/MemBufInputSource.hpp>
#include <xercesc/framework/Wrapper4InputSource.hpp>
#include <pathan/XPathEvaluator.hpp>

/* prototypes and type declarations */
#include "bxh_utils.h"
#include "bxh_dom_utils-xerces.h"

XERCES_CPP_NAMESPACE_USE

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no versions specified)"
#endif
#ifndef XMLH_CVSSTR
#define XMLH_CVSSTR "(no CVS versions specified)"
#endif

#ifndef TOPLEVELELEM
#define TOPLEVELELEM "bxh"
#endif
static const char * BXHNS = "http://www.biac.duke.edu/bxh";
static const char * XCEDENS = "http://nbirn.net/Resources/Users/Applications/xcede/"
static const char * XMLNSNS = "http://www.w3.org/2000/xmlns/";

#define DI_DOMIMPLEMENTATION	DOMImplementation
#define DI_DOMDOCUMENT	DOMDocument
#define DI_DOMNODE	DOMNode
#define DI_DOMELEMENT	DOMElement
#define DI_DOMSTRINGPTR	XMLCh *

#define DI_CREATESTR(x)	XMLString::transcode(x)
#define DI_DELETESTR(x)	XMLString::release(&x); x = NULL

#ifdef __cplusplus
extern "C" {
#endif

#undef FUNC
#define FUNC "bxh_createDocument"
/**
 * Create an empty BXH document.
 *
 * @return The document, or NULL on error
 */
BXHDocPtr
bxh_createDocument()
{
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMSTRINGPTR name = NULL;
    DI_DOMSTRINGPTR value = NULL;
    DI_DOMSTRINGPTR tmpstr = NULL;
    DI_DOMNODE * tmpnode = NULL;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * bxh = NULL;

    static const XMLCh gLS[] = { chLatin_L, chLatin_S, chNull };

    if (domutil_initialize() != 0) {
	goto FAIL;
    }

    di = DOMImplementationRegistry::getDOMImplementation(gLS);

    name = DI_CREATESTR("bxh:" TOPLEVELELEM);
    tmpstr = DI_CREATESTR(BXHNS);
    try {
	doc = di->createDocument(tmpstr, name, NULL);
    } catch (const DOMException & exc) {
	fprintf(stderr, "DOMImplementation.createDocument: failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    DI_DELETESTR(tmpstr);
    DI_DELETESTR(name);

    try {
	bxh = doc->getDocumentElement();
    } catch (const DOMException & exc) {
	fprintf(stderr, "Document.documentElement: failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    
    value = DI_CREATESTR(" This is a BXH (BIAC XML Header) file. ");
    try {
	tmpnode = (DI_DOMNODE *)doc->createComment(value);
    } catch (const DOMException & exc) {
	fprintf(stderr, "Document.createComment: failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    DI_DELETESTR(value);
    try {
	doc->insertBefore((DI_DOMNODE *)tmpnode, (DI_DOMNODE *)bxh);
    } catch (const DOMException & exc) {
	sprintf(domutil_errorbuf, "Appending comment to doc failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    tmpnode = NULL;

    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns:bxh", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "version", "1.0", NULL) != 0)
	goto FAIL;

    bxh_normalizeNamespaces((BXHElementPtr)bxh);

    goto EXIT;

FAIL:

EXIT:
    if (name)
	DI_DELETESTR(name);
    if (value)
	DI_DELETESTR(value);
    if (tmpstr)
	DI_DELETESTR(tmpstr);

    return (BXHDocPtr)doc;
}

#undef FUNC
#define FUNC "bxh_readFile"
/**
 * Read a document from a BXH file.
 *
 * @param filename Name of BXH file
 * @return The document, or NULL on error
 */
BXHDocPtr
bxh_readFile(const char * filename)
{
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * bxh = NULL;

    DOMBuilder * parser = NULL;
    XMLCh * featurename = NULL;
    static const XMLCh gLS[] = { chLatin_L, chLatin_S, chNull };

    if (domutil_initialize() != 0) {
	goto FAIL;
    }

    di = DOMImplementationRegistry::getDOMImplementation(gLS);
    parser = ((DOMImplementationLS *)di)->createDOMBuilder(DOMImplementationLS::MODE_SYNCHRONOUS, 0);
    
    featurename = XMLString::transcode("namespaces");
    if (parser->canSetFeature(featurename, true)) {
	parser->setFeature(featurename, true);
    }
    XMLString::release(&featurename); featurename = NULL;

    featurename = XMLString::transcode("namespace-declarations");
    if (parser->canSetFeature(featurename, true)) {
	parser->setFeature(featurename, true);
    }
    XMLString::release(&featurename); featurename = NULL;

    try {
	doc = parser->parseURI(filename);
    } catch (const DOMException & exc) {
	fprintf(stderr, "Error reading XML file!\n\tException: %s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    if (doc == NULL) {
	fprintf(stderr, "Error reading XML file!  (perhaps it doesn't exist?)\n");
	goto FAIL;
    }

    try {
	bxh = doc->getDocumentElement();
    } catch (const DOMException & exc) {
	fprintf(stderr, "Error getting root element!\n");
	goto FAIL;
    }

    try {
	bxh = ((DI_DOMDOCUMENT *)((void *)doc))->getDocumentElement();
    } catch (const DOMException & exc) {
	fprintf(stderr, "Error getting root element!\n");
	goto FAIL;
    }

    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns:bxh", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns", BXHNS, XMLNSNS) != 0)
	goto FAIL;

    bxh_normalizeNamespaces((BXHElementPtr)bxh);
    goto EXIT;

FAIL:

EXIT:
    return (BXHDocPtr)doc;
}

/**
 * Read a BXH document from a memory buffer
 *
 * @param buf Character buffer
 * @return The document, or NULL on error
 */
BXHDocPtr
bxh_readBuffer(char * buf)
{
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * bxh = NULL;

    DOMBuilder * parser = NULL;
    XMLCh * featurename = NULL;
    MemBufInputSource * mbis = NULL;
    DOMInputSource * dis = NULL;
    static const XMLCh gLS[] = { chLatin_L, chLatin_S, chNull };

    if (domutil_initialize() != 0) {
	goto FAIL;
    }

    di = DOMImplementationRegistry::getDOMImplementation(gLS);
    parser = ((DOMImplementationLS *)di)->createDOMBuilder(DOMImplementationLS::MODE_SYNCHRONOUS, 0);

    featurename = XMLString::transcode("namespaces");
    if (parser->canSetFeature(featurename, true)) {
	parser->setFeature(featurename, true);
    }
    XMLString::release(&featurename); featurename = NULL;

    featurename = XMLString::transcode("namespace-declarations");
    if (parser->canSetFeature(featurename, true)) {
	parser->setFeature(featurename, true);
    }
    XMLString::release(&featurename); featurename = NULL;

    if ((mbis = new MemBufInputSource((XMLByte *)buf, strlen(buf), "[memory buffer]")) == NULL) {
	fprintf(stderr, "Error getting MemBufInputSource!\n");
	goto FAIL;
    }
    if ((dis = new Wrapper4InputSource(mbis)) == NULL) {
	fprintf(stderr, "Error getting Wrapper4InputSource!\n");
	goto FAIL;
    }
    
    try {
	doc = parser->parse(*dis);
    } catch (const DOMException & exc) {
	fprintf(stderr, "Error reading XML Buffer!\n\tException: %s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    if (doc == NULL) {
	fprintf(stderr, "Error reading XML Buffer!\n");
	goto FAIL;
    }
    try {
	bxh = doc->getDocumentElement();
    } catch (const DOMException & exc) {
	fprintf(stderr, "Error getting root element!\n");
	goto FAIL;
    }

    try {
	bxh = ((DI_DOMDOCUMENT *)((void *)doc))->getDocumentElement();
    } catch (const DOMException & exc) {
	fprintf(stderr, "Error getting root element!\n");
	goto FAIL;
    }

    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns:bxh", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns", BXHNS, XMLNSNS) != 0)
	goto FAIL;

    bxh_normalizeNamespaces((BXHElementPtr)bxh);
    goto EXIT;

FAIL:

EXIT:
    if (dis)
	delete dis;
    if (mbis)
	delete mbis;
    return (BXHDocPtr)doc;
}

#undef FUNC
#define FUNC "bxh_writeFile"
/**
 * Write a BXH document to a file.
 *
 * @param docp BXH document pointer
 * @param filename Name of BXH file to write
 * @return 0 on success, non-zero on error.
 */
int
bxh_writeFile(BXHDocPtr docp, const char * filename)
{
    int retval = 0;
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;
    BXHElementPtr root = NULL;

    DOMWriter * dw = NULL;
    XMLFormatTarget * ft = NULL; 
    static const XMLCh gLS[] = { chLatin_L, chLatin_S, chNull };

    di = DOMImplementationRegistry::getDOMImplementation(gLS);

    root = bxh_getRootElement(docp);

    bxh_normalizeNamespaces(root);
    bxh_prettify(docp, 0, 2);

    dw = di->createDOMWriter();
    ft = new LocalFileFormatTarget(filename);
    try {
	dw->writeNode(ft, *doc);
    } catch (const DOMException & exc) {
	fprintf(stderr, "DOMImplementation.saveDocToFile: failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    return retval;
}

#undef FUNC
#define FUNC "bxh_freeDocument"
/**
 * Free the memory associated with a BXH document.
 *
 * @param docp BXH document pointer
 */
void
bxh_freeDocument(BXHDocPtr docp)
{
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;
    delete doc;
}

/*************************/
/*** BXH File Sections ***/
/*************************/

#undef FUNC
#define FUNC "bxh_getDatarec"
/**
 * Find and return the \c \<datarec\> element that matches the given type
 * and subtype attributes.  It is an error if more than one element
 * matches.  Either type or subtype may be NULL.
 * If specified datarec doesn't exist, it is created.
 * Returned element should be freed by the caller using bxh_element_unref.
 *
 * @param docp BXH document pointer
 * @param type value for the "type" attribute
 * @param subtype value for the "subtype" attribute
 * @return a BXH element pointer, or NULL on error.
 */
BXHElementPtr
bxh_getDatarec(BXHDocPtr docp, const char * type, const char * subtype)
{
    BXHElementPtr root = NULL;
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;
    DI_DOMELEMENT * datarec = NULL;
    DI_DOMNODE * retval = NULL;
    const char * pathstart = "bxh:datarec";
    char * buf = NULL;
    int bufsize = 0;

    root = bxh_getRootElement(docp);

    bufsize =
	strlen(pathstart) +
	((type || subtype) ? 1 : 0) +
	(type ? (8 + strlen(type)) : 0) +
	((type && subtype) ? 5 : 0) +
	(subtype ? (11 + strlen(subtype)) : 0) +
	((type || subtype) ? 1 : 0) +
	1;
    buf = (char *)malloc(sizeof(char) * bufsize);

    buf[0] = '\0';
    strcat(&buf[0], pathstart);
    if (type || subtype) {
	strcat(&buf[0], "[");
	if (type) {
	    strcat(&buf[0], "@type='");
	    strcat(&buf[0], type);
	    strcat(&buf[0], "'");
	}
	if (type && subtype) {
	    strcat(&buf[0], " and ");
	}
	if (subtype) {
	    strcat(&buf[0], "@subtype='");
	    strcat(&buf[0], subtype);
	    strcat(&buf[0], "'");
	}
	strcat(&buf[0], "]");
    }
    if (domutil_getSingleNode((DI_DOMNODE *)root, &buf[0], &retval) != 0) {
//fprintf(stderr, "Can't find <datarec type=\"%s\" subtype=\"%s\">, creating it.\n", type ? type : "", subtype ? subtype : "");
	/* create it */
	if ((datarec = domutil_appendNewChildWithTextValue((DI_DOMDOCUMENT *)doc, (DI_DOMELEMENT *)root, "bxh:datarec", NULL, BXHNS)) == NULL)
	    goto FAIL;
//	if (type && domutil_setAttributeWithTextValue((DI_DOMDOCUMENT *)doc, (DI_DOMELEMENT *)datarec, "type", type, BXHNS) != 0)
	if (type && domutil_setAttributeWithTextValue((DI_DOMDOCUMENT *)doc, (DI_DOMELEMENT *)datarec, "type", type, NULL) != 0)
	    goto FAIL;
//	if (subtype && domutil_setAttributeWithTextValue((DI_DOMDOCUMENT *)doc, (DI_DOMELEMENT *)datarec, "subtype", subtype, BXHNS) != 0)
	if (subtype && domutil_setAttributeWithTextValue((DI_DOMDOCUMENT *)doc, (DI_DOMELEMENT *)datarec, "subtype", subtype, NULL) != 0)
	    goto FAIL;
	retval = (DI_DOMNODE *)datarec;
    }
    free(buf); buf = NULL;

    if (retval && retval->getNodeType() != DOMNode::ELEMENT_NODE) {
	fprintf(stderr, FUNC ": datarec node isn't an element?!?\n");
	goto FAIL;
    }
    goto EXIT;

FAIL:
    if (domutil_errorbuf[0])
	fprintf(stderr, "%s", &domutil_errorbuf[0]);

EXIT:
    if (buf) free(buf);

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_getDatarecArray"
/**
 * Find and return all the \c \<datarec\> elements in the document.
 * Returned elements should be freed by the caller using bxh_element_unref,
 * and the array itself should also be free()d.
 *
 * @param docp BXH document pointer
 * @return a null-terminated array of BXH element pointers, or NULL on error.
 */
BXHElementPtr *
bxh_getDatarecArray(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMNODE ** retval = NULL;
    int numnodes;

    root = bxh_getRootElement(docp);

    if ((numnodes = domutil_getNodeArray((DI_DOMNODE *)root, "bxh:datarec", XPathResult::UNORDERED_NODE_ITERATOR_TYPE, &retval)) == -1) {
	if (domutil_errorbuf[0]) {
	    fprintf(stderr, FUNC ": %s\n", domutil_errorbuf);
	}
    }

    if (retval) {
	int nodenum;
	for (nodenum = 0; nodenum < numnodes; nodenum++) {
	    if (retval[nodenum]->getNodeType() != DOMNode::ELEMENT_NODE) {
		fprintf(stderr, FUNC ": datarec node isn't an element?!?\n");
		domutil_freeNodeArray(&retval);
		retval = NULL;
		break;
	    }
	}
    }

    if (retval) {
	retval = (DI_DOMNODE **)realloc(retval, sizeof(DI_DOMNODE *) * (numnodes + 1));
	retval[numnodes] = NULL;
    }

    return (BXHElementPtr *)retval;
}

#undef FUNC
#define FUNC "bxh_getHistoryArray"
/**
 * Find and return all the \c \<history\> elements in the document.
 * Returned elements should be freed by the caller using bxh_element_unref,
 * and the array itself should also be free()d.
 *
 * @param docp BXH document pointer
 * @return a null-terminated array of BXH element pointers, or NULL on error.
 */
BXHElementPtr *
bxh_getHistoryArray(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMNODE ** retval = NULL;
    int numnodes;

    root = bxh_getRootElement(docp);

    if ((numnodes = domutil_getNodeArray((DI_DOMNODE *)root, "bxh:history", XPathResult::UNORDERED_NODE_ITERATOR_TYPE, &retval)) == -1) {
	if (domutil_errorbuf[0]) {
	    fprintf(stderr, FUNC ": %s\n", domutil_errorbuf);
	}
    }

    if (retval) {
	int nodenum;
	for (nodenum = 0; nodenum < numnodes; nodenum++) {
	    if (retval[nodenum]->getNodeType() != DOMNode::ELEMENT_NODE) {
		fprintf(stderr, FUNC ": history node isn't an element?!?\n");
		domutil_freeNodeArray(&retval);
		retval = NULL;
		break;
	    }
	}
    }

    if (retval) {
	retval = (DI_DOMNODE **)realloc(retval, sizeof(DI_DOMNODE *) * (numnodes + 1));
	retval[numnodes] = NULL;
    }

    return (BXHElementPtr *)retval;
}

#undef FUNC
#define FUNC "bxh_getAcquisitionData"
/**
 * Find and return the \c \<acquisitiondata\> element.
 * Returned element should be freed by the caller using bxh_element_unref.
 * If the element doesn't exist, it is created.
 *
 * @param docp BXH document pointer
 * @return a BXH element pointer, or NULL on error.
 */
BXHElementPtr
bxh_getAcquisitionData(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMNODE * retval = NULL;
    const char * path = "bxh:acquisitiondata";

    root = bxh_getRootElement(docp);

    if (domutil_getSingleNode((DI_DOMNODE *)root, path, &retval) != 0) {
	DI_DOMDOCUMENT * doc = NULL;
	if ((doc = ((DI_DOMELEMENT *)root)->getOwnerDocument()) != NULL) {
	    retval = (DI_DOMNODE *)domutil_appendNewChildWithTextValue((DI_DOMDOCUMENT *)doc, (DI_DOMELEMENT *)root, "acquisitiondata", NULL, BXHNS);
	}
    }

    if (retval && retval->getNodeType() != DOMNode::ELEMENT_NODE) {
	fprintf(stderr, FUNC ": acquisitiondata node isn't an element?!?\n");
	retval = NULL;
    }

    bxh_element_unref(root);

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_getSubject"
/**
 * Find and return the \c \<subject\> element.
 * Returned element should be freed by the caller using bxh_element_unref.
 * If the element doesn't exist, it is created.
 *
 * @param docp BXH document pointer
 * @return a BXH element pointer, or NULL on error.
 */
BXHElementPtr
bxh_getSubject(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMELEMENT * retval = NULL;
    const char * path = "bxh:subject";

    root = bxh_getRootElement(docp);

    if (domutil_getSingleNode((DI_DOMNODE *)root, path, (DI_DOMNODE **)&retval) != 0) {
	DI_DOMDOCUMENT * doc = NULL;
	if ((doc = ((DI_DOMELEMENT *)root)->getOwnerDocument()) != NULL) {
	    retval = domutil_appendNewChildWithTextValue((DI_DOMDOCUMENT *)doc, (DI_DOMELEMENT *)root, "subject", NULL, BXHNS);
	}
    }

    if (retval && retval->getNodeType() != DOMNode::ELEMENT_NODE) {
	fprintf(stderr, FUNC ": subject node isn't an element?!?\n");
	retval = NULL;
    }

    bxh_element_unref(root);

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_getRootElement"
/**
 * Get the element corresponding to the top-level \c \<bxh\> element.
 *
 * @param docp BXH document pointer
 * @return BXH element pointer on success, NULL on error.
 */
BXHElementPtr
bxh_getRootElement(BXHDocPtr docp)
{
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;
    DI_DOMNODE * retval = NULL;
    
    try {
	retval = (DI_DOMNODE *)(doc->getDocumentElement());
    } catch (const DOMException & exc) {
	fprintf(stderr, FUNC ": Can't get root document element\n");
    }

    if (retval && retval->getNodeType() != DOMNode::ELEMENT_NODE) {
	fprintf(stderr, FUNC ": root-level node isn't an element?!?\n");
	retval = NULL;
    }

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_addHistoryEntry"
/**
 * Add a history entry to the BXH document.
 *
 * @param docp BXH document pointer
 * @param date timestamp, or 0 if you want none
 * @param desc plain-text description of history entry
 * @return 0 on success, non-zero on error.
 */
int
bxh_addHistoryEntry(BXHDocPtr docp, time_t date, const char * desc)
{
    int retval = 0;
    BXHElementPtr bxh = NULL;
    BXHElementPtr historynode = NULL;
    BXHElementPtr entrynode = NULL;
    BXHElementPtr datenode = NULL;
    BXHElementPtr descnode = NULL;

    bxh = bxh_getRootElement(docp);

    if ((historynode = bxh_getChildElement(bxh, "history")) == NULL) {
	if ((historynode = bxh_appendAndReturnChildElement(bxh, "history", NULL)) == NULL)
	    goto FAIL;
    }
    if ((entrynode = bxh_appendAndReturnChildElement(historynode, "entry", NULL)) == NULL)
	goto FAIL;
    if (date != 0) {
	static char timebuf[64];
	struct tm * tmp;
	tmp = localtime(&date);
	sprintf(timebuf, "%04d-%02d-%02d %02d:%02d:%02d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
	if ((datenode = bxh_appendAndReturnChildElement(entrynode, "date", timebuf)) == NULL)
	    goto FAIL;
    }
    if ((descnode = bxh_appendAndReturnChildElement(entrynode, "description", desc)) == NULL)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (descnode) bxh_element_unref(descnode);
    if (datenode) bxh_element_unref(datenode);
    if (entrynode) bxh_element_unref(entrynode);
    if (historynode) bxh_element_unref(historynode);
    if (bxh) bxh_element_unref(bxh);
    return retval;
}

#undef FUNC
#define FUNC "bxh_addXCEDEHistoryEntry"
/**
 * Add a XCEDE-style history entry to the BXH document.
 *
 * @param docp BXH document pointer
 * @param date timestamp, or 0 if you want none
 * @param desc plain-text description of history entry
 * @return 0 on success, non-zero on error.
 */
static int
bxh_addXCEDEHistoryEntry(BXHDocPtr docp,
			 const char * programName,
			 const char ** programArguments,
			 int numargs)
{
    int retval = 0;
    BXHElementPtr bxh = NULL;
    BXHElementPtr historynode = NULL;
    BXHElementPtr entrynode = NULL;
    BXHElementPtr datenode = NULL;
    BXHElementPtr descnode = NULL;
    BXHElementPtr stepnode = NULL;
    int argnum;
    size_t argslen = 0;
    char * args = NULL;
    char * envval = NULL;
    time_t date;
    double version;
    static char numbuf[20];

    bxh = bxh_getRootElement(docp);

    if ((historynode = bxh_getChildElement(bxh, "history")) == NULL) {
	if ((historynode = bxh_appendAndReturnChildElement(bxh, "history", NULL)) == NULL)
	    goto FAIL;
    }
    if ((entrynode = bxh_appendAndReturnChildElement(historynode, "entry", NULL)) == NULL)
	goto FAIL;
    date = time(NULL);
    if (date != 0) {
	static char timebuf[64];
	struct tm * tmp;
	tmp = localtime(&date);
	sprintf(timebuf, "%04d-%02d-%02d %02d:%02d:%02d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
	if ((datenode = bxh_appendAndReturnChildElement(entrynode, "date", timebuf)) == NULL)
	    goto FAIL;
    }
    if ((descnode = bxh_appendAndReturnChildElement(entrynode, "description", NULL)) == NULL)
	goto FAIL;
    if ((stepnode = bxh_appendAndReturnChildElement(descnode, "processStep", NULL)) == NULL)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "programName", programName) != 0)
	goto FAIL;

    argslen = 0;
    for (argnum = 0; argnum < numargs; argnum++) {
	argslen += strlen(programArguments[argnum]) + 1;
    }
    args = (char *)malloc(sizeof(char)*argslen);
    argslen = 0;
    for (argnum = 0; argnum < numargs; argnum++) {
	strcpy(&args[argslen], programArguments[argnum]);
	argslen += strlen(programArguments[argnum]);
	args[argslen] = ' ';
	argslen++;
    }
    argslen--;
    args[argslen] = '\0';
    if (bxh_appendChildElement(stepnode, "programArgument", args) != 0)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "version", XMLH_VERSIONSTR) != 0)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "cvs", XMLH_CVSSTR) != 0)
	goto FAIL;
    if ((envval = getenv("USER")) != NULL) {
	if (bxh_appendChildElement(stepnode, "user", envval) != 0)
	    goto FAIL;
    } else {
	if (bxh_appendChildElement(stepnode, "user", "(unknown)") != 0)
	    goto FAIL;
    }
    if ((envval = getenv("MACHTYPE")) != NULL) {
	if (bxh_appendChildElement(stepnode, "machine", envval) != 0)
	    goto FAIL;
    } else {
	if (bxh_appendChildElement(stepnode, "machine", "(unknown)") != 0)
	    goto FAIL;
    }
    if ((envval = getenv("OSTYPE")) != NULL) {
	if (bxh_appendChildElement(stepnode, "platform", envval) != 0)
	    goto FAIL;
    } else {
	if (bxh_appendChildElement(stepnode, "platform", "(unknown)") != 0)
	    goto FAIL;
    }
    if (bxh_appendChildElement(stepnode, "platformVersion", "(unknown)") != 0)
	goto FAIL;
#if defined(__GNUC__)
    envval = "GCC";
#elif defined(_MSC_VER)
    envval = "MSC";
#else
    envval = "UNKNOWN";
#endif
    if (bxh_appendChildElement(stepnode, "compilerName", "(unknown)") != 0)
	goto FAIL;

    version = -1;
#if defined(__GNUC__)
#if defined(__GNU_PATCHLEVEL__)
    version = (__GNUC__ * 10000	+ __GNUC_MINOR__ * 100 + __GNUC_PATCHLEVEL__);
# else
    version = (__GNUC__ * 10000 + __GNUC_MINOR__ * 100);
# endif
#elif defined(_MSC_VER)
    version = (_MSC_VER);
#endif
    if (version == -1)
	strcpy(&numbuf[0], "(unknown)");
    else
	sprintf(&numbuf[0], "%g", version);
    if (bxh_appendChildElement(stepnode, "compilerVersion", &numbuf[0]) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (args) free(args);
    if (stepnode) bxh_element_unref(stepnode);
    if (descnode) bxh_element_unref(descnode);
    if (datenode) bxh_element_unref(datenode);
    if (entrynode) bxh_element_unref(entrynode);
    if (historynode) bxh_element_unref(historynode);
    if (bxh) bxh_element_unref(bxh);
    return retval;
}

#undef FUNC
#define FUNC "bxh_addAutoHistoryEntry"
/**
 * Add a history entry to the BXH document.
 * This is a more automated alternative to the the generic bxh_addHistoryEntry.
 *
 * @param docp BXH document pointer
 * @param programName executable name or other label
 * @param programArguments original argument list
 * @param numargs number of arguments
 * @return 0 on success, non-zero on error.
 */
int
bxh_addAutoHistoryEntry(BXHDocPtr docp,
			const char * programName,
			const char ** programArguments,
			int numargs)
{
    int retval = 0;
    static char hostname[32];
    static char descbuf[8192];

#ifdef HAVE_LIBXSLT
    if (bxh_doXCEDE) {
	return bxh_addXCEDEHistoryEntry(docp, programName, programArguments, numargs);
    }
#endif
    
    gethostname(&hostname[0], sizeof(hostname));
    hostname[sizeof(hostname)-1] = '\0';
    if (numargs > 0) {
	sprintf(&descbuf[0], "\nGenerated on %s by '%s' from %d input files (first is '%s').\n%s",
		hostname, programName,
		numargs, programArguments[0], XMLH_VERSIONSTR);
    } else {
	sprintf(&descbuf[0], "\nGenerated on %s by '%s'.\n%s",
		hostname, programName, XMLH_VERSIONSTR);
    }
    if (bxh_addHistoryEntry(docp, time(NULL), &descbuf[0]) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    return retval;
}

/* generic element functions */

#undef FUNC
#define FUNC "bxh_getElementStringValue"
/**
 * Find and return the string value within the element elemp.
 * It is an error if the child is not a simple element enclosing text.
 * The value returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to character array
 * @return 0 on success, -1 on error
 */
int
bxh_getElementStringValue(BXHElementPtr elemp, char ** retp)
{
    return domutil_getStringValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementDoubleValue"
/**
 * Find and return the double value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * numeric value.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double
 * @return 0 on success, -1 on error
 */
int
bxh_getElementDoubleValue(BXHElementPtr elemp, double * retp)
{
    return domutil_getDoubleValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementDoubleListValue"
/**
 * Find and return the double list value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * space-separated list of numeric values.
 * The array returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double array
 * @return length of list on success, -1 on error
 */
int
bxh_getElementDoubleListValue(BXHElementPtr elemp, double ** retp)
{
    return domutil_getDoubleListValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementIntValue"
/**
 * Find and return the integer value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * integral value.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to int
 * @return 0 on success, -1 on error
 */
int
bxh_getElementIntValue(BXHElementPtr elemp, int * retp)
{
    return domutil_getIntValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementIntListValue"
/**
 * Find and return the integer list value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * space-separated list of integral values.
 * The array returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double array
 * @return length of list on success, -1 on error
 */
int
bxh_getElementIntListValue(BXHElementPtr elemp, int ** retp)
{
    return domutil_getIntListValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementUnsignedIntValue"
/**
 * Find and return the unsigned integer value within the element elemp.
 * It is an error if the child is not a simple element enclosing an
 * unsigned integral value.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to unsigned int
 * @return 0 on success, -1 on error
 */
int
bxh_getElementUnsignedIntValue(BXHElementPtr elemp, unsigned int * retp)
{
    return domutil_getUnsignedIntValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementUnsignedIntListValue"
/**
 * Find and return the unsigned integer list value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * space-separated list of unsigned integral values.
 * The array returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double array
 * @return length of list on success, -1 on error
 */
int
bxh_getElementUnsignedIntListValue(BXHElementPtr elemp, unsigned int ** retp)
{
    return domutil_getUnsignedIntListValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getChildElement"
/**
 * Find and return the (sole) child element of parentp whose tag
 * matches the name qname.  It is an error if there is more than
 * one child with the same name.
 * Returned element should be freed by the caller using bxh_element_unref.
 *
 * @param parentp BXH element within which to search
 * @param qname name of child element (may include prefix)
 * @return the child element, or NULL on error.
 */
BXHElementPtr
bxh_getChildElement(BXHElementPtr parentp, const char * qname)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE * result = NULL;
    char * buf = NULL;
    
    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if (domutil_getSingleNode((DI_DOMNODE *)parent, &buf[0], &result) == -1)
	goto FAIL;
    if (result->getNodeType() != DOMNode::ELEMENT_NODE) {
	fprintf(stderr, FUNC ": node %s not an element\n", qname);
	goto FAIL;
    }

    goto EXIT;

FAIL:
    result = NULL;

EXIT:
    return (BXHElementPtr)result;
}

#undef FUNC
#define FUNC "bxh_getChildElementArray"
/**
 * Find and return the child element(s) of parentp whose tags
 * match the name qname.
 * The returned array will have a NULL pointer in the position past
 * the last valid element.
 * It is an error if any of the matched nodes are not elements.
 * Each element in the returned array should be freed by the caller
 * using bxh_element_unref.
 * The returned array itself should also be free()d by the caller.
 *
 * @param parentp BXH element within which to search
 * @param qname name of child element (may include prefix)
 * @return a null-terminated array of child elements, or NULL on error.
 */
BXHElementPtr *
bxh_getChildElementArray(BXHElementPtr parentp, const char * qname)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE ** result = NULL;
    int numnodes;
    int nodenum;
    char * buf = NULL;
    
    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if ((numnodes = domutil_getNodeArray((DI_DOMNODE *)parent, &buf[0], XPathResult::UNORDERED_NODE_ITERATOR_TYPE, &result)) == -1)
	goto FAIL;
    for (nodenum = 0; nodenum < numnodes; nodenum++) {
	if (result[nodenum]->getNodeType() != DOMNode::ELEMENT_NODE) {
	    fprintf(stderr, FUNC ": node %s not an element\n", qname);
	    goto FAIL;
	}
    }
    result = (DI_DOMNODE **)realloc(result, sizeof(DI_DOMNODE *) * (numnodes + 1));
    result[numnodes] = NULL;

    goto EXIT;

FAIL:
    if (result) {
	free(result);
    }
    result = NULL;

EXIT:
    return (BXHElementPtr *)result;
}

#undef FUNC
#define FUNC "bxh_getChildElementStringValue"
/**
 * Find and return the value within the (sole) child element of parentp
 * whose tag matches the name qname.  It is an error if there is more
 * than one child with the same name, or if the child is not a simple
 * element enclosing text.  The return value must be free()d by the caller.
 *
 * @param parentp BXH element within which to search
 * @param qname name of child element (may include prefix)
 * @return a newly-allocated string, or NULL on error.
 */
char *
bxh_getChildElementStringValue(BXHElementPtr parentp, const char * qname)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    char * result = NULL;
    char * buf = NULL;
    
    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if (domutil_getStringValue((DI_DOMNODE *)parent, &buf[0], &result) == -1)
	goto FAIL;

    goto EXIT;

FAIL:
    if (result)
	free(result);
    result = NULL;

EXIT:
    return result;
}

#undef FUNC
#define FUNC "bxh_setChildElement"
/**
 * Replace (if any) the sole child element of parentp whose tag matches
 * the name qname with a new child element of the same name, and with
 * the given text value.  It is an error if there exists more than one
 * child with the same name.
 *
 * @param parentp parent BXH element
 * @param qname name of child element (may include prefix)
 * @param value text string value of child element
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_setChildElement(BXHElementPtr parentp,
		    const char * qname, const char * value)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE * child = NULL;
    char * buf = NULL;
    
    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if (domutil_getSingleNode((DI_DOMNODE *)parent, &buf[0], &child) < 0)
	return -1;
    return bxh_appendChildElement(parentp, &buf[0], value);
}

#undef FUNC
#define FUNC "bxh_appendChildElement"
/**
 * Add a new child element of parentp with tag qname and value value.
 *
 * @param parentp parent BXH element
 * @param qname name of child element (may include prefix)
 * @param value text string value of child element
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_appendChildElement(BXHElementPtr parentp,
		       const char * qname, const char * value)
{
    BXHElementPtr newchild = bxh_appendAndReturnChildElement(parentp, qname, value);
    if (newchild == NULL)
	return -1;
    bxh_element_unref(newchild); newchild = NULL;
    return 0;
}

#undef FUNC
#define FUNC "bxh_appendAndReturnChildElement"
/**
 * Add a new child element of parentp with tag qname and value value,
 * and return the child.  The returned element should be freed by the
 * caller using bxh_element_unref().
 *
 * @param parentp parent BXH element
 * @param qname name of child element (may include prefix)
 * @param value text string value of child element
 * @return 0 on sucess, non-zero on error.
 */
BXHElementPtr
bxh_appendAndReturnChildElement(BXHElementPtr parentp,
				const char * qname, const char * value)
{
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE * child = NULL;
    char * buf = NULL;
    
    if ((doc = parent->getOwnerDocument()) == NULL)
	goto FAIL;

    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if ((child = (DI_DOMNODE *)domutil_appendNewChildWithTextValue(doc, parent, &buf[0], value, BXHNS)) == NULL)
	goto FAIL;

    goto EXIT;

FAIL:
    child = NULL;

EXIT:
    return (BXHElementPtr)child;
}

#undef FUNC
#define FUNC "bxh_removeChildElement"
/**
 * Remove childp from parentp.
 *
 * @param parentp parent BXH element
 * @param childp child BXH element
 * @return 0 on sucess, non-zero on error.
 */
BXHElementPtr
bxh_removeChildElement(BXHElementPtr parentp, BXHElementPtr childp)
{
    DI_DOMNODE * removed = NULL;
    
    if ((removed = ((DI_DOMELEMENT *)parentp)->removeChild((DI_DOMNODE *)childp)) == NULL)
	return NULL;
    return removed;
}

#undef FUNC
#define FUNC "bxh_element_unref"
/**
 * Decrement the reference count for the given element.
 *
 * @param elemp BXH element
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_element_unref(BXHElementPtr elemp)
{
    return 0;
}

/* generic attribute functions */

#undef FUNC
#define FUNC "bxh_getAttributeStringValue"
/**
 * Find and return the value of the attribute of parentp whose tag
 * matches the name qname.  The return value must be free()d by the
 * caller.
 *
 * @param parentp BXH element within which to search
 * @param qname name of child attribute
 * @return a newly-allocated string, or NULL on error.
 */
char *
bxh_getAttributeStringValue(BXHElementPtr parentp, const char * qname)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    char * path = NULL;
    char * result = NULL;

    path = (char *)malloc(sizeof(char)*strlen(qname)+2);
    path[0] = '\0';
    strcat(path, "@");
    strcat(path, qname);
    
    if (domutil_getStringValue((DI_DOMNODE *)parent, path, &result) == -1)
	goto FAIL;
    
    goto EXIT;

FAIL:
    if (result)
	free(result);
    result = NULL;
    
EXIT:
    if (path)
	free(path);
    return result;
}

#undef FUNC
#define FUNC "bxh_setAttribute"
/**
 * Replace (if any) the attribute of parentp whose tag matches the name
 * name with a new attribute with the given name and text value.
 *
 * @param parentp parent BXH element
 * @param qname name of attribute
 * @param value text string value of attribute
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_setAttribute(BXHElementPtr parentp, const char * qname, const char * value)
{
    int retval = 0;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;

    if ((doc = parent->getOwnerDocument()) == NULL)
	goto FAIL;

//    if (domutil_setAttributeWithTextValue(doc, parent, &buf[0], value, BXHNS) == -1)
    if (domutil_setAttributeWithTextValue(doc, parent, qname, value, NULL) == -1)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    return retval;
}

#undef FUNC
#define FUNC "bxh_attribute_unref"
/**
 * Decrement the reference count for the given attribute.
 *
 * @param attrp BXH attribute
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_attribute_unref(BXHAttributePtr attrp)
{
    return 0;
}


/* generic comment functions */

#undef FUNC
#define FUNC "bxh_appendComment"
/**
 * Create a comment with the given text and append it
 * to the given parent element.
 *
 * @param parentp the parent of the new comment
 * @param valuestr text that will appear inside the comment
 *            If NULL, then the element will be empty.
 * @param prependnewline if non-zero, will prepend a newline before comment
 * @return The new child element, or NULL on error
 */
BXHCommentPtr
bxh_appendComment(BXHElementPtr parentp,
		  const char * valuestr,
		  int prependnewline)
{
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DOMComment * retval = NULL;

    if ((doc = parent->getOwnerDocument()) == NULL)
	goto FAIL;

    if ((retval = domutil_appendComment(doc, parent, valuestr, prependnewline)) == NULL)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = NULL;

EXIT:
    return (BXHCommentPtr) retval;
}

#undef FUNC
#define FUNC "bxh_comment_unref"
/**
 * Decrement the reference count for the given comment.
 *
 * @param commentp BXH comment
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_comment_unref(BXHCommentPtr commentp)
{
    return 0;
}


/* miscellaneous BXH functions */

/**
 * See domutil_normalizeNamespaces().
 *
 * @param start root of the subtree to normalize
 */
void
bxh_normalizeNamespaces(BXHElementPtr start)
{
    domutil_normalizeNamespaces((DI_DOMELEMENT *)start);
}

/**
 * See domutil_prettify().
 *
 * @param start root of the subtree to prettify
 * @param startindent number of spaces to indent start element
 * @param indentstep how many spaces to indent each level
 */
void
bxh_prettify(BXHDocPtr start, int startindent, int indentstep) {
    domutil_prettify((DI_DOMNODE *)(DI_DOMDOCUMENT *)start, startindent, indentstep);
}

#ifdef __cplusplus
}
#endif

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.11  2005/02/25 21:48:03  gadde
 * XCEDE updates
 *
 * Revision 1.10  2004/06/08 19:57:46  gadde
 * Some xerces updates
 *
 * Revision 1.9  2004/04/02 16:23:53  gadde
 * Move dom_utils to bxh_dom_utils, and add some missing dependencies
 *
 * Revision 1.8  2004/02/23 19:30:33  gadde
 * Add preliminary support for BIRN XML (requires libxslt)
 *
 * Revision 1.7  2003/08/04 19:22:01  gadde
 * Add support for reading from memory buffer.
 *
 * Revision 1.6  2003/08/04 18:50:54  gadde
 * Add some casts
 *
 * Revision 1.5  2003/08/04 17:15:20  gadde
 * Code rearrangement
 *
 * Revision 1.4  2003/08/01 20:44:02  gadde
 * Fix typo.
 *
 * Revision 1.3  2003/08/01 20:40:28  gadde
 * Some updates to help ease the migration of code between DOM implementations
 *
 * Revision 1.2  2003/08/01 17:55:55  gadde
 * Merge in XERCES-specific files
 *
 * Revision 1.12.2.4  2003/07/31 21:15:41  gadde
 * Cosmetic fixes
 *
 * Revision 1.12.2.3  2003/07/29 16:08:50  gadde
 * Doc fixes.
 *
 * Revision 1.12.2.2  2003/07/28 20:55:36  gadde
 * Slight integration of libxml2/Xerces branches
 *
 * Revision 1.12.2.1  2003/07/25 17:31:38  gadde
 * Committing branch that compiles with Xerces/Pathan
 *
 * Revision 1.11  2003/06/18 19:36:59  gadde
 * Fix DOM call.
 *
 * Revision 1.10  2003/06/18 18:52:53  gadde
 * Add function to read BXH file from memory buffer.
 *
 * Revision 1.9  2003/04/17 19:33:45  gadde
 * Use more bxh functions (as opposed to domutils functions).
 * Add bxh_appendAndReturnChildElement for convenience.
 *
 * Revision 1.8  2003/02/12 19:31:36  gadde
 * Updated doxygen documentation.
 *
 * Revision 1.7  2003/02/11 17:49:15  gadde
 * Fix documentation of List functions
 *
 * Revision 1.6  2003/01/06 18:19:07  gadde
 * Fixed off-by-one month error.
 * Many comments changed/updated.
 *
 * Revision 1.5  2002/12/06 17:19:43  gadde
 * Use #defines for xmlns namespace
 *
 * Revision 1.4  2002/12/04 17:21:50  gadde
 * Adding new module files
 *
 * Revision 1.3  2002/12/03 21:08:35  gadde
 * Fixed some docs
 *
 * Revision 1.2  2002/12/03 20:41:03  gadde
 * Big update --
 *  add new datarec module, convert more programs from domutil_ to bxh_,
 *  add new bxh_getElement* functions, and domutil_prettify.
 *
 * Revision 1.1  2002/11/25 16:23:10  gadde
 * Mega-update to merge in new library functions.
 *
 */
