static const char rcsid[] = "$Id: opts.c,v 1.14 2005-01-11 22:22:40 gadde Exp $";

/* opts.c --
 *
 * A module to parse command-line options
 */

/* Copyright (c) 1997-1999 Syam Gadde and Duke University */

#include <bxh_config.h>

#include "opts.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <limits.h>
#include <errno.h>
#include <math.h>
#ifdef WIN32
#include <float.h>
#endif
#ifdef __linux__
#include <values.h>
#endif

#ifdef WIN32
#define MAXFLOAT FLT_MAX
#endif

/* size_t is always unsigned */
#if defined(HAVE_STRTOULL) && (SIZEOF_SIZE_T == SIZEOF_UNSIGNED_LONG_LONG)
#define OPT_SIZE_TMPTYPE unsigned long long
#define OPT_SIZE_STRTO strtoull
#define OPT_SIZE_PRINTF "%llu"
#elif defined(HAVE_STRTOUQ) && (SIZEOF_SIZE_T == SIZEOF_U_QUAD_T)
#define OPT_SIZE_TMPTYPE u_quad_t
#define OPT_SIZE_STRTO strtouq
#define OPT_SIZE_PRINTF "%qu"
#else
#define OPT_SIZE_TMPTYPE unsigned long
#define OPT_SIZE_STRTO strtoul
#define OPT_SIZE_PRINTF "%lu"
#endif

/* time_t is always signed */
#if defined(HAVE_STRTOLL) && (SIZEOF_TIME_T == SIZEOF_LONG_LONG)
#define OPT_TIME_TMPTYPE long long
#define OPT_TIME_STRTO strtoll
#define OPT_TIME_PRINTF "%lld"
#elif defined(HAVE_STRTOQ) && (SIZEOF_TIME_T == SIZEOF_QUAD_T)
#define OPT_TIME_TMPTYPE quad_t
#define OPT_TIME_STRTO strtoq
#define OPT_TIME_PRINTF "%qd"
#else
#define OPT_TIME_TMPTYPE long
#define OPT_TIME_STRTO strtol
#define OPT_TIME_PRINTF "%ld"
#endif

size_t
opt_type2size[] = {
    0,
    sizeof(int),
    sizeof(int), sizeof(unsigned int),
    sizeof(long), sizeof(unsigned long),
#ifdef HAVE_STRTOLL
    sizeof(long long),
#endif
#ifdef HAVE_STRTOULL
    sizeof(unsigned long long),
#endif
#ifdef HAVE_STRTOQ
    sizeof(quad_t),
#endif
#ifdef HAVE_STRTOUQ
    sizeof(u_quad_t),
#endif
    sizeof(size_t), sizeof(time_t),
    sizeof(float), sizeof(double),
    sizeof(char *)
};

char *
opt_type2str[] = {
    "",
    "",
    "int", "uint",
    "long", "ulong",
#ifdef HAVE_STRTOLL
    "longlong",
#endif
#ifdef HAVE_STRTOULL
    "ulonglong",
#endif
#ifdef HAVE_STRTOQ
    "quad_t",
#endif
#ifdef HAVE_STRTOUQ
    "uquad_t",
#endif
    "size_t", "time_t",
    "float", "double",
    "str"
};

/* User can assign this function: */
void (*opt_ext_usage)() = NULL;

void
opt_usage(int numopts, opt_data * opts)
{
    int optind;

    if (opt_ext_usage) {
	(*opt_ext_usage)();
	return;
    }
    
    for (optind = 0; optind < numopts; optind++) {
	opt_data * curropt = &opts[optind];
	const char * tmpdescr;
	size_t namelen = strlen(curropt->name);
	int outlen = 0;
	int indent = 0;

	if (namelen) {
	    if (curropt->type != OPT_VAL_BOOL) {
		fprintf(stderr, "  %s%s ",
			(curropt->flags & OPT_FLAGS_FULL) ? "--" : "-",
			curropt->name);
		fprintf(stderr, "<%s>", opt_type2str[curropt->type]);
		fprintf(stderr, "\n");
	    }
	    fprintf(stderr, "  %s%s",
		    (curropt->flags & OPT_FLAGS_FULL) ? "--" : "-",
		    curropt->name);
	    if (curropt->type != OPT_VAL_BOOL)
		fprintf(stderr, "%s<%s>",
			(curropt->flags & OPT_FLAGS_FULL) ? "=" : "",
			opt_type2str[curropt->type]);
	    fprintf(stderr, "\n");
	}

	/* display the description with 8 space indent, formatted for
	 * 79 columns.  Yes, I did spend too much time writing this. */
	if (curropt->descr)
	    tmpdescr = curropt->descr;
	else
	    tmpdescr = "";
	if (namelen)
	    indent = 8;
	indent += strspn(tmpdescr, " \t\r\n");
	tmpdescr += strspn(tmpdescr, " \t\r\n");
	outlen = 0;
	while (tmpdescr[0] != '\0') {
	    /* precondition: tmpdescr has something interesting to print */
	    static const char spacestr[16] = "                ";
	    size_t wordsize;
	    char * nlptr;

	    if (outlen == 0) {
		int i;
		for (i = 0; i < indent; i += sizeof(spacestr)) {
		    int j = indent - i;
		    if (j > sizeof(spacestr)) j = sizeof(spacestr);
		    fprintf(stderr, "%.*s", j, spacestr);
		}
		outlen = indent;
	    }

	    wordsize = strspn(tmpdescr, " \t\r\n");
	    wordsize += strcspn(tmpdescr+wordsize, " \t\r\n");
	    nlptr = strchr(tmpdescr, '\n');
	    if ((nlptr && nlptr < tmpdescr + wordsize) ||
		(wordsize + outlen > 79 && outlen != indent)) {
		fprintf(stderr, "\n");
		outlen = 0;
		if (nlptr && nlptr < tmpdescr + wordsize) {
		    tmpdescr = nlptr + 1;
		} else {
		    tmpdescr += strspn(tmpdescr, " \t\r\n");
		}
		continue;
	    }
	    outlen += fprintf(stderr, "%.*s", (int)wordsize, tmpdescr);
	    tmpdescr += wordsize;
	}
	if (curropt->descr)
	    fprintf(stderr, "\n");
    }
}

void
opt_err(opt_err_data * err)
{
    fprintf(stderr, "Error in parsing option '%s'\n", err->arg);
    fprintf(stderr, "Use --help for usage info.\n");
    exit(-1);
}

void
opt_amb_err(opt_err_data * err)
{
    fprintf(stderr, "Option '%s' is ambiguous!\n", err->arg);
    fprintf(stderr, "Use --help for usage info.\n");
    exit(-1);
}

void
opt_range_err(opt_err_data * err)
{
    fprintf(stderr, "Option '%s' value '%s' is out of range for <%s>\n",
	    err->arg, err->val, opt_type2str[err->opts[err->optind].type]);
    fprintf(stderr, "Use --help for usage info.\n");
    exit(-1);
}

void
opt_bool_err(opt_err_data * err)
{
    fprintf(stderr, "Boolean option '%s' cannot take a value\n", err->arg);
    fprintf(stderr, "Use --help for usage info.\n");
    exit(-1);
}

void
opt_val_err(opt_err_data * err)
{
    fprintf(stderr, "Option '%s' value '%s' is invalid\n", err->arg, err->val);
    fprintf(stderr, "Use --help for usage info.\n");
    exit(-1);
}

void
opt_int_err(opt_err_data * err)
{
    fprintf(stderr, "Internal error with data for option '%s'\n",
	    err->opts[err->optind].name);
    exit(-1);
}

void
opt_space_warn(opt_err_data * err)
{
    fprintf(stderr,
	    "More than %d instance%s of option \"%s\" not allowed\n",
	    err->opts[err->optind].numvals,
	    (err->opts[err->optind].numvals == 1) ? "" : "s",
	    err->opts[err->optind].name);
    exit(-1);
}

#undef FUNC
#define FUNC "opt_parse"
/**
 * Parse command-line options.
 *
 * @param argc number of arguments in #argv
 * @param argv argument array
 * @param numopts number of options in #opts array
 * @param opts option array
 * @param print print out specified options
 *
 * This function takes an array of command-line arguments, and parses
 * options contained within it according to the options specificied by
 * the #opts array.  All arguments parsed as options are moved to the
 * end of the argument array, and non-option arguments are moved to
 * the beginning.  The number of arguments parsed as options is returned.
 * So, for example, if a program was called as
 * './myprogram --optA --optB file1 file2', the following code:
 *
 *   argc -= opt_parse(argc, argv, numopts, opts, 0);
 *
 * will effectively leave the caller with a smaller #argv array
 * looking as if the program was called as './myprogram file1 file2',
 * leaving all the arguments that opt_parse() did not deal with
 * (and therefore the caller's responsibility) at the beginning of
 * the array.
 * Note: the executable name (#argv[0]) remains in its original place.
 * Returns: number of args (not options!) successfully parsed as options.
 */
int
opt_parse(int argc, char **argv,
	  int numopts, opt_data * opts,
	  int print)
{
    opt_err_data err;
    int firstoptind = 1;
    int * savednumvals;
    void ** savedvals;
    char * leftover;
    char ** leftoverp = &leftover;
    unsigned int currformat;
    int optind;
    int argind;
    char * equalloc = NULL;
    int ambiguity;

    err.numopts = numopts;
    err.opts = opts;

    savedvals = (void **)malloc(sizeof(void *) * numopts);
    savednumvals = (int *)malloc(sizeof(int) * numopts);
    for (optind = 0; optind < numopts; optind++) {
	/* save vals */
	savedvals[optind] = opts[optind].vals;
	/* save numvals and set them to -1 */
	savednumvals[optind] = opts[optind].numvals;
	opts[optind].numvals = -1;
	if (opts[optind].type == OPT_VAL_BOOL) {
	    /* clear bool options */
	    opts[optind].numvals = 0;
	    /* *((int *)opts[optind].vals) = 0; */
	}
    }
    for (argind = 1; argind < argc; argind++) {
	int bestmatchind = -1;
	unsigned int bestmatchlen = 0;
	opt_data * bestopt;
	char * currarg;
	char * valarg;

	currarg = argv[argind];

	if (strcmp(currarg, "--") == 0) {
	    /* empty option, indicates end of option list
	     * push rest of arguments to before first option */
	    for (argind++; argind < argc; argind++) {
		int shifter;
		currarg = argv[argind];
		for (shifter = argind; shifter > firstoptind; shifter--) {
		    argv[shifter] = argv[shifter-1];
		}
		argv[firstoptind] = currarg;
		firstoptind++;
	    }
	    /* and break out of arg loop */
	    break;
	}

	if (strncmp(currarg, "--", 2) == 0) {
	    currarg += 2;
	    currformat = OPT_FLAGS_FULL;
	} else if (currarg[0] == '-') {
	    currarg++;
	    currformat = OPT_FLAGS_STD;
	} else {
	    /* not an option, so place it before the first option,
	     * shifting args to make space */
	    int shifter;
	    for (shifter = argind; shifter > firstoptind; shifter--) {
		argv[shifter] = argv[shifter-1];
	    }
	    argv[firstoptind] = currarg;
	    firstoptind++;
	    continue;
	}

	/* blank out any equals sign for matching */
	if (currformat == OPT_FLAGS_FULL) {
	    equalloc = strchr(currarg, '=');
	    if (equalloc) {
		equalloc[0] = '\0';
	    }
	}

	ambiguity = 0;
	for (optind = 0; optind < numopts; optind++) {
	    opt_data * curropt = &opts[optind];
	    
	    if ((curropt->flags & currformat & OPT_FLAGS_MASK) == 0) {
		continue;
	    }

	    if (currformat & OPT_FLAGS_STD) {
		if (strncmp(curropt->name, currarg, strlen(curropt->name)) == 0) {
		    bestmatchind = optind;
		}
	    } else {
		if (strstr(curropt->name, currarg) == curropt->name) {
		    if (strncmp(curropt->name, currarg, strlen(curropt->name)) == 0 &&
			(currarg[strlen(curropt->name)] == '=' ||
			 currarg[strlen(curropt->name)] == '\0')) {
			bestmatchind = optind;
			bestmatchlen = strlen(currarg);
			break;
		    } else if (bestmatchlen < strlen(currarg)) {
			bestmatchind = optind;
			bestmatchlen = strlen(currarg);
		    } else {
			ambiguity++;
		    }
		}
	    }
	}
	if (optind == numopts && ambiguity) { /* no exact match, ambiguous */
	    err.optind = optind;
	    err.arg = currarg;
	    err.val = NULL;
	    opt_amb_err(&err);
	}
	if (strcmp(currarg, "help") == 0 &&
	    (bestmatchind == -1 ||
	     strcmp(opts[bestmatchind].name, "help") != 0)) {
	    opt_usage(numopts, opts);
	    exit(-1);
	}
	if (bestmatchind == -1) {
	    /* looks like an option, but isn't */
	    err.optind = -1;
	    err.arg = currarg;
	    err.val = NULL;
	    opt_err(&err);
	}

	bestopt = &opts[bestmatchind];

	/* currarg is now potentially the name of an option
	 * check to see if this argument includes a value */
	valarg = NULL;
	if ((currformat & OPT_FLAGS_STD) &&
	    currarg[strlen(bestopt->name)] != '\0') {
	    valarg = &currarg[strlen(bestopt->name)];
	} else if (currformat & OPT_FLAGS_FULL && equalloc) {
	    /* value is after the equals sign */
	    valarg = equalloc + 1;
	}

	if (bestopt->type == OPT_VAL_BOOL) {
	    if (valarg) {
		err.arg = currarg;
		opt_bool_err(&err);
	    }
	    bestopt->numvals++;
	    (*((int *)bestopt->vals))++;
	    goto NEXTARG;
	}

	if (valarg == NULL) {
	    /* otherwise, the following arg may have a value */
	    if (argind+1 >= argc) {
		if (!(bestopt->flags & OPT_FLAGS_VAL_OPTIONAL)) {
		    /* value expected for option, but not enough args */
		    err.optind = bestmatchind;
		    err.arg = currarg;
		    err.val = "<none>";
		    opt_val_err(&err);
		} /* otherwise, no value */
	    } else {
		valarg = argv[argind+1];
		/* eat up the option value argument */
		argind++;
	    }
	}

	if (bestopt->numvals < 0) {
	    /* -1 means not specified, 0 means specified, but no values,
	     * Here we indicate that the option has at least been specified.
	     * it may be incremented below if there is a value argument */
	    bestopt->numvals = 0;
	}
	if (valarg == NULL) {
	    goto NEXTARG;
	}

	/* setup err structure for all future errors */
	err.optind = bestmatchind;
	err.arg = currarg;
	err.val = valarg;

	if (bestopt->numvals == savednumvals[bestmatchind]) {
	    if (bestopt->flags & OPT_FLAGS_NO_OVERFLOW ||
		savednumvals[bestmatchind] == 0) {
		/* no more space for extra values with this option */
		opt_space_warn(&err);
		goto NEXTARG;
	    } else {
		/* we're at capacity, so shift values and lose least recent */
		char * vals = (char *)bestopt->vals;
		size_t tsize = opt_type2size[bestopt->type];
		if (bestopt->type == OPT_VAL_STR)
		    free(((char **)bestopt->vals)[0]);
		memmove(vals, vals+(tsize*1), tsize*(bestopt->numvals-1));
		bestopt->numvals--;
	    }
	}

	errno = 0;
	switch (bestopt->type) {
	    case OPT_VAL_INT: {
		long val = strtol(valarg, leftoverp, 0);
		if (errno == ERANGE || val < INT_MIN || val > INT_MAX)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((int *)bestopt->vals)[bestopt->numvals] = (int) val;
		bestopt->numvals++;
		break;
	    }
	    case OPT_VAL_UINT: {
		unsigned long val = strtoul(valarg, leftoverp, 0);
		if (errno == ERANGE || val > UINT_MAX)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((unsigned int *)bestopt->vals)[bestopt->numvals] =
		    (unsigned int) val;
		bestopt->numvals++;
		break;
	    }
	    case OPT_VAL_LONG: {
		long val = strtol(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((long *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
	    case OPT_VAL_ULONG: {
		unsigned long val = strtoul(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((unsigned long *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
#ifdef HAVE_STRTOLL
	    case OPT_VAL_LONGLONG: {
		long long val = strtoll(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((long long *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
#endif
#ifdef HAVE_STRTOULL
	    case OPT_VAL_ULONGLONG: {
		unsigned long long val = strtoull(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((unsigned long long *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
#endif
#ifdef HAVE_STRTOQ
	    case OPT_VAL_QUADT: {
		quad_t val = strtoq(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((quad_t *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
#endif
#ifdef HAVE_STRTOUQ
	    case OPT_VAL_UQUADT: {
		u_quad_t val = strtouq(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((u_quad_t *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
#endif
	    case OPT_VAL_SIZE: {
		OPT_SIZE_TMPTYPE val = OPT_SIZE_STRTO(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((OPT_SIZE_TMPTYPE *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
	    case OPT_VAL_TIME: {
		OPT_TIME_TMPTYPE val = OPT_TIME_STRTO(valarg, leftoverp, 0);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((OPT_TIME_TMPTYPE *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
	    case OPT_VAL_FLOAT: {
	        double val = strtod(valarg, leftoverp);
		if (errno == ERANGE || val > MAXFLOAT)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((float *)bestopt->vals)[bestopt->numvals] = (float) val;
		bestopt->numvals++;
		break;
	    }
	    case OPT_VAL_DOUBLE: {
		double val = strtod(valarg, leftoverp);
		if (errno == ERANGE)
		    opt_range_err(&err);
		else if (**leftoverp != '\0')
		    opt_val_err(&err);
		((double *)bestopt->vals)[bestopt->numvals] = val;
		bestopt->numvals++;
		break;
	    }
	    case OPT_VAL_STR: {
		((char **)bestopt->vals)[bestopt->numvals] = strdup(valarg);
		bestopt->numvals++;
		break;
	    }
	    default:
		opt_int_err(&err);
		break;
	}
    NEXTARG:
	if (equalloc)
	    equalloc[0] = '=';
    }
    
    if (print) {
	/* print canonicalized list of arguments and current options */
	FILE * outfp = stdout;

	fprintf(outfp, "Specified options:\n");

	/* print executable name */
/* 	fprintf(outfp, "%s", argv[0]); */

	/* print options */
	for (optind = 0; optind < numopts; optind++) {
	    opt_data * curropt = &opts[optind];
	    int valind;

	    if (curropt->numvals == -1 || curropt->type == OPT_VAL_NONE) {
		continue;
	    }

	    if (curropt->numvals == 0) {
		if (curropt->type == OPT_VAL_BOOL)
		    continue;
		if (curropt->flags & OPT_FLAGS_FULL)
		    fprintf(outfp, " --%s", curropt->name);
		else
		    fprintf(outfp, " -%c", curropt->name[0]);
		continue;
	    }
	    for (valind = 0; valind < curropt->numvals; valind++) {
		if (curropt->flags & OPT_FLAGS_FULL) {
		    fprintf(outfp, " --%s", curropt->name);
		} else {
		    fprintf(outfp, " -%s", curropt->name);
		}

		switch (curropt->type) {
		    case OPT_VAL_BOOL:
			break;
		    case OPT_VAL_INT:
			fprintf(outfp, " %d",
				((int *)curropt->vals)[valind]);
			break;
		    case OPT_VAL_UINT:
			fprintf(outfp, " %u",
				((unsigned int *)curropt->vals)[valind]);
			break;
		    case OPT_VAL_LONG:
			fprintf(outfp, " %ld",
				((long *)curropt->vals)[valind]);
			break;
		    case OPT_VAL_ULONG:
			fprintf(outfp, " %lu",
				((unsigned long *)curropt->vals)[valind]);
			break;
#ifdef HAVE_STRTOLL
		    case OPT_VAL_LONGLONG:
			fprintf(outfp, " %lld",
				((long long *)curropt->vals)[valind]);
			break;
#endif
#ifdef HAVE_STRTOULL
		    case OPT_VAL_ULONGLONG:
			fprintf(outfp, " %llu",
				((unsigned long long *)curropt->vals)[valind]);
			break;
#endif
#ifdef HAVE_STRTOQ
		    case OPT_VAL_QUADT:
			fprintf(outfp, " %qd",
				((quad_t *)curropt->vals)[valind]);
			break;
#endif
#ifdef HAVE_STRTOUQ
		    case OPT_VAL_UQUADT:
			fprintf(outfp, " %qu",
				((u_quad_t *)curropt->vals)[valind]);
			break;
#endif
		    case OPT_VAL_SIZE:
			fprintf(outfp, " "OPT_SIZE_PRINTF,
				((OPT_SIZE_TMPTYPE *)curropt->vals)[valind]);
			break;
		    case OPT_VAL_TIME:
			fprintf(outfp, " "OPT_TIME_PRINTF,
				((OPT_TIME_TMPTYPE *)curropt->vals)[valind]);
			break;
		    case OPT_VAL_FLOAT:
		    case OPT_VAL_DOUBLE:
			fprintf(outfp, " %f",
				((double *)curropt->vals)[valind]);
			break;
		    case OPT_VAL_STR:
			fprintf(outfp, " %s",
				((char **)curropt->vals)[valind]);
			break;
		    default:
			break;
		}
	    }
	    fprintf(outfp, "\n");
	}

	/* print non-option arguments */
	fprintf(outfp, "Non-option arguments: ");
	for (argind = 1; argind < firstoptind; argind++) {
	    fprintf(outfp, " %s", argv[argind]);
	}

	fprintf(outfp, "\n"); /* end with a newline */
    }

    fflush(stdout);
    fflush(stderr);

    free(savedvals);
    free(savednumvals);
    
    return argc - firstoptind;
}

#undef FUNC
#define FUNC "opt_parsefile"
/**
 * Parse command-line options from a file.
 * 
 * @param configfile file to be parsed
 * @param numopts number of options in #opts array
 * @param opts option array
 * @param print print out specified options
 *
 * Same behavior as opt_parse(), but options come from a file
 * containing one option per line.
 * All options are assumed to be long options with the leading
 * '--' omitted, and argument (if any) follows the option
 * name separated by whitespace.
 * Arguments that contain whitespace or other special characters may
 * be protected by single or double quotes, and are parsed in the same
 * manner as most UNIX shells.
 * Any text including and following an unquoted '#' character is
 * considered a comment, and is ignored.
 * Returns: number of "tokens" in file successfully parsed as options.
 */
int
opt_parsefile(const char * configfile,
	      int numopts, opt_data * opts,
	      int print)
{
    int fnum, numfiles;
    char **files;
    int argc;
    char **argv;
    unsigned int bufsize, bufend;
    char *buf, *argbuf, *newarg;
    const char * cset; /* character set used as delimiter for argument parts */
    const char * normset = "#\t\v\r\f \"\'"; /* delimiter of non-quoted expr */
    const char * insquoteset = "\'"; /* delimiter inside single-quoted expr */
    const char * indquoteset = "\""; /* delimiter inside double-quoted expr */
    int returnval;

    numfiles = 1;
    files = (char **)malloc(sizeof(char *)*numfiles);
    files[0] = strdup(configfile);

    argc = 1;
    argv = (char **)malloc(sizeof(char *)*argc);
    argv[0] = "";

    bufsize = 8192;
    buf = (char *)malloc(sizeof(char)*bufsize);
    argbuf = (char *)malloc(sizeof(char)*bufsize);
    buf[bufsize-1] = '\0';

    for (fnum = 0; fnum < numfiles; fnum++) {
	unsigned int within = 0; /* number of .includes seen in this file */
	FILE * fp = fopen(files[fnum], "r");
	if (fp == NULL) {
	    fprintf(stderr, "Error opening config file %s\n",
		    (const char *)files[fnum]);
	    perror("fopen");
	    exit(-1);
	}

	bufend = 0;
	while ((bufend += fread(&buf[bufend], sizeof(char), bufsize-1-bufend, fp)) != 0) {
	    char *eol, *curpos, *argpos;
	    char *opt = NULL;
	    int ignorerest = 0;
	    size_t len;
	    int foundarg = 0;

	    buf[bufend] = '\0';
	    eol = strchr(buf, '\n');
	    if (eol == NULL && !feof(fp)) {
		bufsize *= 2;
		buf = (char *)realloc(buf, sizeof(char)*bufsize);
		argbuf = (char *)realloc(argbuf, sizeof(char)*bufsize);
		buf[bufsize-1] = '\0';
		continue;
	    }
	    if (eol)
		*eol = '\0';
	    argbuf[0] = '\0';

	    curpos = &buf[0];
	    argpos = &argbuf[0];

	    curpos += strspn(curpos,"\t\v\r\f "); /* ignore leading space */

	    /* option name (if any) ends with whitespace or comment chars */
	    len = strcspn(curpos, "#\t\n\v\r\f ");
	    if (len == 0) goto eatline;
	    ignorerest = (curpos[len] == '#');
	    curpos[len] = '\0';
	    opt = (char *)malloc(sizeof(char)*(strlen(curpos)+1));
	    strcpy(opt, curpos);
	    if (curpos + len == eol)
		curpos += len;
	    else
		curpos += len + 1; /* go past the "delimiter" */
	    curpos += strspn(curpos, "\t\n\v\r\f "); /* ignore trailing ws */

	    /* Now parse the argument.  The while loop below allows
	     * parts of the argument to be protected/unprotected, e.g.:
	     *     optionname    arg"ument h"as' four 'words
	     * is more complicated than, but equivalent to:
	     *     optionname    "argument has four words"
	     */
	    ignorerest = (*curpos == '#');
	    cset = normset;
	    while (!ignorerest && *curpos != '\0') {
		len = strcspn(curpos, cset);
		strncpy(argpos, curpos, len);
		foundarg = 1;
		argpos += len;
		curpos += len;
		ignorerest = (*curpos == '#');
		switch (*curpos) {
		case '\"':
		    if (cset == indquoteset)
			cset = normset;
		    else
			cset = indquoteset;
		    break;
		case '\'':
		    if (cset == insquoteset)
			cset = normset;
		    else
			cset = insquoteset;
		    break;
		}
		if (*curpos != '\0')
		    curpos++;
	    }
	    *argpos = '\0';

	    /* parse the special option ".include" */
	    if (strcmp(opt, ".include") == 0) {
		unsigned int newpos = fnum + within + 1;
		if (argbuf[0] == '\0') {
		    fprintf(stderr,
			    ".include in config file has no argument!\n");
		    exit(-1);
		}
		within++;
		numfiles++;
		files = (char **)realloc(files, sizeof(char *)*numfiles);
		/* move things one to the right */
		memmove(files+newpos+1, files+newpos, sizeof(files[0])*(numfiles-newpos-1));
		files[newpos] = strdup(argbuf);
		free(opt);
		goto eatline;
	    }

	    /* add option (and argument if any) to argv array */
	    newarg = (char *)malloc(sizeof(char)*(strlen(opt)+strlen(argbuf)+4));
	    strcpy(newarg, "--");
	    strcat(newarg, opt);
	    if (foundarg || argbuf[0] != '\0') {
		strcat(newarg, "=");
		strcat(newarg, argbuf);
	    }
	    free(opt);
	    argc++;
	    argv = (char **)realloc(argv, sizeof(char *)*argc);
	    argv[argc-1] = newarg;

	eatline:
	    if (eol) {
		bufend -= eol + 1 - &buf[0];
		memmove(&buf[0], eol+1, bufend);
	    } else {
		bufend = 0;
	    }
	}
	fclose(fp);
    }

    returnval = opt_parse(argc, argv, numopts, opts, print);

    /* clean up */
    for (argc--; argc > 0; argc--) {
	free(argv[argc]);
    }
    free(argv);
    free(buf);
    free(argbuf);
    for (numfiles--; numfiles >= 0; numfiles--) {
	free(files[numfiles]);
    }
    free(files);
    
    return returnval;
}
