/* $Id: opts.h,v 1.6 2004-05-27 22:02:30 gadde Exp $ */

/* opts.h --
 *
 * A module to parse command-line options
 */

/* Copyright (c) 1997-1999 Syam Gadde and Duke University */

#ifndef _OPTS_H
#define _OPTS_H

#include <bxh_config.h>

#include <stdlib.h>

/** \defgroup opts Parse command line options */
/*@{*/

typedef enum {
    OPT_VAL_NONE = 0,
    OPT_VAL_BOOL,
    OPT_VAL_INT,
    OPT_VAL_UINT,
    OPT_VAL_LONG,
    OPT_VAL_ULONG,
#ifdef HAVE_STRTOLL
    OPT_VAL_LONGLONG,
#endif
#ifdef HAVE_STRTOULL
    OPT_VAL_ULONGLONG,
#endif
#ifdef HAVE_STRTOQ
    OPT_VAL_QUADT,
#endif
#ifdef HAVE_STRTOUQ
    OPT_VAL_UQUADT,
#endif
    OPT_VAL_SIZE,
    OPT_VAL_TIME,
    OPT_VAL_FLOAT,
    OPT_VAL_DOUBLE,
    OPT_VAL_STR
} opt_val_type;

extern char * opt_type2str[];

#define OPT_FLAGS_STD		0x0001
#define OPT_FLAGS_FULL		0x0002
#define OPT_FLAGS_VAL_OPTIONAL	0x0004
#define OPT_FLAGS_NO_OVERFLOW	0x0008

#define OPT_FLAGS_MASK (OPT_FLAGS_STD|OPT_FLAGS_FULL)

typedef struct opt_data_i {
    /**
     * Attributes for this option.
     *  - OPT_FLAGS_STD
     *    - Option will be specified as "-Cval" or "-C val",
     *      where 'C' is a string
     *  - OPT_FLAGS_FULL
     *    - Option will be specified as "--opt=val", or "--opt val",
     *      where 'opt' is a string
     *  - OPT_FLAGS_VAL_OPTIONAL
     *    - though a value type may be specified, the value is optional;
     *      also see #numvals below
     *  - OPT_FLAGS_NO_OVERFLOW
     *    - Normally, if an option has been specified more than #numvals
     *      times, only the arguments of the last #numvals instances are
     *      kept; however, specifying #OPT_FLAGS_NO_OVERFLOW enforces a
     *      maximum of "numvals" instances.
     */
    unsigned int flags;

    /** type of the value (if any) associated with the option */
    opt_val_type type;

    /**
     * Pointer to storage for parsed option arguments.
     * #vals should point to an array big enough to store
     * #numvals values of type #type.
     * If #type is #OPT_VAL_BOOL, this should be a pointer
     * to an int, and on return from opt_parse() will contain
     * the number of times the option was specified.
     */
    void * vals;

    /**
     * The maximum number of values you can store in #vals or, after a
     * call to opt_parse(), the number of valid values stored in #vals.
     * This maximum is not enforced unless #OPT_FLAGS_NO_OVERFLOW is
     * specified.\n
     * On return from opt_parse(), if the option was not specified,
     * this will have -1.  If the option was specified, but without
     * a value (i.e. #OPT_FLAGS_VAL_OPTIONAL), this will have 0.
     * Otherwise, it will indicate the number of elements in #vals
     * that were filled in.\n
     * If #type is #OPT_VAL_BOOL, #vals is assumed to point to a
     * a single integer, and on return from opt_parse() #numvals will
     * indicate how many times the option was specified (i.e. will
     * have the same value as <tt>*(int *)#vals</tt>.
     */
    int numvals;

    /** the option name (one character long for #OPT_FLAGS_STD) */
    const char * name;

    /** the option description */
    const char * descr;
} opt_data;

typedef struct {
    int numopts;
    opt_data * opts;
    int optind;
    char * arg;
    char * val;
} opt_err_data;

#if defined(__cplusplus)
extern "C" {
#endif
    
extern void (*opt_ext_usage)();
void opt_usage(int numopts, opt_data * opts);
int opt_parse(int argc, char **argv, int numopts, opt_data * opts, int print);
int opt_parsefile(const char * configfile, int numopts, opt_data * opts, int print);

#if defined(__cplusplus)
}
#endif

/*@}*/

#endif /* #ifndef _OPTS_H */
