static const char rcsid[] = "$Id: bxh2pgm.cpp,v 1.22 2009-02-18 15:16:42 gadde Exp $";

/*
 * bxh2pgm.cpp --
 * 
 *  Create a PGM file containing 2-D images from an N-D BXH file.
 *  This is designed to produce images for display or other image
 *  conversion tools, not for further data processing, because data
 *  values may be rescaled and gamma corrected.
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>
#include <string.h>

#include <vector>

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#define CONVERTTEMPLATE(inbuf, fromtype, bufsize, retbuf, totype) {	\
    fromtype * buf = NULL;						\
    fromtype * endbuf = (fromtype *)((char *)inbuf + (bufsize));	\
    size_t retsize = sizeof(totype)*((bufsize)/sizeof(*buf));		\
    totype * newbuf = NULL;						\
    newbuf = (totype *)malloc(retsize);					\
    (retbuf) = newbuf;							\
    if ((newbuf) == NULL) {						\
	fprintf(stderr, "Error allocating %d bytes\n", retsize);	\
    }									\
    for (buf = (fromtype *)(inbuf); buf < (endbuf); newbuf++, buf++) {	\
	*(newbuf) = (totype)*buf;					\
    }									\
}

static float *
convertBufToFloat(const void * inbuf, size_t bufsize, const char * elemtype)
{
    float * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "float32") == 0) {
	retbuf = (float *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    }
    return retbuf;
}

static
int
writePGM(FILE * fp, unsigned int width, unsigned int length, unsigned int maxval, unsigned short * imptr)
{
    unsigned int numpixels = width * length;
    unsigned int pixelnum;
    fprintf(fp, "P5 %u %u %u\n", width, length, maxval);
    if (maxval < 256) {
	/* inelegant but true */
	for (pixelnum = 0; pixelnum < numpixels; pixelnum++) {
	    ((unsigned char *)imptr)[pixelnum] = (unsigned char)imptr[pixelnum];
	}
	if (fwrite(imptr, sizeof(unsigned char)*numpixels, 1, fp) != 1) {
	    fprintf(stderr, "Error writing to file.\n");
	    return -1;
	}
    } else {
	int msbfirst = 1;
	msbfirst = (((char *)&msbfirst)[0] == 0);
	if (!msbfirst) {
	    unsigned short * ptr;
	    unsigned short * endptr = imptr + numpixels;
	    char swap;
	    for (ptr = imptr; ptr < endptr; ptr++) {
		swap = ((char *)ptr)[0];
		((char *)ptr)[0] = ((char *)ptr)[1];
		((char *)ptr)[1] = swap;
	    }
	}
	if (fwrite(imptr, sizeof(unsigned short)*numpixels, 1, fp) != 1) {
	    fprintf(stderr, "Error writing to file.\n");
	    return -1;
	}
    }
    return 0;
}


int
main(int argc, char *argv[])
{
    struct stat statbuf;
    const char * inputfile = NULL;
    const char * outputfile = NULL;
    int dimnum;
    int msbfirst = 1;
    FILE * fp = NULL;
    double opt_maxinval = HUGE_VAL;
    double opt_mininval = -HUGE_VAL;
    const char * opt_colorbar = NULL;
    const char * opt_colorbarorient = "horizontal";
    unsigned int opt_barwidth = 16;
    unsigned int opt_barlength = 256;
    const char * opt_select[4] = {":", ":", ":", ":"};
    const char * opt_dimorder = "x,y,z,t";
    int opt_version = 0;
    const char ** ordereddimnames = NULL;
    int numorddims = 0;
    msbfirst = (((char *)&msbfirst)[0] == 0);

    const int numopts = 14;
    opt_data myOpts[14] = { 
      { 0x0, OPT_VAL_NONE, NULL, 0, "",
	"Usage:\n"
	"  bxh2pgm input.bxh output.pgm\n\n"
	"This program converts images wrapped with a BXH or XCEDE header "
	"into PGM format.  3-D or higher dimensionality images are "
	"represented as a sequence of 2-D images." },
      { 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
      { OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	"Print version string and exit." },
      { OPT_FLAGS_FULL, OPT_VAL_STR, &opt_colorbar, 1, "colorbar",
	"Write a horizontal colormap 'bar' to this PGM file."
      },
      { OPT_FLAGS_FULL, OPT_VAL_STR, &opt_colorbarorient, 1, "colorbarorient",
	"Orientation of colorbar, either 'horizontal' (default) or 'vertical'."
      },
      { OPT_FLAGS_FULL, OPT_VAL_UINT, &opt_barwidth, 1, "barwidth",
	"Width (in pixels) of colormap 'bar' (default 16)."
      },
      { OPT_FLAGS_FULL, OPT_VAL_UINT, &opt_barlength, 1, "barlength",
	"Length (in pixels) of colormap 'bar' (default 256)."
      },
      { OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_maxinval, 1, "maxval",
	"By default, if the input element type is floating-point or "
	"if the maximum input value is greater than 65535, "
	"the maximum value in the input will be mapped to "
	"65535 (the highest possible PGM value) in the output PGM image.  "
	"--maxval specifies an alternative maximum input value.  "
	"Input values greater than this will be clipped."
      },
      { OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_mininval, 1, "minval",
	"By default, the minimum value in the input will be mapped to "
	"0 (the lowest possible PGM value) in the output PGM image.  "
	"--minval specifies an alternative minimum input value.  "
	"Input values smaller than this will be clipped."
      },
      { OPT_FLAGS_FULL, OPT_VAL_STR, &opt_dimorder, 1, "dimorder",
	"Specify dimension order as a comma-separated list of dimension names."
      },
      { OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	"Comma-separated list of timepoints to use (first timepoint is 0).  "
	"Any timepoint can be a contiguous range, specified as two "
	"numbers separated by a colon, i.e. 'START:END'.  "
	"An empty END implies the last timepoint.  "
	"The default step of 1 (one) in ranges can be changed using "
	"'START:STEP:END', which is equivalent to "
	"'START,START+STEP,START+(2*STEP),...,END'." },
      { OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	"Just like timeselect, but for the 'x' dimension." },
      { OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	"Just like timeselect, but for the 'y' dimension." },
      { OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	"Just like timeselect, but for the 'z' dimension." }
    }; 

    argc -= opt_parse(argc, argv, numopts, &myOpts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 3) {
	opt_usage(numopts, &myOpts[0]);
	return -1;
    }

    if (strcmp(opt_colorbarorient, "horizontal") != 0 &&
	strcmp(opt_colorbarorient, "vertical") != 0) {
	fprintf(stderr, "colorbar orientation must be 'horizontal' or 'vertical'\n");
	return -1;
    }

    inputfile = argv[1];
    outputfile = argv[2];
    if (stat(outputfile, &statbuf) == 0) {
	fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputfile);
	return -1;
    }

    ordereddimnames = NULL;
    /* parse dimorder */
    {
	char * temporder = strdup(opt_dimorder);
	char * curpos = temporder;
	numorddims = 0;
	while (*curpos != '\0') {
	    size_t valuelen = 0;
	    char * comma = NULL;
	    comma = strchr(curpos, ',');
	    if (comma) {
		valuelen = comma - curpos;
		*comma = '\0';
	    } else {
		valuelen = strlen(curpos);
	    }
	    ordereddimnames = (const char **)realloc(ordereddimnames, sizeof(char *)*(numorddims+1));
	    ordereddimnames[numorddims] = strdup(curpos);
	    numorddims++;
	    if (comma) {
		*comma = ',';
		curpos = comma + 1;
	    } else {
		curpos += valuelen;
	    }
	}
	free(temporder);
    }

    struct bxhdataread bdr;

    memset(&bdr, '\0', sizeof(bdr));

    if (bxh_dataReadFileStart(inputfile, "image", NULL, numorddims, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", argv[1]);
	return -1;
    }
    if (bdr.datarec->numdims < 2) {
	fprintf(stderr, "Data must be at least 2-dimensional.\n");
	return -1;
    }
    if (bxh_dataReadFinish(&bdr, "float") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	return -1;
    }
    float * dataptr = (float *)bdr.dataptr;

    if ((fp = fopen(outputfile, "wb")) == NULL) {
	fprintf(stderr, "Error opening file %s\n", outputfile);
	return -1;
    }

    /* calculate maxval */
    double maxinval = 0;
    double mininval = 0;
    float offset = 0;
    float inscale = 0;
    float outscale = 0;
    size_t voxelnum = 0;
    for (voxelnum = 0; voxelnum < bdr.pagesizes[bdr.datarec->numdims-1]; voxelnum++) {
	if (voxelnum == 0 || dataptr[voxelnum] > maxinval) {
	    maxinval = dataptr[voxelnum];
	}
	if (voxelnum == 0 || dataptr[voxelnum] < mininval) {
	    mininval = dataptr[voxelnum];
	}
	if (opt_maxinval != HUGE_VAL) {
	    if (dataptr[voxelnum] > opt_maxinval) {
		dataptr[voxelnum] = (float)opt_maxinval;
	    }
	}
	if (opt_mininval != -HUGE_VAL) {
	    if (dataptr[voxelnum] < opt_mininval) {
		dataptr[voxelnum] = (float)opt_mininval;
	    }
	}
    }
    if (opt_maxinval == HUGE_VAL) {
	opt_maxinval = maxinval;
    }
    if (opt_mininval == -HUGE_VAL) {
	opt_mininval = mininval;
    }
    /* min/max may have been clipped; if so, fix */
    if (mininval < opt_mininval) {
	mininval = opt_mininval;
    }
    if (maxinval > opt_maxinval) {
	maxinval = opt_maxinval;
    }
    /* At this point, mininval->maxinval is the actual range of the
     * input data (after clipping).
     * opt_mininval->opt_maxinval is the input range (same size or
     * larger than actual input range) that will be mapped to
     * 0->outscale (determined below) */

    /* Set up rescale to 16-bit space */
    offset = (float)(-1 * opt_mininval);
    inscale = (float)(opt_maxinval + offset);
    outscale = (float)(opt_maxinval + offset);
    if (strncmp(bdr.datarec->elemtype, "float", 5) == 0 ||
	strcmp(bdr.datarec->elemtype, "double") == 0) {
	outscale = 65535;
    } else {
	if (opt_maxinval + offset > 65535) {
	    outscale = 65535;
	}
    }
    /* inscale is the "size" of the input range.
     * outscale is the "size" of the output range. */
    /* set up colormap */

    /*** Write PGM file ***/
    size_t imagenum = 0;
    for (imagenum = 0; imagenum < bdr.pagesizes[bdr.datarec->numdims-1]/bdr.pagesizes[1]; imagenum++) {
	size_t voxelnum = 0;
	unsigned short * imptr = NULL;
	imptr = (unsigned short *)malloc(sizeof(unsigned short)*bdr.pagesizes[1]);

	/* rescale to 16-bit space, doing gamma correction at same time */
	for (voxelnum = 0; voxelnum < bdr.pagesizes[1]; voxelnum++) {
	    float val = dataptr[(imagenum * bdr.pagesizes[1]) + voxelnum];
	    val = (val + offset) / inscale; /* [0.0,1.0] */
	    if (val <= 0.018) {
		val = (float)(val * 4.5);
	    } else {
		val = (float)(1.099 * pow((double)val, 0.45) - 0.099);
	    }
	    val *= outscale;
	    imptr[voxelnum] = (unsigned short)val;
	}
	if (writePGM(fp, bdr.dimsizes[0], bdr.dimsizes[1], (unsigned int)outscale, imptr) == -1) {
	    fprintf(stderr, "Error writing PGM image.\n");
	    return -1;
	}
	free(imptr);
    }
    fclose(fp);
    fp = NULL;

    /*** Write PGM colormap bar ***/
    if (opt_colorbar) {
	unsigned short * imptr = NULL;
	unsigned int x, y;
	unsigned int mininvalbarpos, maxinvalbarpos;
	unsigned int br1, br2;
	int horiz = 1;
	if (strcmp(opt_colorbarorient, "vertical") == 0) {
	    horiz = 0;
	}
	if ((fp = fopen(opt_colorbar, "wb")) == NULL) {
	    fprintf(stderr, "Error opening file %s\n", opt_colorbar);
	    return -1;
	}

	/* edges of bracket */
	mininvalbarpos = (unsigned short)(((mininval + offset) / inscale) * opt_barlength);
	maxinvalbarpos = (unsigned short)(((maxinval + offset) / inscale) * opt_barlength);
	br1 = opt_barwidth / 16;
	br2 = opt_barwidth - 1 - br1;

	imptr = (unsigned short *)malloc(sizeof(unsigned short) * opt_barwidth * opt_barlength);
	for (y = 0; y < opt_barlength; y++) {
	    /* calculate gamma corrected map values */
	    float val = (float)y / opt_barlength; /* [0.0,1.0] */
	    float val2 = val;
	    /* gamma correct */
	    if (val <= 0.018) {
		val = (float)(val * 4.5);
	    } else {
		val = (float)(1.099 * pow((double)val, 0.45) - 0.099);
	    }
	    val *= outscale;
	    if (val2 <= 0.018) {
		val2 = (float)(val2 * 4.5);
	    } else {
		val2 = (float)(1.099 * pow((double)val2, 0.45) - 0.099);
	    }
	    val2 *= outscale;
	    for (x = 0; x < opt_barwidth; x++) {
		unsigned int impos;
		if (horiz) {
		    impos = (x * opt_barlength * 3) + (y * 3);
		} else {
		    impos = (y * opt_barwidth * 3) + (x * 3);
		}
		if (y == mininvalbarpos || y == maxinvalbarpos ||
		    (y > mininvalbarpos && y < maxinvalbarpos &&
		     (x <= br1 || x >= br2))) {
		    imptr[impos] = (unsigned short)val2;
		} else {
		    imptr[impos] = (unsigned short)val;
		}
	    }
	}
	{
	    unsigned int pgmxsize, pgmysize;
	    if (horiz) {
		pgmxsize = opt_barlength;
		pgmysize = opt_barwidth;
	    } else {
		pgmysize = opt_barlength;
		pgmxsize = opt_barwidth;
	    }
	    if (writePGM(fp, pgmxsize, pgmysize, (unsigned int)outscale, imptr) == -1) {
		fprintf(stderr, "Error writing PGM image.\n");
		return -1;
	    }
	}
	free(imptr);
	fclose(fp);
    }

    bxh_datareaddata_free(&bdr);

    return 0;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.21  2009/02/17 18:32:02  gadde
 * Free bdr.
 *
 * Revision 1.20  2009/02/17 14:58:24  gadde
 * Move to dataread API
 *
 * Revision 1.19  2008/03/07 23:05:08  gadde
 * Stop using off_t for signed data
 *
 * Revision 1.18  2006/06/01 20:16:49  gadde
 * const fixes
 *
 * Revision 1.17  2005/09/20 18:37:54  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.16  2005/09/19 16:31:55  gadde
 * Documentation and help message updates.
 *
 * Revision 1.15  2005/09/14 14:49:29  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.14  2005/02/17 21:59:34  gadde
 * Fix option numbering bug.
 *
 * Revision 1.13  2004/11/15 16:04:27  gadde
 * Don't modify constant strings.
 *
 * Revision 1.12  2004/06/18 13:54:24  gadde
 * Add to usage message
 *
 * Revision 1.11  2004/06/03 21:47:12  gadde
 * minor fixes
 *
 * Revision 1.10  2004/05/14 18:12:47  gadde
 * Fix some min/max issues.
 *
 * Revision 1.9  2004/05/13 19:02:57  gadde
 * Increase width of bracket.
 *
 * Revision 1.8  2004/05/13 16:44:11  gadde
 * Add input range bracket to colorbar.
 *
 * Revision 1.7  2004/05/12 21:56:05  gadde
 * Allow vertical colormap.
 *
 * Revision 1.6  2004/04/28 21:34:38  gadde
 * Add return value for writePGM.
 *
 * Revision 1.5  2004/04/28 21:33:36  gadde
 * option name change: writemap => colorbar
 *
 * Revision 1.4  2004/04/28 21:18:04  gadde
 * Add colorbar option.
 *
 * Revision 1.3  2004/03/26 22:16:54  gadde
 * fix some AIX bugs
 *
 * Revision 1.2  2004/03/25 19:23:37  gadde
 * Add ppm conversion, and various other updates
 *
 * Revision 1.1  2004/03/24 22:53:09  gadde
 * Initial commit.
 *
 *
 */
