static const char rcsid[] = "$Id: bxh_absorb_dicom.cpp,v 1.1 2009-01-15 20:56:16 gadde Exp $";

/*
 * dicom2bxh.cpp --
 * 
 * Creates a BXH file based on given DICOM file.
 * Requires DCMTK.  Based on dcmdump.
 */

#include "osconfig.h"    /* make sure OS specific configuration is included first */

#include "bxh_convert.h"
#include "bxh_dicom.h"
#include "bxh_utils.h"
#include "opts.h"

#ifdef HAVE_GUSI_H
    /* needed for Macintosh */
#include <GUSI.h>
#include <SIOUX.h>
#endif

#include <dcxfer.h>
//#include <dcdebug.h>
#include <dcdict.h>
#include <ofstd.h>
#include <ofconsol.h>

#define SHORTCOL 3
#define LONGCOL 20

#ifdef __cplusplus
extern "C" {
#endif

static int opt_debug = 0;
static int opt_searchForOthers = 0;
static int opt_readDataset = 0;
static int opt_readXferAuto = 0;
static int opt_readXferLittle = 0;
static int opt_readXferBig = 0;
static int opt_readXferImplicit = 0;
static int opt_loadShort = 0;
static int opt_ignoreErrors = 0;
static int opt_forceConcat = 0;
static int opt_filenameSort = 0;
static int opt_noSort = 0;
static int opt_sortopt = 0;
static int opt_printfields = 0;
static int opt_cornerorigin = 0;

extern size_t opt_hintsize[4];
extern double opt_hintorigin[4];
extern double opt_hintspacing[4];
extern double opt_hintgap[4];
extern char * opt_forceorient;

static int
addopts_dicom2bxh_common(opt_data * optsin[], int numoptsin)
{
    const int numopts = 30;
    opt_data opts[30] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "general options:" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_debug, 0, "debug",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_debug, 0, "d",
	  "debug mode, print debug information" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "input options:" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_forceConcat, 0, "force-concat",
	  "If the input images have different orientation, Study UID, "
	  "Series UID, ImageType, etc., then this option nevertheless "
	  "forces them to be concatenated into the same volume.  "
	  "(They would otherwise be encapsulated within separate XML files.)  "
	  "This option may result in XML files that do not correctly "
	  "describe the DICOM data -- use only if you know what you're doing!"
	},
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_filenameSort, 0, "filename-sort",
	  "This program normally sorts input files by various fields in the "
	  "DICOM headers.  This option forces a sort by filename only.  This "
	  "can be useful in the case that the fields are unreliable."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_noSort, 0, "no-sort",
	  "This program normally sorts input files by various fields in the "
	  "DICOM headers.  This option disables sorting and relies on the "
	  "order in which files are provided on the command line.  This "
	  "can be useful in the case that the fields are unreliable."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_cornerorigin, 0, "cornerorigin",
	  "If specified, this assumes the origin specified in the DICOM file "
	  "comes (incorrectly) from the outer corner of the voxel rather than "
	  "the center.  This is then corrected for further processing.  This "
	  "is the case for older GE DICOM files."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_printfields, 0, "printfields",
	  "Print out DICOM file name and various fields from that file, "
	  "one line for each DICOM file.  Fields will be printed as "
	  "NAME=VALUE, with VALUE protected with single-quotes if it contains "
	  "whitespace.  Newlines, backslashes, and single quotes will be "
	  "escaped as \\n, \\\\, and \\', and non-printable characters will "
	  "be ignored."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "   input file format:" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_searchForOthers, 0, "search-for-others",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_searchForOthers, 0, "s",
	  "search for matching files in the same directory" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_readDataset, 0, "read-dataset",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_readDataset, 0, "f",
	  "read data set without file meta information" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  " input transfer syntax (only with --read-dataset):" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_readXferAuto, 0, "read-xfer-auto",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_readXferAuto, 0, "t",
	  "use TS recognition (default)" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_readXferLittle, 0, "read-xfer-little",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_readXferLittle, 0, "te",
	  "read with explicit VR little endian TS" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_readXferBig, 0, "read-xfer-big",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_readXferBig, 0, "tb",
	  "read with explicit VR big endian TS" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_readXferImplicit, 0, "read-xfer-implicit",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_readXferImplicit, 0, "ti",
	  "read with implicit VR little endian TS" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "output options:" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  " converting:" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_loadShort, 0, "load-short",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_loadShort, 0, "M",
	  "do not load very long values (e.g. pixel data)" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  " error handling:" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_ignoreErrors, 0, "ignore-errors",
	  NULL },
	{ OPT_FLAGS_STD, OPT_VAL_BOOL, &opt_ignoreErrors, 0, "E",
	  "attempt to convert even if file is damaged" }
    };
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts));
    memcpy(&(*optsin)[numoptsin], &opts[0], sizeof(opt_data)*numopts);
    return numopts;
}

int
addopts_bxhabsorb_dicom2bxh(opt_data * optsin[], int numoptsin)
{
    const int numopts = 3;
    opt_data opts[3] = {
        { 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "DICOM USAGE " },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "  bxhabsorb --fromtype dicom [opts] [dicomfiles...] output.bxh" }
    };
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts));
    memcpy(&(*optsin)[numoptsin], &opts[0], sizeof(opt_data)*numopts);
    return numopts + addopts_dicom2bxh_common(optsin, numoptsin + numopts);
}

int
addopts_dicom2bxh(opt_data * optsin[], int numoptsin)
{
    const int numopts = 2;
    opt_data opts[2] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  dicom2bxh [opts] [dicomfiles...] output.bxh\n\n"
	  "This program creates an XML wrapper for DICOM images." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" }
    };
    const int numopts2 = 1;
    opt_data opts2[1] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "additional options:" }
    };
    int newopts = 0;
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts));
    memcpy(&(*optsin)[numoptsin], &opts[0], sizeof(opt_data)*numopts);
    newopts = addopts_dicom2bxh_common(optsin, numoptsin + numopts);
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts+newopts+numopts2));
    memcpy(&(*optsin)[numoptsin+numopts+newopts], &opts2[0], sizeof(opt_data)*numopts2);
    return numopts + newopts + numopts2;
}

int
main_dicom2bxh(int argc, char *argv[])
{
    OFBool loadIntoMemory = OFTrue;
    OFBool showFullData = OFFalse;
    OFBool isDataset = OFFalse;
    E_TransferSyntax xfer = EXS_Unknown;
    OFBool stopOnErrors = OFTrue;
    OFBool searchForOthers = OFFalse;
    OFBool forceConcat = OFFalse;
    int sortopt = 0;
    int printfields = 0;
    int cornerorigin = 0;
    int numinfiles = 0;
    const char **infiles = NULL;
    const char *outfile = NULL;
    int paramnum;

#ifdef HAVE_GUSI_H
    /* needed for Macintosh */
    /* set options for the Metrowerks CodeWarrior SIOUX console */
    SIOUXSettings.autocloseonquit = OFFalse;
    SIOUXSettings.asktosaveonclose = OFFalse;
    SIOUXSettings.showstatusline = OFTrue;
    SIOUXSettings.setupmenus = OFTrue;
    /* set options for the GUSI sockets library */
    GUSISetup(GUSIwithSIOUXSockets);
    GUSISetup(GUSIwithInternetSockets);
#endif


    /* evaluate command line */
    if (opt_debug) {
      OFLog::configure(OFLogger::DEBUG_LOG_LEVEL);
    }

    if (opt_searchForOthers) searchForOthers = OFTrue;
    if (opt_forceConcat) forceConcat = OFTrue;
    if (opt_filenameSort && opt_noSort) {
      fprintf(stderr, "You cannot specify both --filename-sort and --no-sort options!\n");
      return 1;
    }
    if (opt_filenameSort) sortopt = 1;
    if (opt_noSort) sortopt = -1;
    if (opt_printfields) printfields = 1;
    if (opt_cornerorigin) cornerorigin = 1;
    if (opt_readDataset) isDataset = OFTrue;
    if ((opt_readXferAuto || opt_readXferLittle ||
	 opt_readXferBig || opt_readXferImplicit) &&
	!opt_readDataset) {
      fprintf(stderr, "--read-xfer-auto, --read-xfer-little, --read-xfer-big, and\n--read-xfer-implicit are only allowed when using --read-dataset!\n");
      exit(-1);
    }
    if (opt_readXferAuto)
      xfer = EXS_Unknown;
    if (opt_readXferLittle)
      xfer = EXS_LittleEndianExplicit;
    if (opt_readXferBig)
      xfer = EXS_BigEndianExplicit;
    if (opt_readXferImplicit)
      xfer = EXS_LittleEndianImplicit;

    if (opt_loadShort) loadIntoMemory = OFFalse;

    if (opt_ignoreErrors) stopOnErrors = OFFalse;

    /* make sure data dictionary is loaded */
    if (!dcmDataDict.isDictionaryLoaded())
    {
    CERR << "Warning: no data dictionary loaded, "
         << "check environment variable: "
         << DCM_DICT_ENVIRONMENT_VARIABLE
	 << std::endl;
    }

    if (argc < 3) {
	fprintf(stderr, "Not enough arguments! use\n %s --help\n", argv[0]);
	return 1;
    }

    /* params indexed starting at 1, and last param is output file */
    if (searchForOthers && argc > 3) {
	fprintf(stderr, "dicom2bxh: --search-for-others only accepts a single input file!\n");
	return 1;
    }
    for (paramnum = 1; paramnum < argc - 1; paramnum++) {
	infiles = (const char **)realloc(infiles, sizeof(char *)*(numinfiles+1));
	infiles[paramnum-1] = argv[paramnum];
	numinfiles++;
    }
    outfile = argv[argc-1];

    int errorCount = 0;
    errorCount += abs(createBXHFromDICOM(numinfiles, infiles, outfile, isDataset, xfer, showFullData, loadIntoMemory, stopOnErrors, searchForOthers, forceConcat, sortopt, printfields, cornerorigin, &opt_hintsize[0], &opt_hintorigin[0], &opt_hintspacing[0], &opt_hintgap[0], opt_forceorient, argv[0]));
    free(infiles); infiles = NULL;
    return errorCount;
}

#ifdef __cplusplus
};
#endif

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.46  2008/01/27 22:44:35  gadde
 * Fix memory leak.
 *
 * Revision 1.45  2006/10/27 13:56:06  gadde
 * Add --filename-sort option to dicom2bxh
 *
 * Revision 1.44  2006/03/23 17:36:51  gadde
 * Update for DCMTK 3.5.4.
 *
 * Revision 1.43  2005/09/19 16:31:58  gadde
 * Documentation and help message updates.
 *
 * Revision 1.42  2005/07/07 15:10:51  gadde
 * Memory fixes.
 *
 * Revision 1.41  2005/03/08 18:45:15  gadde
 * Some AIX fixes
 *
 * Revision 1.40  2005/03/08 18:09:57  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.39  2004/09/16 15:03:28  gadde
 * Add missing newline to error message.
 *
 * Revision 1.38  2004/07/06 18:34:17  gadde
 * Some memory fixes.
 *
 * Revision 1.37  2004/07/06 13:56:26  gadde
 * Some memory fixes.
 *
 * Revision 1.36  2004/06/15 18:42:34  gadde
 * Update fullifnames with ifnames.
 *
 * Revision 1.35  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.34  2004/05/17 17:54:11  gadde
 * Add force-concat option.
 *
 * Revision 1.33  2004/04/22 12:57:47  gadde
 * Remove extra extern "C".
 *
 * Revision 1.32  2004/03/12 16:29:04  gadde
 * Minor updates
 *
 * Revision 1.31  2004/02/24 18:01:07  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.30  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.29  2003/08/19 19:35:41  gadde
 * New options scheme, so that options are available using whichever form of bxhabsorb you use.
 *
 * Revision 1.28  2003/08/08 18:56:18  gadde
 * P670 updates
 *
 * Revision 1.27  2003/08/01 17:51:29  gadde
 * On the road to merging in xerces branch
 *
 * Revision 1.26  2003/06/18 16:14:14  gadde
 * win32 fixes
 *
 * Revision 1.25  2003/05/16 19:56:48  gadde
 * Require a basepath for datarec.
 *
 * Revision 1.24  2003/04/28 19:46:53  gadde
 * Move to catch-all BXH encapsulation program bxhabsorb -- dicom2bxh, pfile2bxh,
 * etc. will all be symbolic links to bxhabsorb.
 *
 * Revision 1.23  2003/03/31 17:51:36  gadde
 * Remove unparsed options.
 *
 * Revision 1.22  2003/02/12 21:07:19  gadde
 * Free some memory.
 *
 * Revision 1.21  2003/02/12 21:02:15  gadde
 * Added support for creating multiple BXH files from list of given
 * DICOM files if bxh_DICOM_readFiles doesn't use up all the files.
 * Also, added --search-for-others option which now must be specified
 * if you wish to search for other matching DICOM files in the same
 * directory.
 *
 * Revision 1.20  2003/01/17 16:16:17  gadde
 * Add version string to history
 *
 * Revision 1.19  2003/01/16 20:55:35  gadde
 * cosmetic change in history
 *
 * Revision 1.18  2003/01/06 18:41:57  gadde
 * Many changes/updates to comments.
 * Minor bug fixes.
 *
 * Revision 1.17  2002/11/25 16:21:31  gadde
 * Mega-update to merge in new library functions.
 *
 * Revision 1.16  2002/11/15 20:52:32  gadde
 * Now BXH files have the correct namespace declarations (at the expense
 * of some fragile code in dom_utils.c that breaks the GDOME abstraction
 * layer).
 *
 * Revision 1.15  2002/11/08 16:30:20  gadde
 * <imagedata> =-> <datarec type="image">
 *
 * Revision 1.14  2002/10/04 18:33:26  gadde
 * Decreased the precision for checking normals
 *
 * Revision 1.13  2002/09/18 17:31:36  gadde
 * R,A,S dims calculated differently.
 * Added bandwidth element.
 *
 * Revision 1.12  2002/09/13 19:54:02  gadde
 * Changed to non-default namespace
 *
 * Revision 1.11  2002/09/10 04:44:53  gadde
 * Comment positioning.
 *
 * Revision 1.10  2002/09/10 04:27:34  gadde
 * /mriheader -> /bxh
 *
 * Revision 1.9  2002/09/09 22:11:21  gadde
 * Added more fields that should be used in matching files.
 *
 * Revision 1.8  2002/09/09 20:02:28  gadde
 * When parent directory is ".", it is removed from filename.
 *
 * Revision 1.7  2002/09/09 19:48:10  gadde
 * Adds appropriate code for missing SpacingBetweenSlices and for
 * ignoring data not of the same type (derived vs. original) in
 * the search for associated slices.
 * Also, /mriheader -> /bxh
 *
 * Revision 1.6  2002/09/09 17:05:34  gadde
 * Now outputs orientation vectors (dimension/direction).
 *
 * Revision 1.5  2002/08/23 16:54:33  gadde
 * Compiler warning fix.
 *
 * Revision 1.4  2002/08/21 19:45:03  gadde
 * mxh -> bxh
 *
 * Revision 1.3  2002/08/20 20:11:13  gadde
 * Added const to rcsid to avoid compiler warning.
 *
 * Revision 1.2  2002/08/20 15:32:33  gadde
 * Everything moved from libxml to libgdome (DOM).  All utility functions
 * prefixed with domutil_.  XPath support still shaky in gdome, but dom_utils.c
 * accounts for this, and extra code can be removed when gdome fully supports
 * the DOM level 3 XPath specification.
 *
 */
