static const char rcsid[] = "$Id: bxh_concat.c,v 1.3 2009-01-15 20:55:19 gadde Exp $";

/*
 * bxh_concat.c --
 * 
 * Concatenate multiple datasets together.  All dimensions but the
 * last must agree.
 */

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>

#include "nifti1_io.h"
#include "bxh_utils.h"
#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    char * outputbase = NULL;
    char * outputfile = NULL;
    char * outputfilegz = NULL;
    char * outputbxh = NULL;
    int argnum;
    int msbfirst = 1;
    struct bxhdataread bdr;
    int n = 0;
    char * extpos = NULL;
    size_t newdimsize = 0;
    size_t newdatasize = 0;
    int numdims = 0;
    void * buf = NULL;
    
    int oldargc = argc;

    int opt_version = 0;
    int opt_overwrite = 0;
    char * opt_newdim = NULL;
    
    const int numopts = 5;
    opt_data opts[5] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxh_concat [opts] inputs.bxh... output.bxh\n\n"
	  "This program concatenates multiple datasets together and "
	  "produces a single output file.  "
	  "By default, datasets are concatenated along the last "
	  "(slowest-moving) dimension, and all dimensions except the last "
	  "dimension must match in all data sets (but see --newdim)."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite existing output files (otherwise error and exit). " },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_newdim, 1, "newdim",
	  "If specfied, this option specifies the name of a new dimension "
	  "under which each input dataset is placed as a point.  The new "
	  "dimension's size will be equal to the number of input datasets, "
	  "and all input datasets must agree in the sizes of all dimensions." }
    };

    memset(&bdr, '\0', sizeof(bdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc < 3) {
	fprintf(stderr, "Usage: %s [opts] inputs.bxh... output.bxh\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	return -1;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    outputbase = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 1));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 5));
    outputfile = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 5));
    outputfilegz = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 8));
    strcpy(outputbxh, argv[argc-1]);
    {
	char * extpos = NULL;
	extpos = strrchr(outputbxh, '.');
	if (extpos == NULL) {
	    /* no extension on output */
	    strcpy(outputbase, outputbxh);
	    strcpy(outputfile, outputbxh);
	    strcpy(outputfilegz, outputbxh);
	    strcat(outputfile, ".nii");
	    strcat(outputfilegz, ".nii.gz");
	} else {
	    size_t baselen = (extpos - outputbxh);
	    strncpy(outputbase, outputbxh, baselen);
	    strncpy(outputfile, outputbxh, baselen);
	    strncpy(outputfilegz, outputbxh, baselen);
	    outputbase[baselen] = '\0';
	    strcpy(outputfile + baselen, ".nii");
	    strcpy(outputfilegz + baselen, ".nii.gz");
	}
    }
    if (!opt_overwrite) {
	if (stat(outputfilegz, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputfile);
	    goto FAIL;
	}
	if (stat(outputbxh, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputbxh);
	    goto FAIL;
	}
    }

    n = 0;
    for (argnum = 1; argnum < argc - 1; argnum++) {
	struct bxhdataread newbdr;
	int dimnum;

	memset(&newbdr, '\0', sizeof(newbdr));
	
	if (bxh_dataReadFileStart(argv[argnum], "image", NULL, 0, NULL, NULL, &newbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", argv[argnum]);
	    bxh_datareaddata_free(&newbdr);
	    goto FAIL;
	}
	for (dimnum = 0; dimnum < numdims; dimnum++) {
	    bxhdimension * dimp = &bdr.readdatarec->dimensions[dimnum];
	    if (opt_newdim && strcmp(dimp->type, opt_newdim) == 0) {
		fprintf(stderr, "Error: name specified for --newdim (%s) already exists as a dimension in %s.\n", opt_newdim, argv[argnum]);
		bxh_datareaddata_free(&newbdr);
		goto FAIL;
	    }
	}
	if (argnum == 1) {
	    /* first input file */
	} else {
	    /* not the first input file */
	    size_t lastorigdim = 0;
	    if (bdr.readdatarec->numdims != newbdr.readdatarec->numdims) {
		fprintf(stderr, "Error: number of dimensions in '%s' must match '%s'.\n", argv[argnum], argv[1]);
		bxh_datareaddata_free(&newbdr);
		goto FAIL;
	    }
	    numdims = newbdr.readdatarec->numdims;
	    for (dimnum = 0; dimnum < numdims; dimnum++) {
	        bxhdimension * dimp = &bdr.readdatarec->dimensions[dimnum];
	        bxhdimension * newdimp = &newbdr.readdatarec->dimensions[dimnum];
		int dpstructind = 0;
		int newdpstructind = 0;
		if (strcmp(dimp->type, newdimp->type) != 0) {
		    fprintf(stderr, "Error: name of dimension %d does not match in '%s' (%s) and '%s' (%s).\n", dimnum, argv[1], dimp->type, argv[argnum], newdimp->type);
		    bxh_datareaddata_free(&newbdr);
		    goto FAIL;
		}
		/* get the intersection of dpstructs from old and new dims */
		if (dimp->dpstructs == NULL) {
		    if (newdimp->dpstructs != NULL) {
			free(newdimp->dpstructs);
			newdimp->dpstructs = NULL;
			newdimp->numdpstructs = 0;
		    }
		} else if (newdimp->dpstructs == NULL) {
		    if (dimp->dpstructs != NULL) {
			free(dimp->dpstructs);
			dimp->dpstructs = NULL;
			dimp->numdpstructs = 0;
		    }
		} else {
		    int tmpnumdpstructs = 0;
		    bxhdatapoints * tmpdpstructs = (bxhdatapoints *)malloc(sizeof(bxhdatapoints) * (dimp->numdpstructs + newdimp->numdpstructs));
		    for (dpstructind = 0; dpstructind < dimp->numdpstructs; dpstructind++) {
			if (dimp->dpstructs[dpstructind].values == NULL) {
			    continue;
			}
			for (newdpstructind = 0; newdpstructind < newdimp->numdpstructs; newdpstructind++) {
			    bxhdatapoints * dpstruct = &dimp->dpstructs[dpstructind];
			    bxhdatapoints * newdpstruct = &newdimp->dpstructs[newdpstructind];
			    if (newdpstruct->values == NULL) {
				continue;
			    }
			    if (strcmp(dpstruct->label, newdpstruct->label) != 0) {
				continue;
			    }
			    if (!opt_newdim && dimnum == numdims - 1) {
				/* concatenate the datapoints */
				if (strcmp(dpstruct->label, "diffusiondirection") == 0 &&
				    dimp->measurementframe != NULL &&
				    newdimp->measurementframe != NULL) {
				    /* first check for diffusion directions and
				     * see if they and the measurement frame
				     * need to be adjusted before concatenation
				     */
				    if (strcmp(dimp->measurementframeversion, newdimp->measurementframeversion) != 0) {
					fprintf(stderr, "Error: measurement frame version of dimension %d does not match in '%s' (%s) and '%s' (%s).\n", dimnum, argv[1], dimp->measurementframeversion, argv[argnum], newdimp->measurementframeversion);
					bxh_datareaddata_free(&newbdr);
					goto FAIL;
				    }
				    if ((dimp->measurementframe != NULL) !=
					(newdimp->measurementframe != NULL)) {
					fprintf(stderr, "Error: one input has measurement frame in dimension %d ('%s') and another does not ('%s').\n", dimnum, argv[1], argv[argnum]);
					bxh_datareaddata_free(&newbdr);
					goto FAIL;
				    }
				    if (dimp->measurementframe != NULL &&
					newdimp->measurementframe != NULL) {
					int recalculate = 0;
					double epsilon = 1e-8;
					int i;
					for (i = 0; i < 9; i++) {
					    if (fabs(dimp->measurementframe[i] - newdimp->measurementframe[i]) > epsilon) {
						recalculate = 1;
						break;
					    }
					}
					if (recalculate) {
					    /* we need to recalculate diffusion
					     * directions.  We'll maintain the
					     * earlier measurementframe so we
					     * would only need to recalculate
					     * the new vectors */
					    double * mf1 = dimp->measurementframe;
					    double * mf2 = newdimp->measurementframe;
					    mat33 mfmat1;
					    mat33 mfmat1inv;
					    mat33 mfmat2;
					    int i = 0, j = 0, k = 0;
					    size_t valueind;
					    fprintf(stderr, "Warning: measurement frames differ -- recalculating diffusion directions...\n");
					    for (i = 0; i<3; i++) {
						for (j = 0; j<3; j++) {
						    mfmat1.m[i][j] = mf1[k];
						    mfmat2.m[i][j] = mf2[k];
						    k++;
						}
					    }
					    mfmat1inv = nifti_mat33_inverse(mfmat1);
					    for (valueind = 0;
						 valueind < newdpstruct->numvalues;
						 valueind++) {
						static char buf[32];
						char * vecstr = newdpstruct->values[valueind];
						double vec[3];
						double newvec[3];
						char * endptr = NULL;
						int i, j;
						size_t vecstrlen = 0;
						for (i = 0; i < 3; i++) {
						    vec[i] = strtod(vecstr, &endptr);
						    if (endptr == vecstr ||
							errno == ERANGE) {
							fprintf(stderr, "Error: can't read floating-point value from '%s'!\n", vecstr);
							bxh_datareaddata_free(&newbdr);
							goto FAIL;

						    }
						    vecstr = endptr;
						}
						/* convert vector to RAS */
						newvec[0] = newvec[1] = newvec[2] = 0;
						for (i = 0; i < 3; i++) {
						    for (j = 0; j < 3; j++) {
							newvec[i] += mfmat2.m[i][j] * vec[j];
						    }
						}
						/* represent vector in new
						 * measurement frame */
						vec[0] = newvec[0];
						vec[1] = newvec[1];
						vec[2] = newvec[2];
						newvec[0] = newvec[1] = newvec[2] = 0;
						for (i = 0; i < 3; i++) {
						    for (j = 0; j < 3; j++) {
							newvec[i] += mfmat1inv.m[i][j] * vec[j];
						    }
						}
						vecstr = NULL;
						for (i = 0; i < 3; i++) {
						    sprintf(&buf[0], " %.15g", newvec[i]);
						    size_t buflen = strlen(buf);
						    vecstr = (char *)realloc(vecstr, sizeof(char) * (vecstrlen + buflen + 1));
						    strcpy(vecstr + vecstrlen,
							   &buf[0]);
						    vecstrlen += buflen;
						}
						free(newdpstruct->values[valueind]);
						newdpstruct->values[valueind] = vecstr;
					    }
					    /* move measurementframe over */
					    free(newdimp->measurementframe);
					    newdimp->measurementframe = dimp->measurementframe;
					    dimp->measurementframe = NULL;
					}
				    }
				}
				/* we'll cannibalize some of the existing
				 * structs' memory, and remove them from
				 * original arrays so we don't free them */
				int valueind = 0;
				bxhdatapoints * tmpdpstruct = &tmpdpstructs[tmpnumdpstructs];
				tmpnumdpstructs++;
				tmpdpstruct->numvalues = 0;
				tmpdpstruct->values = (char **)malloc(sizeof(char *) * (dpstruct->numvalues + newdpstruct->numvalues));
				for (valueind = 0; valueind < dpstruct->numvalues; valueind++) {
				    tmpdpstruct->values[tmpdpstruct->numvalues] = dpstruct->values[valueind];
				    tmpdpstruct->numvalues++;
				}
				tmpdpstruct->label = dpstruct->label;
				free(dpstruct->values);
				dimp->dpstructs[dpstructind] = dimp->dpstructs[dimp->numdpstructs - 1];
				dimp->numdpstructs--;
				dpstructind--;
				for (valueind = 0; valueind < newdpstruct->numvalues; valueind++) {
				    tmpdpstruct->values[tmpdpstruct->numvalues] = newdpstruct->values[valueind];
				    tmpdpstruct->numvalues++;
				}
				free(newdpstruct->label);
				free(newdpstruct->values);
				newdimp->dpstructs[newdpstructind] = newdimp->dpstructs[newdimp->numdpstructs - 1];
				newdimp->numdpstructs--;
				break;
			    } else {
				/* make sure the datapoints match */
				if (dpstruct->numvalues != newdpstruct->numvalues) {
				    continue;
				}
				int valueind = 0;
				for (valueind = 0; valueind < dpstruct->numvalues; valueind++) {
				    if (strcmp(dpstruct->values[valueind], newdpstruct->values[valueind]) != 0) {
					break;
				    }
				}
				if (valueind != dpstruct->numvalues) {
				    continue;
				}
				/* found a matching datapoints struct */
				/* copy it and remove it from original array
				 * so we don't double-free() things */
				tmpdpstructs[tmpnumdpstructs] = *newdpstruct;
				tmpnumdpstructs++;
				if (newdpstructind < newdimp->numdpstructs - 1) {
				    /* replace with last dpstruct */
				    newdimp->dpstructs[newdpstructind] = newdimp->dpstructs[newdimp->numdpstructs - 1];
				}
				newdimp->numdpstructs--;
				break;
			    }
			}
		    }
		    /* free all dpstructs in newdimp and replace with tmpdpstructs */
		    for (newdpstructind = 0; newdpstructind < newdimp->numdpstructs; newdpstructind++) {
			bxh_datarec_datapoints_free(&newdimp->dpstructs[newdpstructind]);
		    }
		    free(newdimp->dpstructs);
		    if (tmpnumdpstructs == 0) {
			free(tmpdpstructs);
			tmpdpstructs = NULL;
		    }
		    /* release unused entries if any */
		    tmpdpstructs = (bxhdatapoints *)realloc(tmpdpstructs, sizeof(bxhdatapoints) * tmpnumdpstructs);
		    newdimp->dpstructs = tmpdpstructs;
		    newdimp->numdpstructs = tmpnumdpstructs;
		}
	    }
	    if (opt_newdim) {
		lastorigdim = numdims - 1;
	    } else {
		lastorigdim = numdims - 2;
	    }
	    for (dimnum = 0; dimnum <= lastorigdim; dimnum++) {
		if (bdr.readdatarec->dimensions[dimnum].size != newbdr.readdatarec->dimensions[dimnum].size) {
		    fprintf(stderr, "Error: size of dimension %d does not match in '%s' (%d) and '%s' (%d).\n", dimnum, argv[1], (int)bdr.readdatarec->dimensions[dimnum].size, argv[argnum], (int)newbdr.readdatarec->dimensions[dimnum].size);
		    bxh_datareaddata_free(&newbdr);
		    goto FAIL;
		}
	    }
	    if (strcmp(bdr.readdatarec->elemtype, newbdr.readdatarec->elemtype) != 0) {
		fprintf(stderr, "Error: data type does not match in '%s' (%s) and '%s' (%s).\n", argv[1], bdr.readdatarec->elemtype, argv[argnum], newbdr.readdatarec->elemtype);
		bxh_datareaddata_free(&newbdr);
		goto FAIL;
	    }
	    bxh_datareaddata_free(&bdr);
	}
	memcpy(&bdr, &newbdr, sizeof(bdr));

	if (bxh_dataReadFinish(&bdr, NULL) != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", argv[argnum]);
	    goto FAIL;
	}
	buf = (void *)realloc(buf, newdatasize + bdr.datarec->datasize);
	memcpy(buf + newdatasize, bdr.dataptr, bdr.datarec->datasize);
	free(bdr.dataptr); bdr.dataptr = NULL;

	if (opt_newdim) {
	    newdimsize += 1;
	} else {
	    newdimsize += bdr.datarec->dimensions[bdr.datarec->numdims-1].size;
	}
	newdatasize += bdr.datarec->datasize;
    }

    /* write out results */
    if (opt_newdim) {
	bxhdimension * newdim = NULL;
	bdr.datarec->dimensions = (bxhdimension *)realloc(bdr.datarec->dimensions, sizeof(bxhdimension)*(numdims+1));
	newdim = &bdr.datarec->dimensions[numdims];
	memset(newdim, '\0', sizeof(*newdim));
	newdim->type = strdup(opt_newdim);
	newdim->size = newdimsize;
	numdims++;
	bdr.datarec->numdims++;
    } else {
	bdr.datarec->dimensions[bdr.datarec->numdims-1].size = newdimsize;
    }
    bdr.datarec->datasize = newdatasize;
    if (bxh_datarec_writeToElement(bdr.imagedatap, bdr.datarec) != 0) {
	fprintf(stderr, "Failed writing datarec\n");
	goto FAIL;
    }
    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], (const char **)&argv[1], argc-1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    writeBXHAndNIIGZ(outputbase, &bdr, buf, 0);
    goto EXIT;
    
  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr);
    if (outputbase) {
	free(outputbase); outputbase = NULL;
    }
    if (outputbxh) {
	free(outputbxh); outputbxh = NULL;
    }
    if (outputfile) {
	free(outputfile); outputfile = NULL;
    }
    if (outputfilegz) {
	free(outputfilegz); outputfilegz = NULL;
    }
    if (buf) {
	free(buf); buf = NULL;
    }
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.2  2007/02/23 20:09:25  gadde
 * Fix data reading.
 *
 * Revision 1.1  2007/01/16 19:04:24  gadde
 * Initial import.
 *
 */
