static const char rcsid[] = "$Id: bxh_mean.c,v 1.19 2009-02-24 21:24:41 gadde Exp $";

/*
 * bxh_mean.cpp --
 * 
 *  Collapse given dataset(s) across a given dimension by calculating
 *  the per-voxel mean.  Choosing "dataset" as the dimension averages
 *  several datasets to create an output that has the same dimensions
 *  as any of the originals.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>

#include "bxh_utils.h"
#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    char * outputfile = NULL;
    char * outputbxh = NULL;
    char * outputbase = NULL;
    char * stddevfile = NULL;
    char * stddevbxh = NULL;
    char * stddevbase = NULL;
    int dimnum;
    int argnum;
    int msbfirst = 1;
    struct bxhdataread bdr;
    size_t seldimsize = 0;
    double * results = NULL;
    double * stddevresults = NULL;
    size_t resultsize = 0;
    size_t numpages = 0;
    size_t pagenum = 0;
    int n = 0;
    int seldimnum = -1;
    size_t pagesize = 0;
    char * extpos = NULL;
    char * newelemtype = NULL;
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    
    int oldargc = argc;

    char * opt_dimension = NULL;
    char * opt_stddev = NULL;
    int opt_sumonly = 0;
    int opt_version = 0;
    char * opt_outtype = NULL;
    
    const int numopts = 7;
    opt_data opts[7] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxh_mean [opts] inputs.bxh... output.bxh\n\n"
	  "This program calculates per-voxel averages across a selected "
	  "dimension, and produces an output dataset 'collapsed' across "
	  "that dimension.  If --dimension 'dataset' is specified, "
	  "then corresponding "
	  "voxels in each input dataset are averaged to create an output "
	  "dataset of the same dimensionality; in this case, all of the "
	  "dimensions in all input datasets must match.  If multiple input "
	  "datasets are provided and --dimension 'dataset' is not specified, "
	  "then they are concatenated along the last (slowest-moving) "
	  "dimension; i.e. if one specifies an XYZT 64x64x27x120 time series "
	  "and an XYZT 64x64x27x130 time series as inputs, they will be "
	  "considered together as a single 64x64x27x250 time series.  In "
	  "this case, all dimensions except the last dimension must match "
	  "in all data sets."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_stddev, 1, "stddev",
	  "Calculate standard deviation too, and put the output in this "
	  "file." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_sumonly, 1, "sumonly",
	  "Calculate only the sum of the data, and don't divide by "
	  "the number of inputs.  This option can not be used with --stddev."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_dimension, 1, "dimension",
	  "Select the dimension over which to average.  The dimension must "
	  "be one that exists in the input dataset, or must be 'dataset'.  "
	  "Default is the last (slowest-moving) dimension."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_outtype, 1, "outtype",
	  "The output will be of this type.  Valid types are: "
	  "float64, float32, uint32, int32, uint16, int16, uint8, int8.  "
	  "Note: using this option may result in overflow/underflow or "
	  "precision errors if the output type can not represent the output "
	  "appropriately.  Default is float64 if either of the inputs are "
	  "float64, or float32 otherwise."
	}
    };

    memset(&bdr, '\0', sizeof(bdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (opt_sumonly && opt_stddev) {
	fprintf(stderr, "Error: --stddev and --sumonly options are incompatible.\n");
    }
    if (argc < 3) {
	fprintf(stderr, "Usage: %s [opts] inputs.bxh... output.bxh\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	return -1;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    outputbase = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 1));
    outputfile = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 8));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(argv[argc-1]) + 5));
    strcpy(outputbase, argv[argc-1]);
    extpos = strrchr(outputbase, '.');
    if (extpos != NULL) {
	/* output file already ends in .bxh */
	*extpos = '\0';
	strcpy(outputfile, outputbase);
	strcat(outputfile, ".nii.gz");
	strcpy(outputbxh, outputbase);
	strcat(outputbxh, ".bxh");
    } else {
	strcpy(outputfile, outputbase);
	strcat(outputfile, ".nii.gz");
	strcpy(outputbxh, outputbase);
    }
    if (stat(outputfile, &statbuf) == 0) {
	fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputfile);
	return -1;
    }
    if (stat(outputbxh, &statbuf) == 0) {
	fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputbxh);
	return -1;
    }
    if (opt_stddev) {
	stddevbase = (char *)malloc(sizeof(char)*(strlen(opt_stddev) + 1));
	stddevfile = (char *)malloc(sizeof(char)*(strlen(opt_stddev) + 8));
	stddevbxh = (char *)malloc(sizeof(char)*(strlen(opt_stddev) + 5));
	strcpy(stddevbase, opt_stddev);
	extpos = strrchr(stddevbase, '.');
	if (extpos != NULL && strcmp(extpos, ".bxh") == 0) {
	    /* stddev file already ends in .bxh */
	    *extpos = '\0';
	    strcpy(stddevfile, stddevbase);
	    strcat(stddevfile, ".nii.gz");
	    strcpy(stddevbxh, stddevbase);
	    strcat(stddevbxh, ".bxh");
	} else {
	    strcpy(stddevfile, stddevbase);
	    strcat(stddevfile, ".nii.gz");
	    strcpy(stddevbxh, stddevbase);
	}
	if (stat(stddevfile, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], stddevfile);
	    return -1;
	}
	if (stat(stddevbxh, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], stddevbxh);
	    return -1;
	}
    }

    n = 0;
    seldimsize = 0;
    for (argnum = 1; argnum < argc - 1; argnum++) {
	size_t seldimind = 0;
	struct bxhdataread newbdr;

	memset(&newbdr, '\0', sizeof(newbdr));
	
	if (bxh_dataReadFileStart(argv[argnum], "image", NULL, 4, ordereddimnames, NULL, &newbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", argv[argnum]);
	    bxh_datareaddata_free(&newbdr);
	    goto FAIL;
	}
	if (opt_dimension == NULL) {
	    opt_dimension = strdup(newbdr.datarec->dimensions[newbdr.datarec->numdims-1].type);
	}
	if (opt_outtype) {
	    newelemtype = opt_outtype;
	} else {
	    char * oldelemtype = newbdr.datarec->elemtype;
	    if (strcmp(oldelemtype, "float64") == 0 ||
		strcmp(oldelemtype, "double") == 0) {
		newelemtype = "float64";
	    } else {
		newelemtype = "float32";
	    }
	}
	if (argnum == 1) {
	    /* first input file */
	    if (strcmp(opt_dimension, "dataset") == 0) {
		seldimnum = newbdr.datarec->numdims;
	    } else {
		for (seldimnum = 0; seldimnum < newbdr.datarec->numdims; seldimnum++) {
		    if (strcmp(opt_dimension, newbdr.datarec->dimensions[seldimnum].type) == 0)
			break;
		}
		if (seldimnum == newbdr.datarec->numdims) {
		    fprintf(stderr, "Error: didn't find selected dimension '%s' in %s'.\n", opt_dimension, argv[argnum]);
		    bxh_datareaddata_free(&newbdr);
		    goto FAIL;
		}
	    }
	    if (seldimnum == newbdr.datarec->numdims) {
		seldimsize = 1;
	    } else {
		seldimsize = newbdr.dimsizes[seldimnum];
	    }
	    if (seldimnum == 0)
		pagesize = 1;
	    else
		pagesize = newbdr.pagesizes[seldimnum-1];
	    resultsize = 1;
	    numpages = 1;
	    for (dimnum = 0; dimnum < newbdr.datarec->numdims; dimnum++) {
		if (dimnum == seldimnum)
		    continue;
		resultsize *= newbdr.dimsizes[dimnum];
		if (dimnum > seldimnum)
		    numpages *= newbdr.dimsizes[dimnum];
	    }
	} else {
	    /* not the first input file */
	    int dimnum;
	    if (strcmp(opt_dimension, "dataset") == 0 &&
		bdr.datarec->numdims != newbdr.datarec->numdims) {
		fprintf(stderr, "Error: number of dimensions in '%s' must match '%s' when using --dataset.\n", argv[argnum], argv[1]);
		bxh_datareaddata_free(&newbdr);
		goto FAIL;
	    }
	    for (dimnum = 0; dimnum <= seldimnum && dimnum < bdr.datarec->numdims && dimnum < newbdr.datarec->numdims; dimnum++) {
		if (strcmp(bdr.datarec->dimensions[dimnum].type, newbdr.datarec->dimensions[dimnum].type) != 0) {
		    fprintf(stderr, "Error: name of dimension %d does not match in '%s' (%s) and '%s' (%s).\n", dimnum, argv[1], bdr.datarec->dimensions[dimnum].type, argv[argnum], newbdr.datarec->dimensions[dimnum].type);
		    bxh_datareaddata_free(&newbdr);
		    goto FAIL;
		}
	    }
	    for (dimnum = 0; dimnum < seldimnum && dimnum < bdr.datarec->numdims && dimnum < newbdr.datarec->numdims; dimnum++) {
	        if (bdr.datarec->dimensions[dimnum].size !=
		    newbdr.datarec->dimensions[dimnum].size) {
		    fprintf(stderr, "Error: size of dimension %d does not match in '%s' (%u) and '%s' (%u).\n", dimnum, argv[1], (unsigned int)bdr.datarec->dimensions[dimnum].size, argv[argnum], (unsigned int)newbdr.datarec->dimensions[dimnum].size);
		    bxh_datareaddata_free(&newbdr);
		    goto FAIL;
		}
	    }
	    if (strcmp(opt_dimension, "dataset") != 0 && dimnum != seldimnum) {
		fprintf(stderr, "Error: ran out of dimensions in '%s' before finding selected dimension '%s' (which is dimension %u in '%s').\n", argv[argnum], opt_dimension, seldimnum, argv[1]);
		bxh_datareaddata_free(&newbdr);
		goto FAIL;
	    }
	    bxh_datareaddata_free(&bdr);
	}
	memcpy(&bdr, &newbdr, sizeof(bdr));

	if (results == NULL) {
	    results = (double *)malloc(sizeof(double)*resultsize);
	    memset(results, '\0', sizeof(double)*resultsize);
	}
	if (opt_stddev && stddevresults == NULL) {
	    stddevresults = (double *)malloc(sizeof(double)*resultsize);
	    memset(stddevresults, '\0', sizeof(double)*resultsize);
	}

	if (bxh_dataReadFinish(&bdr, "double") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", argv[argnum]);
	    goto FAIL;
	}

	for (seldimind = 0; seldimind < seldimsize; seldimind++) {
	    for (pagenum = 0; pagenum < numpages; pagenum++) {
		size_t respagestart;
		size_t inpagestart;
		size_t pageind;
		respagestart = pagenum*pagesize;
		inpagestart = (seldimind+(pagenum*seldimsize))*pagesize;
		if (opt_sumonly) {
		    for (pageind = 0; pageind < pagesize; pageind++) {
		        double newval = ((double*)bdr.dataptr)[pageind + inpagestart];
			results[pageind + respagestart] += newval;
		    }
		} else {
		    for (pageind = 0; pageind < pagesize; pageind++) {
		        double newval = ((double*)bdr.dataptr)[pageind + inpagestart];
			double oldmean = results[pageind + respagestart];
			double newmean = oldmean + (newval - oldmean) / (n + 1);
			results[pageind + respagestart] = newmean;
			if (opt_stddev) {
			    double oldstddev = stddevresults[pageind + respagestart];
			    double newstddev = oldstddev + ((newval - oldmean) * (newval - newmean));
			    stddevresults[pageind + respagestart] = newstddev;
			}
		    }
		}
	    }
	    n++;
	}
    }
    {
	if (opt_stddev) {
	    for (pagenum = 0; pagenum < numpages; pagenum++) {
		size_t respagestart;
		size_t pageind;
		respagestart = pagenum*pagesize;
		for (pageind = 0; pageind < pagesize; pageind++) {
		    double oldstddev = stddevresults[pageind + respagestart];
		    stddevresults[pageind + respagestart] =
			sqrt(oldstddev / (double)n);
		}
	    }
	}
    }

    /* write out results */
    if (seldimnum < bdr.datarec->numdims) {
	bxh_datarec_dimdata_free(&bdr.datarec->dimensions[seldimnum]);
	memmove(&bdr.datarec->dimensions[seldimnum], &bdr.datarec->dimensions[seldimnum+1], sizeof(bdr.datarec->dimensions[seldimnum])*(bdr.datarec->numdims - seldimnum - 1));
	bdr.datarec->numdims--;
    }
    {
	void * newbuf = NULL;
	void * newstdbuf = NULL;
	int newelemsize = 0;
	if (strcmp(newelemtype, "int8") == 0) {
	    newbuf = bxh_convertBufToChar(results, sizeof(double)*resultsize, "float64");
	} else if (strcmp(newelemtype, "uint8") == 0) {
	    newbuf = bxh_convertBufToUnsignedChar(results, sizeof(double)*resultsize, "float64");
	} else if (strcmp(newelemtype, "int16") == 0) {
	    newbuf = bxh_convertBufToShort(results, sizeof(double)*resultsize, "float64");
	} else if (strcmp(newelemtype, "uint16") == 0) {
	    newbuf = bxh_convertBufToUnsignedShort(results, sizeof(double)*resultsize, "float64");
	} else if (strcmp(newelemtype, "int32") == 0) {
	    newbuf = bxh_convertBufToInt(results, sizeof(double)*resultsize, "float64");
	} else if (strcmp(newelemtype, "uint32") == 0) {
	    newbuf = bxh_convertBufToUnsignedInt(results, sizeof(double)*resultsize, "float64");
	} else if (strcmp(newelemtype, "float32") == 0) {
	    newbuf = bxh_convertBufToFloat(results, sizeof(double)*resultsize, "float64");
	} else if (strcmp(newelemtype, "float64") == 0 || strcmp(newelemtype, "double") == 0) {
	    /* do nothing */
	} else {
	    fprintf(stderr, "Internal error: unrecognized type %s!\n", newelemtype);
	    goto FAIL;
	}
	if (strcmp(newelemtype, "float64") != 0 && strcmp(newelemtype, "double") != 0) {
	    free(results);
	    results = newbuf;
	}
	if (opt_stddev) {
	    if (strcmp(newelemtype, "int8") == 0) {
		newstdbuf = bxh_convertBufToChar(stddevresults, sizeof(double)*resultsize, "float64");
	    } else if (strcmp(newelemtype, "uint8") == 0) {
		newstdbuf = bxh_convertBufToUnsignedChar(stddevresults, sizeof(double)*resultsize, "float64");
	    } else if (strcmp(newelemtype, "int16") == 0) {
		newstdbuf = bxh_convertBufToShort(stddevresults, sizeof(double)*resultsize, "float64");
	    } else if (strcmp(newelemtype, "uint16") == 0) {
		newstdbuf = bxh_convertBufToUnsignedShort(stddevresults, sizeof(double)*resultsize, "float64");
	    } else if (strcmp(newelemtype, "int32") == 0) {
		newstdbuf = bxh_convertBufToInt(stddevresults, sizeof(double)*resultsize, "float64");
	    } else if (strcmp(newelemtype, "uint32") == 0) {
		newstdbuf = bxh_convertBufToUnsignedInt(stddevresults, sizeof(double)*resultsize, "float64");
	    } else if (strcmp(newelemtype, "float32") == 0) {
		newstdbuf = bxh_convertBufToFloat(stddevresults, sizeof(double)*resultsize, "float64");
	    } else if (strcmp(newelemtype, "float64") == 0 || strcmp(newelemtype, "double") == 0) {
		/* do nothing */
	    } else {
		fprintf(stderr, "Internal error: unrecognized type %s!\n", newelemtype);
		goto FAIL;
	    }
	    if (strcmp(newelemtype, "float64") != 0 && strcmp(newelemtype, "double") != 0) {
		free(stddevresults);
		stddevresults = newstdbuf;
	    }
	}
    }
    free(bdr.datarec->elemtype);
    bdr.datarec->elemtype = strdup(newelemtype);
    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], (const char **)&argv[1], oldargc-1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	return -1;
    }
    writeBXHAndNIIGZ(outputbase, &bdr, results, 0);
    free(results);
    if (opt_stddev) {
	writeBXHAndNIIGZ(stddevbase, &bdr, stddevresults, 0);
	free(stddevresults);
    }
    goto EXIT;
    
  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr);
    free(outputbase); outputbase = NULL;
    free(outputbxh); outputbxh = NULL;
    free(outputfile); outputfile = NULL;
    if (opt_stddev) {
	free(stddevbase); stddevbase = NULL;
	free(stddevbxh); stddevbxh = NULL;
	free(stddevfile); stddevfile = NULL;
    }
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.18  2009/02/24 15:37:45  gadde
 * Fix output element type.
 *
 * Revision 1.17  2009/01/15 20:55:19  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.16  2008/12/17 16:25:02  gadde
 * Convert to float32 by default unless any of the inputs are float64.
 * Add --outtype option.
 *
 * Revision 1.15  2008/04/21 16:59:36  gadde
 * Convert more tools to write NIFTI.
 *
 * Revision 1.14  2007/04/18 16:54:02  gadde
 * Add --sumonly option.
 *
 * Revision 1.13  2007/03/07 15:47:37  gadde
 * Check that relevant dimensions match in all input datasets.
 *
 * Revision 1.12  2007/02/07 21:35:30  gadde
 * Add --stddev option.
 *
 * Revision 1.11  2007/02/03 19:57:59  gadde
 * Don't reinitialize seldimsize for each argument: this caused only the
 * first input file to be used for the average!
 *
 * Revision 1.10  2006/04/13 21:27:41  gadde
 * Fix several bugs in dimension selection and error-checking.
 *
 * Revision 1.9  2005/09/20 18:37:55  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.8  2005/09/19 16:31:57  gadde
 * Documentation and help message updates.
 *
 * Revision 1.7  2005/09/14 15:10:38  gadde
 * Some -Wall fixes.
 *
 * Revision 1.6  2005/04/22 20:54:54  gadde
 * Don't reset n for each file.
 *
 * Revision 1.5  2005/04/19 19:24:28  gadde
 * Don't declare variables in the middle of a block.
 *
 * Revision 1.4  2005/04/19 14:33:54  gadde
 * Add option arguments to history too.
 *
 * Revision 1.3  2005/04/19 14:16:44  gadde
 * Don't delete dimensions if --dimension "dataset"
 *
 * Revision 1.2  2005/04/19 13:33:29  gadde
 * Fix argument handling.
 *
 * Revision 1.1  2005/04/19 13:23:00  gadde
 * Initial import.
 *
 */
