static const char rcsid[] = "$Id: bxh_unop.c,v 1.10 2009-02-17 18:32:29 gadde Exp $";

/*
 * bxh_unop.c --
 * 
 *  Apply a unary operator to the input dataset.
 *  Output dimensions are the same as the input.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#ifndef WIN32
#include <unistd.h>
#endif

#include "bxh_niftilib.h"
#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

/* linked in from nifti_stats.c */
extern int nifti_intent_code( char *name );
extern double nifti_stat2cdf( double val, int code, double p1,double p2,double p3 );
extern double nifti_stat2rcdf( double val, int code, double p1,double p2,double p3 );
extern double nifti_cdf2stat( double p , int code, double p1,double p2,double p3 );
extern double nifti_rcdf2stat( double q , int code, double p1,double p2,double p3 );
extern double nifti_stat2zscore( double val , int code, double p1,double p2,double p3 );
extern double nifti_stat2hzscore( double val, int code, double p1,double p2,double p3 );

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    const char * inputfile = NULL;
    const char * outputarg = NULL;
    char * outputbase = NULL;
    char * outputfile = NULL;
    char * outputbxh = NULL;
    char * opt_maskfile = NULL;
    int msbfirst = 1;
    FILE * fp = NULL;
    char * extpos = NULL;
    int opt_overwrite = 0;
    int opt_version = 0;
    int opt_sqrt = 0;
    int opt_fabs = 0;
    int opt_exp = 0;
    int opt_log = 0;
    int opt_cos = 0;
    int opt_sin = 0;
    int opt_tan = 0;
    int opt_acos = 0;
    int opt_asin = 0;
    int opt_atan = 0;
    char * opt_nifticode = NULL;
    char * opt_niftiopt = "p";
    double opt_niftiparams[3] = { 0, 0, 0 };
    char * opt_niftiparamfiles[3] = { NULL, NULL, NULL };
    int nifticode = 0;
    int nifti_dop = 0;
    int nifti_doq = 0;
    int nifti_dod = 0;
    int nifti_doi = 0;
    int nifti_doz = 0;
    int nifti_doh = 0;

    struct bxhdataread bdr;
    struct bxhdataread npbdrs[3];
    struct bxhdataread maskbdr;
    float * dataptr = NULL;
    float * npdataptrs[3] = { NULL, NULL, NULL };
    size_t npnumelems[3] = { 0, 0, 0 };
    char * maskdataptr = NULL;

    const int numopts = 26;
    opt_data opts[26] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxh_unop [opts] xmlfile outputfile\n\n"
	  "This program takes one input BXH or XCEDE image, and "
	  "applies the operator specified by the following options "
	  "(definitions are the same as the standard C math library).  "
	  "Dimensions of the output are the same as the input."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "mask",
	  "Use this mask (should be an XML file) before doing "
	  "calculations." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_sqrt, 1, "sqrt", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_fabs, 1, "fabs", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_exp, 1, "exp", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_log, 1, "log", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_cos, 1, "cos", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_sin, 1, "sin", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_tan, 1, "tan", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_acos, 1, "acos", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_asin, 1, "asin", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_atan, 1, "atan", "" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "NIFTI stats:" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_nifticode, 1, "nifticode",
	  "Also available are the statistical functions provided by "
	  "the nifti_stats program in the NIfTI-1 library.  You can "
	  "specify the 'code' using this option.  Further parameterizations "
	  "and options are giving by the options that follow.  "
	  "For more information please see the nifti_stats documentation in "
	  "the NIfTI-1 library, available from http://nifti.nimh.nih.gov/"
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_niftiopt, 1, "niftiopt",
	  "This corresponds to the -p, -q, -d, -1, -z, or -h options to "
	  "nifti_stats.  Valid values are 'p', 'q', 'd', '1', 'z', or 'h'."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_niftiparams[0], 1, "niftiparam1", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_niftiparams[1], 1, "niftiparam2", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_niftiparams[2], 1, "niftiparam3", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_niftiparamfiles[0], 1, "niftiparamfile1", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_niftiparamfiles[1], 1, "niftiparamfile2", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_niftiparamfiles[2], 1, "niftiparamfile3",
	  "These options correspond to the p1, p2, p3 parameters in "
	  "nifti_stats (only some are required by certain 'codes').  "
	  "In the niftiparamfile* options, the values are taken from "
	  "a BXH/XCEDE file.  If necessary, the data in the file is "
	  "replicated to match the size of the input file."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite output files if they exist." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
    };

    memset(&bdr, '\0', sizeof(bdr));
    memset(&npbdrs[0], '\0', sizeof(bdr));
    memset(&npbdrs[1], '\0', sizeof(bdr));
    memset(&npbdrs[2], '\0', sizeof(bdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc < 3) {
	fprintf(stderr, "Not enough arguments!  Use --help for help\n");
	goto FAIL;
    }
    if (opt_sqrt + opt_fabs + opt_exp + opt_log + opt_cos + opt_sin + opt_tan + opt_acos + opt_asin + opt_atan + (opt_nifticode == NULL ? 0 : 1) != 1) {
	fprintf(stderr, "One and only one of --sqrt, --fabs, --exp, --log, --cos, --sin, --tan, --acos, --asin, --atan, --nifticode must be specified!\n");
	goto FAIL;
    }
    if (opt_nifticode != NULL) {
	nifticode = nifti_intent_code(opt_nifticode) ;
	if (nifticode < 0) {
	    fprintf(stderr, "Unsupported nifticode %s\n", opt_nifticode);
	    goto FAIL;
	}
	if (strlen(opt_niftiopt) != 1 || strspn(opt_niftiopt, "pqd1zh") != 1) {
	    fprintf(stderr, "Unsupported niftiopt %s\n", opt_niftiopt);
	    goto FAIL;
	}
	switch (*opt_niftiopt) {
	case 'p': nifti_dop = 1; break;
	case 'q': nifti_doq = 1; break;
	case 'd': nifti_dod = 1; break;
	case 'i': nifti_doi = 1; break;
	case 'z': nifti_doz = 1; break;
	case 'h': nifti_doh = 1; break;
	}
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile = argv[1];
    outputarg = argv[2];
    outputbase = (char *)malloc(sizeof(char)*(strlen(outputarg) + 1));
    outputfile = (char *)malloc(sizeof(char)*(strlen(outputarg) + 8));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(outputarg) + 5));
    strcpy(outputbase, outputarg);
    strcpy(outputbxh, outputarg);
    extpos = strrchr(outputbase, '.');
    if (extpos != NULL) {
	*extpos = '\0';
    }
    strcpy(outputfile, outputbase);
    strcat(outputfile, ".nii.gz");
    if (!opt_overwrite) {
	if (stat(outputfile, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputfile);
	    goto FAIL;
	}
	if (stat(outputbxh, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputbxh);
	    goto FAIL;
	}
    }

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, NULL, NULL, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }

    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, NULL, NULL, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be at least 3-dimensional.\n");
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims > bdr.datarec->numdims) {
	    fprintf(stderr, "Number of dimensions in mask (%u) must not be greater than those in input data (%u).\n", (unsigned int)maskbdr.datarec->numdims, (unsigned int)bdr.datarec->numdims);
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, bdr.dimsizes, sizeof(maskbdr.dimsizes[0]) * maskbdr.datarec->numdims) != 0) {
	    fprintf(stderr, "Mask dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
    }

    if (bxh_dataReadFinish(&bdr, "float") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (opt_maskfile) {
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
    }
    dataptr = (float *)bdr.dataptr;
    if (opt_maskfile)
	maskdataptr = (char *)maskbdr.dataptr;

    {
	int npnum = 0;
	for (npnum = 0; npnum < 3; npnum++) {
	    struct bxhdataread * npbdrp = &npbdrs[npnum];
	    if (opt_niftiparamfiles[npnum] != NULL) {
		if (bxh_dataReadFileStart(opt_niftiparamfiles[npnum], "image", NULL, 4, NULL, NULL, npbdrp) != 0) {
		    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_niftiparamfiles[npnum]);
		    goto FAIL;
		}
		if (bxh_dataReadFinish(npbdrp, "float") != 0) {
		    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_niftiparamfiles[npnum]);
		    goto FAIL;
		}
		{
		    int dimnum;
		    int numdims1 = bdr.datarec->numdims;
		    int numdims2 = npbdrp->datarec->numdims;
		    int datasize1 = bdr.pagesizes[numdims1 - 1];
		    int datasize2 = npbdrp->pagesizes[numdims2 - 1];
		    double ratio = 0;
		    for (dimnum = 0;
			 dimnum < numdims1 && dimnum < numdims2;
			 dimnum++) {
			if (bdr.dimsizes[dimnum] != npbdrp->dimsizes[dimnum]) {
			    break;
			}
		    }
		    if (numdims1 - dimnum > 1 &&
			numdims2 - dimnum > 1) {
			fprintf(stderr, "Error: The sizes of the two input datasets must differ (if at all) only in the\nlast dimension of the smaller dataset.  Input %s has %u dimensions and\nInput %s has %u dimensions.  They both differ in dimension %u (%u vs. %u)\n", inputfile, bdr.datarec->numdims, opt_niftiparamfiles[npnum], npbdrp->datarec->numdims, dimnum + 1, (unsigned int)bdr.dimsizes[dimnum], (unsigned int)npbdrp->dimsizes[dimnum]);
			goto FAIL;
		    }
		    if (datasize1 >= datasize2) {
			ratio = (double)datasize1 / (double)datasize2;
		    } else {
			fprintf(stderr, "Error: size of niftiparam file %s (%u) must not be bigger than that of input file %s (%u).\n", opt_niftiparamfiles[npnum], (unsigned int)datasize1, inputfile, (unsigned int)datasize2);
			goto FAIL;
		    }
		    if ((double)(int)ratio != ratio) {
			fprintf(stderr, "Error: larger dataset %s size (%u) is not an exact multiple of\nthe smaller data set %s size (%u).\n", inputfile, (unsigned int)datasize1, opt_niftiparamfiles[npnum], (unsigned int)datasize2);
			goto FAIL;
		    }
		}
		npnumelems[npnum] = npbdrp->pagesizes[npbdrp->datarec->numdims-1];
	    } else {
		npbdrp->datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
		memset(npbdrp->datarec, '\0', sizeof(bxhrawdatarec));
		npbdrp->datarec->numdims = 1;
		npbdrp->dimsizes = (size_t *)malloc(sizeof(size_t) * 1);
		npbdrp->pagesizes = (size_t *)malloc(sizeof(size_t) * 1);
		npbdrp->dimsizes[0] = 1;
		npbdrp->pagesizes[0] = 1;
		npbdrp->dataptr = (void *)malloc(sizeof(float) * 1);
		((float *)npbdrp->dataptr)[0] = opt_niftiparams[npnum];
		npbdrp->datarec->datasize = sizeof(float);
		npbdrp->dataptrsize = sizeof(float);
		npnumelems[npnum] = 1;
	    }
	    npdataptrs[npnum] = (float *)npbdrp->dataptr;
	}
    }
	
    /*** Do the dirty work here ***/
    {
	/* results go back into dataptr */
	size_t ind = 0; /* within volume */
	off_t numelems = bdr.pagesizes[bdr.datarec->numdims-1];
	if (opt_sqrt) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = sqrt(dataptr[ind]);
	    }
	} else if (opt_fabs) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = fabs(dataptr[ind]);
	    }
	} else if (opt_exp) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = exp(dataptr[ind]);
	    }
	} else if (opt_log) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = log(dataptr[ind]);
	    }
	} else if (opt_cos) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = cos(dataptr[ind]);
	    }
	} else if (opt_sin) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = sin(dataptr[ind]);
	    }
	} else if (opt_tan) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = tan(dataptr[ind]);
	    }
	} else if (opt_acos) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = acos(dataptr[ind]);
	    }
	} else if (opt_asin) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = asin(dataptr[ind]);
	    }
	} else if (opt_atan) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr[ind] = atan(dataptr[ind]);
	    }
	} else if (opt_nifticode) {
	    int code = nifticode;
	    for (ind = 0; ind < numelems; ind++) {
		double val = (double)dataptr[ind];
		double newval = 0;
		double p1 = npdataptrs[0][ind % npnumelems[0]];
		double p2 = npdataptrs[1][ind % npnumelems[1]];
		double p3 = npdataptrs[2][ind % npnumelems[2]];
		if (nifti_doq)                                        /* output = 1-cdf */
		    newval = nifti_stat2rcdf(val,code,p1,p2,p3);
		else if (nifti_dod)                                   /* output = density */
		    newval =
			1000.0*(nifti_stat2cdf(val+.001,code,p1,p2,p3)
				-nifti_stat2cdf(val,code,p1,p2,p3));
		else if (nifti_doi)                                   /* output = inverse */
		    newval = nifti_cdf2stat(val,code,p1,p2,p3);
		else if (nifti_doz)                                   /* output = z score */
		    newval = nifti_stat2zscore(val,code,p1,p2,p3);
		else if (nifti_doh)                                   /* output = halfz score */
		    newval = nifti_stat2hzscore(val,code,p1,p2,p3);
		else if (nifti_dop)                                   /* output = cdf */
		    newval = nifti_stat2cdf(val,code,p1,p2,p3);
		dataptr[ind] = (float)newval;
	    }
	}
	
	if (maskdataptr != NULL) {
	    for (ind = 0; ind < numelems; ind++) {
	        size_t maskvoxelnum = ind % maskbdr.pagesizes[maskbdr.datarec->numdims-1];
		if (maskdataptr[maskvoxelnum] == 0) {
		    dataptr[ind] = 0;
		}
	    }
	}
    }

    /* write out results */
    free(bdr.datarec->elemtype);
    bdr.datarec->elemtype = strdup("float32");
    if (bxh_datarec_writeToElement(bdr.imagedatap, bdr.datarec) != 0) {
	fprintf(stderr, "Failed writing datarec\n");
	goto FAIL;
    }
    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], &inputfile, 1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    writeBXHAndNIIGZ(outputbase, &bdr, dataptr, 0);

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr);
    if (opt_maskfile)
	bxh_datareaddata_free(&maskbdr);
    {
	int npnum;
	for (npnum = 0; npnum < 3; npnum++) {
	    bxh_datareaddata_free(&npbdrs[npnum]);
	}
    }
    free(outputbxh); outputbxh = NULL;
    free(outputfile); outputfile = NULL;
    free(outputbase); outputbase = NULL;
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.9  2009/01/15 20:55:19  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.8  2008/04/21 16:59:36  gadde
 * Convert more tools to write NIFTI.
 *
 * Revision 1.7  2007/05/11 15:50:18  gadde
 * Add dataptrsize field to reflect size of data (returned by bxh_dataReadFinish) in memory
 *
 * Revision 1.6  2007/03/20 17:41:57  gadde
 * Add --mask option
 *
 * Revision 1.5  2007/02/26 17:13:52  gadde
 * Add NIFTI stats
 *
 * Revision 1.4  2007/02/12 21:18:03  gadde
 * updates to opts
 *
 * Revision 1.3  2007/02/08 18:51:11  gadde
 * Update error message.
 *
 * Revision 1.2  2007/02/08 17:35:22  gadde
 * Add nifti_stats support.
 *
 * Revision 1.1  2007/01/16 19:04:38  gadde
 * Initial import.
 *
 */
