static const char rcsid[] = "$Id: bxhselect.c,v 1.13 2009-02-17 18:31:43 gadde Exp $";

/*
 * bxhselect.c --
 *
 *  Select a portion of BXH dataset and output it to new dataset
 *
 */

#include <bxh_config.h>

#include <stdio.h>

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "bxh_datarec.h"
#include "bxh_niftilib.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#define PROGNAME "bxhselect"

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    const char * inputfile = NULL;
    char * outputbase = NULL;
    char * outputfile = NULL;
    char * outputbxh = NULL;
    int dimnum;
    int msbfirst = 1;
    FILE * fp = NULL;
    int opt_overwrite = 0;
    const char * opt_select[4] = {":", ":", ":", ":"};
    int opt_version = 0;
    
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdr;
    char * extpos = NULL;

    const int numopts = 8;
    opt_data opts[8] = {
	{ 0x0, 0x0, NULL, 0, "",
	  "Usage:\n"
	  "  " PROGNAME " [options] inputfile outputfile\n\n"
	  "This program copies a subset of the input image data "
	  "based on the various selection options.  The selected subset "
	  "is written to the output file." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	  "Just like timeselect, but for the 'x' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	  "Just like timeselect, but for the 'y' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Just like timeselect, but for the 'z' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite output files if they exist." },
    };

    memset(&bdr, '\0', sizeof(bdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 3) {
	fprintf(stderr, "Usage: %s [opts] xmlfile outputfile\n", argv[0]);
	fprintf(stderr, "Use --help for more info.\n");
	goto FAIL;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile = argv[1];
    outputbase = (char *)malloc(sizeof(char)*(strlen(argv[2]) + 1));
    outputfile = (char *)malloc(sizeof(char)*(strlen(argv[2]) + 8));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(argv[2]) + 5));
    strcpy(outputbase, argv[2]);
    strcpy(outputbxh, argv[2]);
    extpos = strrchr(outputbase, '.');
    if (extpos != NULL) {
	*extpos = '\0';
    }
    strcpy(outputfile, outputbase);
    strcat(outputfile, ".nii.gz");
    if (!opt_overwrite) {
        if (stat(outputfile, &statbuf) == 0) {
	  fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputfile);
	  goto FAIL;
	}
	if (stat(outputbxh, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputbxh);
	    goto FAIL;
	}
    }

    /* read the input file and data */
    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (bxh_dataReadFinish(&bdr, NULL) != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }

    {
	/* add missing dimensions */
	int odimnum;
	for (odimnum = 0; odimnum < 3; odimnum++) {
	    if (odimnum < bdr.datarec->numdims &&
		strcmp(bdr.datarec->dimensions[odimnum].type, ordereddimnames[odimnum]) != 0) {
		bxhdimension * newdim = NULL;
		bdr.datarec->numdims++;
		bdr.datarec->dimensions = (bxhdimension *)realloc(bdr.datarec->dimensions, sizeof(bxhdimension)*bdr.datarec->numdims);
		memmove(&bdr.datarec->dimensions[odimnum+1], &bdr.datarec->dimensions[odimnum], sizeof(bxhdimension) * (bdr.datarec->numdims - odimnum - 1));
		newdim = &bdr.datarec->dimensions[odimnum];
		memset(newdim, '\0', sizeof(bxhdimension));
		newdim->type = strdup(ordereddimnames[odimnum]);
		newdim->size = 1;
	    }
	}
    }

    /* write out data */
    if (bxh_datarec_writeToElement(bdr.imagedatap, bdr.datarec) != 0) {
	fprintf(stderr, "Failed writing datarec\n");
	goto FAIL;
    }
    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], &inputfile, 1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    writeBXHAndNIIGZ(outputbase, &bdr, bdr.dataptr, 0);

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr);
    free(outputbxh); outputbxh = NULL;
    free(outputfile); outputfile = NULL;
    free(outputbase); outputbase = NULL;
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.12  2009/01/15 20:55:19  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.11  2008/05/21 17:12:47  gadde
 * Remove extraneous fprintf.
 *
 * Revision 1.10  2008/05/21 15:48:47  gadde
 * Add overwrite option
 *
 * Revision 1.9  2008/04/21 16:59:36  gadde
 * Convert more tools to write NIFTI.
 *
 * Revision 1.8  2006/11/16 16:58:31  gadde
 * Work on data that's not 4D and doesn't have xyzt order.
 *
 * Revision 1.7  2005/09/20 18:37:56  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.6  2005/09/19 16:31:57  gadde
 * Documentation and help message updates.
 *
 * Revision 1.5  2004/06/22 17:49:11  gadde
 * Clean up a little more when freeing memory.
 *
 * Revision 1.4  2004/06/18 18:26:21  gadde
 * Delete useless usage.
 *
 * Revision 1.3  2004/06/18 18:19:23  gadde
 * Delete C++-ism.
 *
 * Revision 1.2  2004/06/18 13:54:24  gadde
 * Add to usage message
 *
 * Revision 1.1  2004/06/17 22:51:59  gadde
 * *** empty log message ***
 *
 *
 */
