static const char rcsid[] = "$Id: bxhsetorient.c,v 1.5 2005-09-20 18:37:56 gadde Exp $";

/*
 * bxhsetorient.c --
 * 
 * Sets the orientation of a BXH file
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <sys/stat.h>
#include <math.h>
#include "bxh_types.h"
#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

static int opt_inplace = 0;

#define PROGNAME "bxhsetorient"

int
main(int argc, char *argv[])
{
    struct stat statbuf;
    char * ifname = NULL;
    char * ofname = NULL;
    bxhrawdatarec * datarec = NULL;
    BXHDocPtr docp = NULL;
    BXHElementPtr imagedatap = NULL;
    off_t iflen;
    int i;
    int result = 0;
    char * opt_orientation = "RAS";
    int opt_version = 0;
    int oldargc = argc;

    const int numopts = 4;
    opt_data myOpts[4] = {
	{ 0x0, 0x0, NULL, 0, "",
	  "Usage:\n"
	  "  " PROGNAME " [options] orient inputfile [outputfile]\n\n"
	  "This program sets the orientation vectors in the BXH or XCEDE "
	  "file.  NOTE: this program does not reorient or otherwise touch "
	  "the image data itself.  This program is useful to fix "
	  "incorrect or missing orientation vectors in a BXH/XCEDE file.  "
	  "bxhfileout is required if not using --inplace option "
	  "(and vice-versa).  "
	  "'orient' specifies the new orientation by R/L A/P S/I letters, "
	  "upper- or lower-case, in X,Y,Z order, "
	  "where R means that dimension starts on the left "
	  "and goes TO THE RIGHT, "
	  "A means the dimension goes from posterior TO ANTERIOR, etc.  "
	  "For example, IPR means X goes S->I, Y goes A->P, and Z goes L->R." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_inplace, 1, "inplace",
	  "Do the reorientation in-place, overwriting the original BXH file."}
    }; 

    argc -= opt_parse(argc, argv, numopts, &myOpts[0], 0);
    
    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc < 3 || (opt_inplace && argc > 3) || (!opt_inplace && argc < 4) || argc > 4) {
	opt_usage(numopts, &myOpts[0]);
	return -1;
    }

    opt_orientation = argv[1];
    ifname = strdup(argv[2]);
    if (opt_inplace) {
	ofname = strdup(ifname);
    } else {
	ofname = strdup(argv[3]);
    }

    iflen = 0;

    if (stat(ifname, &statbuf) != 0) {
	fprintf(stderr, PROGNAME ": Input BXH file('%s') does not exist \n", ifname); 
	goto FAIL;
    }
    if (!opt_inplace && stat(ofname, &statbuf) == 0) {
	fprintf(stderr, PROGNAME ": Output BXH file('%s') exists!\n", ofname); 
	goto FAIL;
    }

    iflen = strlen(ifname);
    if (iflen < 4 || (strcmp(ifname+iflen-4, ".bxh") != 0)) {
	fprintf(stderr, PROGNAME ": Bad BXH filename? (must end in .bxh)\n");
	goto FAIL; 
    } 
    if ((docp = bxh_readFile(ifname)) == NULL) { 
	fprintf(stderr, "Reading BXH file failed.\n"); 
	goto FAIL;
    } 
    if ((imagedatap = bxh_getDatarec(docp, "image", NULL)) == NULL) {
	fprintf(stderr, "Getting 'image' datarec failed.\n");
	goto FAIL; 
    } 
    if ((datarec = bxh_datarec_createFromElement(imagedatap, ifname)) == NULL) {
	fprintf(stderr, "Creating raw datarec failed.\n");
	goto FAIL; 
    }

    /* create matrix that transforms coordinates in original
     * orientation to that of new orientation */
    for (i = 0; i < 3; i++) {
	bxhdimension * dimp = &datarec->dimensions[i];
	double * dirp = dimp->direction;
	if (dirp == NULL) {
	    fprintf(stderr, PROGNAME ": Dimension '%s' is missing direction field!\n", dimp->type);
	    goto FAIL;
	}
	memset(dirp, '\0', sizeof(*dirp)*3);
	if (tolower(opt_orientation[i]) == 'r') {
	    dirp[0] = 1;
	} else if (tolower(opt_orientation[i]) == 'l') {
	    dirp[0] = -1;
	} else if (tolower(opt_orientation[i]) == 'a') {
	    dirp[1] = 1;
	} else if (tolower(opt_orientation[i]) == 'p') {
	    dirp[1] = -1;
	} else if (tolower(opt_orientation[i]) == 's') {
	    dirp[2] = 1;
	} else if (tolower(opt_orientation[i]) == 'i') {
	    dirp[2] = -1;
	} else {
	    fprintf(stderr, "Unrecognized orientation letter '%c'!\n",
		    opt_orientation[i]);
	    goto FAIL;
	}
    }

    /* datarec now contains new orientation */
    /* write the BXH file out */
    if (bxh_datarec_writeToElement(imagedatap, datarec) == -1) {
	fprintf(stderr, "Error writing new orientation to datarec element\n");
	goto FAIL;
    } 
    if (bxh_addAutoHistoryEntry(docp, argv[0], (const char **)&argv[1], oldargc-1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	return -1;
    }
    if (bxh_writeFile(docp, ofname) != 0) {
	fprintf(stderr, "Error writing output BXH file\n");
	goto FAIL;
    } 

    goto EXIT;

FAIL:
    fprintf(stderr, PROGNAME ": failed.\n");
    result = -1;

EXIT:
    if (ifname)
	free(ifname);
    if (ofname)
	free(ofname);
    
    if (datarec)
      bxh_datarec_free(datarec);
    if (imagedatap)
      bxh_element_unref(imagedatap);
    if (docp)
      bxh_freeDocument(docp);

    return result;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.4  2005/09/19 16:31:57  gadde
 * Documentation and help message updates.
 *
 * Revision 1.3  2005/09/14 15:19:17  gadde
 * Some -Wall fixes.
 *
 * Revision 1.2  2005/02/03 16:57:11  gadde
 * Do better argument validation and add history entry.
 *
 * Revision 1.1  2005/02/03 16:45:48  gadde
 * Initial import.
 *
 */
