/*
 * xcede2edit.c -- do some simple manipulation of metadata in XCEDE2 files
 */

#include <stdio.h>
#include <string.h>
#include <gdome.h>
#include "opts.h"

int
main(int argc, char *argv[])
{
    int retval = 0;

    GdomeDOMImplementation * di = NULL;
    GdomeDocument * doc = NULL;
    GdomeElement * xcede = NULL;
    GdomeNode * node = NULL;
    GdomeDOMString * domstr = NULL;
    GdomeException exc;
    GdomeDOMString * idname = NULL;

    char * buf = NULL;

    enum {
	LEVELREF_NONE = 0,
	LEVELREF_PROJECT = 1,
	LEVELREF_SUBJECT = 2,
	LEVELREF_VISIT = 3,
	LEVELREF_STUDY = 4,
	LEVELREF_EPISODE = 5,
	LEVELREF_ACQUISITION = 6,
	LEVELREF_CATALOG = 7,
	LEVELREF_ANALYSIS = 8,
	LEVELREF_RESOURCE = 9,
	LEVELREF_DATA = 10
    };

    char * opt_projectid = NULL;
    char * opt_subjectgroupid = NULL;
    char * opt_subjectid = NULL;
    char * opt_visitid = NULL;
    char * opt_studyid = NULL;
    char * opt_episodeid = NULL;
    char * opt_acquisitionid = NULL;
    char * opt_analysisid = NULL;
    char * opt_resourceid = NULL;
    char * opt_level = NULL;
    const int numopts = 12;
    opt_data opts[12] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  xcede2edit [opts] input.xcede2 [ output.xcede2 ]\n\n"
	  "This program changes fields in an XCEDE2 dataset, stored in "
	  "input.xcede2 and writes the output to output.xcede2.  If "
	  "output.xcede2 is missing, it will write to standard output." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_projectid, 1, "projectid",
	  "Set the projectID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_subjectid, 1, "subjectid",
	  "Set the subjectID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_subjectgroupid, 1, "subjectgroupid",
	  "Set the subjectGroupID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_visitid, 1, "visitid",
	  "Set the visitID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_studyid, 1, "studyid",
	  "Set the studyID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_episodeid, 1, "episodeid",
	  "Set the episodeID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_acquisitionid, 1, "acquisitionid",
	  "Set the acquisitionID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_analysisid, 1, "analysisid",
	  "Set the analysisID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_resourceid, 1, "resourceid",
	  "Set the resource ID attribute in any element that supports it." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_level, 1, "level",
	  "Set the level attribute in any element that supports it." }
    };

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (argc < 2 || argc > 3) {
	fprintf(stderr, "ERROR: one or two arguments required!\n");
	return 1;
    }

    di = gdome_di_mkref();

    if ((doc = gdome_di_createDocFromURI(di, argv[1], GDOME_LOAD_PARSING, &exc)) == NULL) {
	fprintf(stderr, "DOMImplementation.createDocFromURI: failed\n\tException #%d\n", exc);
	goto FAIL;
    }

    if ((xcede = gdome_doc_documentElement(doc, &exc)) == NULL) {
	fprintf(stderr, "Document.documentElement: failed\n\tException #%d\n", exc);
	goto FAIL;
    }

    if ((domstr = gdome_el_localName(xcede, &exc)) == NULL) {
	fprintf(stderr, "Element.localName: failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    if (strcmp(domstr->str, "XCEDE") != 0) {
	fprintf(stderr, "Root element is not <XCEDE>!");
	goto FAIL;
    }
    gdome_str_unref(domstr); domstr = NULL;
    if ((domstr = gdome_el_namespaceURI(xcede, &exc)) == NULL) {
	fprintf(stderr, "Element.namespaceURI: failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    if (strcmp(domstr->str, "http://www.xcede.org/xcede-2") != 0) {
	fprintf(stderr, "Root <XCEDE> element is not in namespace http://www.xcede.org/xcede-2 !\n");
	goto FAIL;
    }
    gdome_str_unref(domstr); domstr = NULL;

    node = gdome_el_firstChild(xcede, &exc);
    if (node == NULL && exc > 0) {
	fprintf(stderr, "Element.firstChild: failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    idname = gdome_str_mkref("ID");
    while (node) {
	int levelref = LEVELREF_NONE;
	GdomeElement * elem = NULL;
	GdomeDOMString * namestr = NULL;
	GdomeDOMString * valuestr = NULL;
	unsigned short nodetype = 0;

	if ((nodetype = gdome_n_nodeType(node, &exc)) == 0) {
	    fprintf(stderr, "Node.nodeType: failed\n\tException #%d\n", exc);
	    goto FAIL;
	}
	if (nodetype != GDOME_ELEMENT_NODE) {
	    goto NEXTELEM;
	}
	elem = (GdomeElement *)node;

	if ((domstr = gdome_el_namespaceURI(elem, &exc)) == NULL) {
	    fprintf(stderr, "Element.namespaceURI: failed\n\tException #%d\n", exc);
	    goto FAIL;
	}
	if (strcmp(domstr->str, "http://www.xcede.org/xcede-2") != 0) {
	    gdome_str_unref(domstr); domstr = NULL;
	    goto NEXTELEM;
	}
	gdome_str_unref(domstr); domstr = NULL;

	if ((domstr = gdome_el_localName(elem, &exc)) == NULL) {
	    fprintf(stderr, "Element.localName: failed\n\tException #%d\n", exc);
	    goto FAIL;
	}
	if (strcmp(domstr->str, "project") == 0) {
	    levelref = LEVELREF_PROJECT;
	} else if (strcmp(domstr->str, "subject") == 0) {
	    levelref = LEVELREF_SUBJECT;
	} else if (strcmp(domstr->str, "visit") == 0) {
	    levelref = LEVELREF_VISIT;
	} else if (strcmp(domstr->str, "study") == 0) {
	    levelref = LEVELREF_STUDY;
	} else if (strcmp(domstr->str, "episode") == 0) {
	    levelref = LEVELREF_EPISODE;
	} else if (strcmp(domstr->str, "acquisition") == 0) {
	    levelref = LEVELREF_ACQUISITION;
	} else if (strcmp(domstr->str, "catalog") == 0) {
	    levelref = LEVELREF_CATALOG;
	} else if (strcmp(domstr->str, "analysis") == 0) {
	    levelref = LEVELREF_ANALYSIS;
	} else if (strcmp(domstr->str, "resource") == 0) {
	    levelref = LEVELREF_RESOURCE;
	} else if (strcmp(domstr->str, "data") == 0) {
	    levelref = LEVELREF_DATA;
	}
	gdome_str_unref(domstr); domstr = NULL;
	if (levelref == LEVELREF_NONE) {
	    goto NEXTELEM;
	}

	switch (levelref) {
	case LEVELREF_RESOURCE:
	    if (opt_resourceid != NULL) {
		valuestr = gdome_str_mkref(opt_resourceid);
		if (levelref == LEVELREF_RESOURCE) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	case LEVELREF_DATA:
	case LEVELREF_CATALOG:
	case LEVELREF_ANALYSIS:
	    if (opt_level != NULL) {
		valuestr = gdome_str_mkref(opt_level);
		if (levelref == LEVELREF_ANALYSIS) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else {
		    domstr = gdome_str_mkref("level");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	    if (opt_analysisid != NULL) {
		valuestr = gdome_str_mkref(opt_analysisid);
		if (levelref == LEVELREF_ANALYSIS) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else {
		    domstr = gdome_str_mkref("analysisID");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	case LEVELREF_ACQUISITION:
	    if (opt_acquisitionid != NULL) {
		valuestr = gdome_str_mkref(opt_acquisitionid);
		if (levelref == LEVELREF_ACQUISITION) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else {
		    domstr = gdome_str_mkref("acquisitionID");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	case LEVELREF_EPISODE:
	    if (opt_episodeid != NULL) {
		valuestr = gdome_str_mkref(opt_episodeid);
		if (levelref == LEVELREF_EPISODE) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else {
		    domstr = gdome_str_mkref("episodeID");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	case LEVELREF_STUDY:
	    if (opt_studyid != NULL) {
		valuestr = gdome_str_mkref(opt_studyid);
		if (levelref == LEVELREF_STUDY) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else {
		    domstr = gdome_str_mkref("studyID");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
		gdome_str_unref(valuestr);
	    }
	case LEVELREF_VISIT:
	    if (opt_visitid != NULL) {
		valuestr = gdome_str_mkref(opt_visitid);
		if (levelref == LEVELREF_VISIT) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else {
		    domstr = gdome_str_mkref("visitID");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	    if (opt_subjectgroupid != NULL) {
		valuestr = gdome_str_mkref(opt_subjectgroupid);
		domstr = gdome_str_mkref("subjectGroupID");
		gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		gdome_str_unref(domstr); domstr = NULL;
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	case LEVELREF_PROJECT:
	    if (opt_projectid != NULL) {
		valuestr = gdome_str_mkref(opt_projectid);
		if (levelref == LEVELREF_PROJECT) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else {
		    domstr = gdome_str_mkref("projectID");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	case LEVELREF_SUBJECT:
	    if (opt_subjectid != NULL) {
		valuestr = gdome_str_mkref(opt_subjectid);
		if (levelref == LEVELREF_SUBJECT) {
		    gdome_el_setAttribute(elem, idname, valuestr, &exc);
		} else if (levelref != LEVELREF_PROJECT) {
		    domstr = gdome_str_mkref("subjectID");
		    gdome_el_setAttribute(elem, domstr, valuestr, &exc);
		    gdome_str_unref(domstr); domstr = NULL;
		}
		gdome_str_unref(valuestr);
		if (exc != GDOME_NOEXCEPTION_ERR) {
		    goto FAIL;
		}
	    }
	}

NEXTELEM:
	node = gdome_n_nextSibling(node, &exc);
	gdome_el_unref(elem, &exc);
    }

    if (gdome_di_saveDocToMemory(di, doc, &buf, GDOME_SAVE_STANDARD, &exc) == FALSE) {
	fprintf(stderr, "DOMImplementation.saveDocToFile: Failed\n\tException #%d\n", exc);
	goto FAIL;
    }

    {
	FILE * outfp = NULL;
	FILE * newfp = NULL;
	if (argc == 2) {
	    outfp = stdout;
	} else {
	    if ((newfp = fopen(argv[2], "w")) == NULL) {
		fprintf(stderr, "ERROR opening file '%s' for writing.\n", argv[2]);
		perror("fopen");
		goto FAIL;
	    }
	    outfp = newfp;
	}
	fwrite(buf, strlen(buf), 1, outfp);
	if (newfp) {
	    fclose(newfp);
	}
    }


FAIL:
    retval = -1;

EXIT:
    if (buf) {
	free(buf);
	buf = NULL;
    }
    if (domstr) {
	gdome_str_unref(domstr);
	domstr = NULL;
    }
    if (idname) {
	gdome_str_unref(idname);
	idname = NULL;
    }
    if (xcede) {
	gdome_el_unref(xcede, &exc);
	xcede = NULL;
    }
    if (node) {
	gdome_n_unref(node, &exc);
	node = NULL;
    }
    if (doc) {
	gdome_doc_unref(doc, &exc);
	doc = NULL;
    }
    if (di) {
	gdome_di_unref(di, &exc);
	di = NULL;
    }
    return retval;
}
