package edu.masi.hyperadvisor.distancemetrics;

import java.util.ArrayList;
import java.util.regex.PatternSyntaxException;

import edu.masi.hyperadvisor.structures.Algo;
import edu.masi.hyperadvisor.util.Log;

/**
 * This class computes a Euclidean distance between JIST parameters.
 * It is designed to be called by the KNN class.
 * Note that determining a distance between some of these parameters is not straightforward.
 * 
 * @author covingkj
 *
 */
public class EuclideanDistance {
	
	/**
	 * Gets a Euclidean distance between two sets of inputs to the same algorithm
	 * 
	 * @param prevAlg		the input data from a previously recorded execution of this algorithm
	 * @param curAlg		the input data and class data from the proposed execution of this algorithm
	 * @return 				the distance between the two input sets
	 */
	public static float getDistance(Algo prevAlg, Algo curAlg)
	{
		float distance = 0;
		ArrayList<ArrayList<String>> prevInValues	= prevAlg.inputs;
		ArrayList<ArrayList<String>> curInValues 	= curAlg.inputs;
		ArrayList<String>			 curInClasses	= curAlg.classes;
		
		for(int i = 0; i < prevInValues.size(); i++){
			distance += getDistanceHelper(	prevInValues.get(i).toArray(new String [0]), 
											curInValues.get(i).toArray(new String [0]), 
											curInClasses.get(i));

		}
		return distance;
	}
		
	/**
	 * Helper function for the getDistance function. Allows collections to be enumerated through recursion.
	 * 
	 * @param prevVals	an input parameter from a previously recorded execution of this algorithm
	 * @param curVals	an input parameter from the current execution of this algorithm
	 * @param classes	the JIST parameter type of the current input, which determines how difference is measured
	 * @return the distance for an individual parameter
	 */
	private static float getDistanceHelper(String [] prevVals, String [] curVals, String classes){
		float distance = 0.0f;
		
		if(	classes.equalsIgnoreCase("ParamString")	||
			classes.equalsIgnoreCase("ParamOption")	||
			classes.equalsIgnoreCase("ParamBoolean")){
			String p = (String)prevVals[0];
			String c = (String)curVals[0];
			if(!p.equals(c)){
				distance += 1f;
			}
			
		} else if(classes.contains("ParamCollection")){
			try{
				String [] allClasses = classes.substring(classes.indexOf(";")+1).split(",");
				String [] prevArr 	= new String[1];
				String [] curArr 	= new String[1];			
				
				for(int i = 0; i < allClasses.length; i++){
					prevArr[0] 	= "0.0";
					curArr[0] 	= "0.0";
					if(i<prevVals.length)
						prevArr[0] 	= prevVals[i];
					if(i<curVals.length)
						curArr[0] 	= curVals[i];
					distance += getDistanceHelper(prevArr, curArr, allClasses[i]);
				}
			} catch (IndexOutOfBoundsException e){
				Log.writeToLog("Malformed class description for params of a ParamCollection variable.");
				Log.writeToLog(e.getMessage());
			} catch (PatternSyntaxException e){
				Log.writeToLog(e.getMessage());
			}
			
		} else if(	classes.equalsIgnoreCase("ParamNumber") ||
					classes.equalsIgnoreCase("ParamVolume")	||classes.equalsIgnoreCase("ParamInteger")){
			double p = Double.parseDouble(prevVals[0].toString());
			double c = Double.parseDouble(curVals[0].toString());
			distance += (float)(Math.abs((c - p)*2));
			
		} 
		else if(classes.equalsIgnoreCase("ParamVolumeCollection")){
			
			/*String [] prevArr 	= new String[1];
			String [] curArr 	= new String[1];
			for(int i = 0; i < prevVals.length; i++){
				prevArr[0] 	= prevVals[i];
				curArr[0] 	= curVals[i];
				distance += getDistanceHelper(prevArr, curArr, "ParamVolume");
			}*/
			//String [] prevArr 	= new String[1];
			//String [] curArr 	= new String[1];
			/*
			for(int i=0; i<prevVals.length; i++)
			{
				double p = Double.parseDouble(prevVals[i].toString());
				double c = Double.parseDouble(curVals[i].toString());
				distance+=(c-p)*2;
			}*/
			//String [] prevArr 	= new String[1];
			//String [] curArr 	= new String[1];
			int maxVal = Math.max(prevVals.length, curVals.length);
			double p;
			double c;
			for(int i = 0; i < maxVal; i++)
			{
				//prevArr[0] 	= prevVals[i];
				//curArr[0] 	= curVals[i];
				p=0;
				c=0;
				if(i<prevVals.length)
					p = Double.parseDouble(prevVals[i]);
				if(i<curVals.length)
					c = Double.parseDouble(curVals[i]);
				
				distance +=(float)(Math.abs((c - p)*2));

				//distance += getDistanceHelper(prevArr, curArr, "ParamNumber");
			}
			
		} else if(classes.equalsIgnoreCase("ParamFile")){

		} else if(classes.equalsIgnoreCase("ParamFileCollection")){

		} else if(classes.equalsIgnoreCase("ParamHidden")){
			
		} else if(classes.equalsIgnoreCase("ParamInformation")){

		} else if(classes.equalsIgnoreCase("ParamMatrix")){
			/*
			 * There's no intuitive comparison between two matrices that does not take into account what the matrices represent.
			 * Since we don't have this, 
			 * we will say if the two matrices are not equal distance += 1, else distance += 0
			 * This is identical to our function for ParamString, ParamOption and ParamBoolean.
			 * 
			 * The code is extendible enough that other distance metrics could be added in the future.
			 */
			String p = (String)prevVals[0];
			String c = (String)curVals[0];
			if(!p.equals(c)){
				distance += 1f;
			}
			
		} else if(classes.equalsIgnoreCase("ParamMipavDialog")){

		} else if(classes.equalsIgnoreCase("ParamMultiOption")){
			//String [] prevArr 	= new String[1];
			//String [] curArr 	= new String[1];
			int maxVal = Math.max(prevVals.length, curVals.length);
			for(int i = 0; i < maxVal; i++){
				//prevArr[0] 	= prevVals[i];
				//curArr[0] 	= curVals[i];
				if(i<prevVals.length && i<curVals.length)
				{
					if(!prevVals[i].equals(curVals[i]))
						distance += 1f;//getDistanceHelper(prevArr, curArr, "ParamOption");
				}
				else
				{
					distance+=1f;
				}
			}
			
		} else if(classes.equalsIgnoreCase("ParamNumberCollection")){

			int maxVal = Math.max(prevVals.length, curVals.length);
			double p;
			double c;
			for(int i = 0; i < maxVal; i++)
			{
				//prevArr[0] 	= prevVals[i];
				//curArr[0] 	= curVals[i];
				p=0;
				c=0;
				if(i<prevVals.length)
					p = Double.parseDouble(prevVals[i]);
				if(i<curVals.length)
					c = Double.parseDouble(curVals[i]);
				
				distance += (float)(Math.abs((c - p)*2));
				//Log.writeToLog("Numbers: "+p+","+c+","+distance);
				//distance += getDistanceHelper(prevArr, curArr, "ParamNumber");
			}
			
		} else if(classes.equalsIgnoreCase("ParamPerformance")){

		} else if(	classes.equalsIgnoreCase("ParamPointDouble")	||
					classes.equalsIgnoreCase("ParamPointFloat")		||
					classes.equalsIgnoreCase("ParamPointInteger")	){
			
			//Distance metric for points fits well with the Euclidean distance metric
			distance += (float)(Math.sqrt(	Math.pow(Double.parseDouble(curVals[0].toString()) - Double.parseDouble(prevVals[0].toString()),2) +
									Math.pow(Double.parseDouble(curVals[1].toString()) - Double.parseDouble(prevVals[1].toString()),2) +
									Math.pow(Double.parseDouble(curVals[2].toString()) - Double.parseDouble(prevVals[2].toString()),2)) );
			
		}
		return distance;
	}
}
