/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline;

import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

import edu.jhu.ece.iacl.jist.io.MipavController;

/**************
 * Store preferences for output viewer
 * @author Muqun Li(muqun.li@vanderbilt.edu)
 */
public class OutputViewerPreferences {

	public enum DataType {
	    VOLUME, SURFACE 
	}

	/**
	 * Listens for changes in preferences.
	 * 
	 * @author Muqun Li (muqun.li@vanderbilt.edu)
	 */
	public static interface PreferenceListener {

		/**
		 * History change.
		 * 
		 * @param prefs the preferences
		 */
		public void historyChange(OutputViewerPreferences viewerprefs);
	}
	/** Singleton reference to preferences. */
	private static OutputViewerPreferences viewerprefs = null;

	/** Default volume viewer. */
	protected String defVolViewer;

	/** Default surface viewer. */
	protected String defSurfViewer;
	
	/**
	 * Singleton method to get User Interface.
	 * 
	 * @return userInterface
	 */
	public static OutputViewerPreferences getPreferences() {
		if (viewerprefs == null) {
			viewerprefs = new OutputViewerPreferences();
		}
		return viewerprefs;
	}

	/**
	 * Returns true if MAPS has preferences already.
	 * 
	 * @return true if preferences exist
	 */
	public static boolean hasPreferences() {
		return (viewerprefs != null);
}
	/**
	 * Set singleton reference to preferences.
	 * 
	 * @param prefs preferences
	 */
	public static void setPreferences(OutputViewerPreferences viewerprefs) {
		OutputViewerPreferences.viewerprefs = viewerprefs;
	}


	public void setPreferredViewer(DataType dataType, String viewerType) {
		// TODO Auto-generated method stub
		
		switch(dataType)
		{
			case VOLUME:
				defVolViewer = viewerType;
				break;
			case SURFACE:
				defSurfViewer  = viewerType;
				break;
			default:
				break;
		}			
	}
	
	/**
	 * Open file chooser to select directory.
	 * 
	 * @param oldDir the old directory
	 * 
	 * @return absolute path of the file
	 */
	public static File askForCustomViewer() {
		JFileChooser openDialog = new JFileChooser();
		openDialog.setSelectedFile(MipavController.getDefaultWorkingDirectory());
		openDialog.setDialogTitle("Select custom viewer");
		openDialog.setFileSelectionMode(JFileChooser.FILES_ONLY);
		File jreFile = new File(JistPreferences.getPreferences().getJre());
		if (jreFile != null) {
			openDialog.setSelectedFile(jreFile);
		}
		openDialog.setDialogType(JFileChooser.OPEN_DIALOG);
		openDialog.setAcceptAllFileFilterUsed(false);
		openDialog.setFileFilter(new FileFilter() {
			public boolean accept(File file) {
				if (file.exists() && file.canExecute()) {
					return true;
				} else {
					return false;
				}
			}

			public String getDescription() {
				return "custom";
			}
		});
		int returnVal = openDialog.showOpenDialog(null);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			MipavController.setDefaultWorkingDirectory(openDialog.getSelectedFile().getParentFile());
			File f = openDialog.getSelectedFile();
			if (f.canExecute()) {
				return f;
			} else {
				return null;
			}
		} else {
			return null;
		}
	}

}
