/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.gui;

/**
 * Configure cloud
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingWorker;
import javax.swing.Timer;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.text.BadLocationException;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.MultipartPostMethod;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.profile.ProfileCredentialsProvider;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.AmazonEC2Client;
import com.amazonaws.services.ec2.model.DescribeSecurityGroupsResult;
import com.amazonaws.services.ec2.model.SecurityGroup;

import edu.jhu.ece.iacl.jist.io.MipavController;
import edu.jhu.ece.iacl.jist.pipeline.ExecutionContext;
import edu.jhu.ece.iacl.jist.pipeline.PipeLayout;
import edu.jhu.ece.iacl.jist.pipeline.PipeLayout.RunParameters;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.jistcloud.JistCloudExtension;
import edu.vanderbilt.masi.jistcloud.MyUtils;


public class CloudDashboardPanel extends JPanel implements ActionListener {	
    private static String experimentEndFlag = ">> END OF EXPERIMENT. <<";
    private static String switchServerFlag = ">> REDIRECT: ";
    private static String clusterSuccessFlag = ">> SUCCESS: cluster configuration finished";
    private static String downloadedZip = "/tmp/jist_out_downloaded.zip";
    
    private String downloadURL = "";
    private boolean downloadFinished = false;
    // if cluster Master is alive (yet)
    private boolean isMasterAlive = false;
    private String serverToQuery = "";
	private JTextField currentLayout, serverURL, cloudKey;
	
	private JTextArea serverStatus;
	
	JButton cloudKeyButton, createClusterButton, terminateClusterButton, cleanOutputButton,
			submitJobButton, queryButton, stopQueryingButton, getResultButton;
	JCheckBox useCloud;
	
	/** number of virtual machines, max job memory */
	private JSpinner numberOfInstances, maxJobMemory, maxHeapSize;
	
	/** The params. */
	private RunParameters params;
	
	/** The layout. */
	private PipeLayout layout;	
	
	// To standardize URL: remove trailing slash if any, then append one slash
	private String getServerURL() {
		return serverURL.getText().replaceAll("/$", "") + "/";
	}

	private static String selectKey(String oldKey) {
		JFileChooser loadDialog = new JFileChooser("Specify cloud key");
		File oldFile = new File(oldKey);
		
		if ( (!oldKey.isEmpty()) && oldFile.exists() ) {
			loadDialog.setCurrentDirectory(oldFile.getParentFile());
		} else {
			loadDialog.setCurrentDirectory(MipavController.getDefaultWorkingDirectory());
		}
		
		loadDialog.setDialogType(JFileChooser.OPEN_DIALOG);
		loadDialog.setFileSelectionMode(JFileChooser.FILES_ONLY);
		loadDialog.setFileFilter(new FileNameExtensionFilter("ZIP file", "zip"));
		loadDialog.removeChoosableFileFilter(loadDialog.getAcceptAllFileFilter());
		
		int returnVal = loadDialog.showOpenDialog(null);
		if ((returnVal == JFileChooser.APPROVE_OPTION) && loadDialog.getSelectedFile().getName().toLowerCase().endsWith(".zip")) {
			return loadDialog.getSelectedFile().getAbsolutePath();
		} else {
			return null;
		}
	}
		
    // Periodically query server for status report
	final Timer queryServer = new Timer(2000, new ActionListener() {	
		public void actionPerformed(ActionEvent e) {			
			// First check if cluster Master is alive			
			if ( !(serverToQuery.isEmpty() || isMasterAlive) ) {
				HttpURLConnection conn;
				
				try {
					conn = (HttpURLConnection) new URL(serverToQuery + "get-status").openConnection();
					conn.setRequestMethod("HEAD");
				
					// in milliseconds
					conn.setConnectTimeout(8000);
					conn.setReadTimeout(8000);
					
					if (conn.getResponseCode() == HttpURLConnection.HTTP_OK) {
						isMasterAlive = true;
					}
					
				} catch (Exception ex) {
					JistLogger.logError(JistLogger.SEVERE, "Error: " + ex.getMessage());
					return;
				}
			}
			
			
			// do actual update
			String tmpServerURL = serverToQuery.isEmpty() ? getServerURL() : serverToQuery;
            String serverResponse = "";

            HttpClient client = new HttpClient();
            GetMethod get = new GetMethod(tmpServerURL + "get-status");
            
            try {
                client.executeMethod(get);
                serverResponse = get.getResponseBodyAsString();
	            serverStatus.setText(serverResponse);
	
	            // Switch from querying Web to cluster Master
	            if (serverResponse.contains(switchServerFlag)) {
	            	serverToQuery = parseServerURL(serverResponse);
	            
	            } else if (serverResponse.contains(clusterSuccessFlag)) {
	            	queryServer.stop();
	            	
	            } else if (serverResponse.trim().endsWith(experimentEndFlag)) {	
		            // stop querying when experiment finishes
	                queryServer.stop();
	                
	                // Show download link
	                downloadURL = parseDownloadLink(serverResponse);
	                if (!downloadURL.isEmpty()) {
	                	getResultButton.setEnabled(true);
	                }
	            }
	            
            } catch (IOException e1) {
                e1.printStackTrace();
            } finally {
                get.releaseConnection();
            }

		}
	});
	
	// Extract URL from string of the form: >> REDIRECT: http://
	private static String parseServerURL(String str) {
		String parsedURL = "";
		String[] lines = str.split("\n");
		Pattern pattern = Pattern.compile(">>\\s*REDIRECT:\\s*http://\\s*(.+)$");

		for (String line: lines) {
			if (line.startsWith(switchServerFlag)) {
				Matcher m = pattern.matcher(line);
				
				if (m.find()) {
					parsedURL = m.group(1) + ":8080";
				}
			}
		}
		
		return parsedURL.isEmpty() ? "" : ("http://" + parsedURL + "/");
	}
	
	private static String parseDownloadLink(String text) {
		String parsedURL = "";
		String[] lines = text.split("\n");
		boolean found = false;
		
		for (String line: lines) {
			if (found) {
				parsedURL = line.trim();
				break;
			}
			
			if (line.startsWith(">> Final result URL <<:")) {
				found = true;
			}
		}

		return parsedURL;
	}

	/**
	 * Extract zip file to desired location
	 * @param zipFile zip file containing output
	 * @return true if everything works well; otherwise return false
	 */
	private boolean prepareOutput(String zipFile) {
		if ( zipFile.isEmpty() || (! new File(zipFile).exists()) ) {
			return false;
		}
		
		// then extract to corresponding folder
		String targetFolder = layout.getRunParameters().getOutputDirectory().getParent();
		
		try {
			MyUtils.extractFolder(zipFile, targetFolder);
			return true;
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		}
	}
	
	// Save configs for experiment
	private void saveConfigs() {
		if (layout == null) {
			return;
		}
		
		Properties prop = new Properties();
		try {
			prop.setProperty( "use_cloud", String.valueOf(useCloud.isSelected()) );
			prop.setProperty("server_url", getServerURL());
			prop.setProperty("max_job_memory", maxJobMemory.getValue().toString());

			prop.setProperty("current_layout", currentLayout.getText());
			prop.setProperty("cloud_credential", cloudKey.getText());
			prop.setProperty("total_number_of_vms", numberOfInstances.getValue().toString());
			prop.setProperty("master_url", serverToQuery);
			
			
			FileOutputStream fos = new FileOutputStream(MyUtils.getPropertiesPath(layout));
			prop.store(fos, null);
			
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * Show dialog.
	 * 
	 * @param comp
	 *            the comp
	 * @param layout
	 *            the layout
	 * @return true, if successful
	 * @throws BadLocationException 
	 */
	public static boolean showDialog(Component comp, PipeLayout layout) throws BadLocationException {
		CloudDashboardPanel panel = new CloudDashboardPanel(layout);

		Object[] options = {"Save", "Cancel"};
		int n = JOptionPane.showOptionDialog(comp, panel, "Cloud Dashboard [" + layout.getTitle() + "]",
				JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null, options, options[0]);
	
		// save configs
		if (n == 0) {
			JistLogger.logOutput(JistLogger.INFO, "> To save experiment-specific configs...");
			panel.saveConfigs();
			
		} else {
			return true;
		}
		
		return true;
	}


	/**
	 * Constructor.
	 * 
	 * @param layout
	 *            layout
	 * @throws BadLocationException 
	 */
	public CloudDashboardPanel(PipeLayout layout) throws BadLocationException {
		super();
		createPane();
		this.layout = layout;
		init(layout);
	}

    protected String uploadFilePath;

    // Upload data to Cluster
    final SwingWorker<Integer, Integer> uploadWorker = new SwingWorker<Integer, Integer>() {
        @Override
        protected final Integer doInBackground() throws Exception {
            serverStatus.append("\nSubmitting job to cloud cluster...");
            
            try {
                JistCloudExtension.uploadToServer(uploadFilePath, serverToQuery + "upload-data");
                serverStatus.append("\nFinished upload. Please check for job execution status.");
            } catch (Exception e) {
                serverStatus.append("\nErrors occurred during uploading.");
                e.printStackTrace();
            } finally {
            	submitJobButton.setEnabled(true);
            }

            // wait for server to clean up previous log
            Thread.sleep(1500);
			queryServer.start();

            return 0;
        }
    };
    
    private void createCluster() {
		
    	//File file = new File("/home/kktest/backup/cloud_Wei.zip");
    	File file = new File(cloudKey.getText());
    	
    	if (cloudKey.getText().isEmpty() || (!file.exists()) ) {
    		serverStatus.append("\nERROR: need to specify your cloud key.");
    		return;
    	}
    	
        serverStatus.setText("Waiting to create cluster...");
	
        HttpClient client = new HttpClient();
        MultipartPostMethod mPost = new MultipartPostMethod(getServerURL() + "create-cluster");

        try {
            JistLogger.logOutput(JistLogger.INFO, "> To create cluster...");
            mPost.addParameter("myFile", file);
            mPost.addParameter("instancesNumber", numberOfInstances.getValue().toString());
    		
            client.executeMethod(mPost);
            serverStatus.setText(mPost.getResponseBodyAsString());
            
            Thread.sleep(1000);
    		queryServer.start();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            mPost.releaseConnection();
        }
    }

    /**
     * Clean output directory (reduce upload overhead)
     */
    private void cleanOuput() {
    	if (layout == null) {
    		return;
    	}
    	
    	serverStatus.append("\nTo empty output directory for current experiment.....");
		File outputFolder = layout.getRunParameters().getOutputDirectory();
		
		for (File file : outputFolder.listFiles()) {
			if (file.isDirectory()) {
				if (file.getName().startsWith("exp-")) {
					ExecutionContext.deleteDir(file, null);
				}
			}
		}
		
    	serverStatus.append(" Finished.\n");
	}
    
    /**
     * Download output package from remote server
     * @param url URL to output package on remote server
     * @return Temporary location where downloaded zip file was stored
     */   
    final SwingWorker<Integer, Integer> downloadWorker = new SwingWorker<Integer, Integer>() {
        @Override
        protected final Integer doInBackground() throws Exception {
        	if (downloadURL.isEmpty()) {
        		serverStatus.append("\nDownload URL is not available.");
        		return -1;
        	}
        	
        	int result = 0;
           	HttpClient client = new HttpClient();
        	GetMethod get = new GetMethod(downloadURL);
        	InputStream inStream = null;
        	OutputStream outStream = null;
        	
            try {
            	File zipFile = new File(downloadedZip);
            	if (zipFile.exists()) {
            		zipFile.delete();
            	}
            	
            	serverStatus.append("\nTrying to download output package...");
                client.executeMethod(get);
                inStream = get.getResponseBodyAsStream();
                outStream = new FileOutputStream(zipFile);
                int read = 0;
                byte[] bytes = new byte[1024];
                
                while ( (read = inStream.read(bytes)) != -1 ) {
                	outStream.write(bytes, 0, read);
                }
            	serverStatus.append("\nFinished downloading. Will extract output to appropriate directory...");
            	
        	    if (prepareOutput(downloadedZip)) {
        	    	serverStatus.append("\nOutput saved in desired location.");
        	    } else {
        	    	serverStatus.append("\nERROR: failed to extract to output directory");
        	    	result = -1;
        	    }
        	    
            	// show process manager for output
        		ProcessManager pm = ProcessManager.getInstance();
    			pm.load(layout);
    			pm.setVisible(true);
            	
            } catch (IOException e1) {
                e1.printStackTrace();
                result = -1;
            } finally {
            	if (inStream != null) {
            		try {
    					inStream.close();
    				} catch (IOException e) {
    					e.printStackTrace();
    				}
            	}

            	if (outStream != null) {
            		try {
    					outStream.close();
    				} catch (IOException e) {
    					e.printStackTrace();
    				}
            	}
            	
                get.releaseConnection();
            }
            
        	return result;
        }
    };
    
    /**
     * Respond to user actions
     */
	public void actionPerformed(ActionEvent evt) {
		// clean experiment output folder
		if (evt.getSource() == cleanOutputButton) {
			cleanOuput();
			
		} else if (evt.getSource() == createClusterButton) {
			createClusterButton.setEnabled(false);
			terminateClusterButton.setEnabled(true);
			serverToQuery = "";
			createCluster();
			
		} else if (evt.getSource() == terminateClusterButton) {
			if (serverURL.getText().isEmpty()) {
				return;
			}
			
			terminateClusterButton.setEnabled(false);
			serverStatus.append("\nTo terminate cluster...\n");
			JistCloudExtension.terminateCluster(getServerURL() + "terminate-cluster");
			serverStatus.append("SUCCESS: cloud cluster is terminated.\n");
			createClusterButton.setEnabled(true);
			
		} else if (evt.getSource() == submitJobButton) {
			if (serverToQuery == null || (!serverToQuery.startsWith("http://")) ) {
				serverStatus.append("\nERROR: cloud cluster does NOT seem to be available.");
				return;
			}
			
			saveConfigs();
			
			submitJobButton.setEnabled(false);
			serverStatus.setText("");
            this.uploadFilePath = layout.getFileLocation().getAbsolutePath();

            uploadWorker.execute();

		} else if (evt.getSource() == cloudKeyButton){
			cloudKey.setText(selectKey(cloudKey.getText()));

		} else if (evt.getSource() == queryButton) {
			if (!queryServer.isRunning()) {
				queryServer.start();
			}
			
		} else if (evt.getSource() == stopQueryingButton) {
            if (queryServer.isRunning()) {
                queryServer.stop();
                serverStatus.append("\nStopped querying server.");
            }
            
        } else if (evt.getSource() == getResultButton) {
        	//DEBUG
        	//downloadURL = "http://jist.qqinput.com:8000/jist_user_download.zip";
        	
        	// Download zip result
        	try {
        		downloadWorker.execute();
    			
        	} catch (Exception ex) {
        		ex.printStackTrace();
        	}
        }
	}
	
	private void setStatesForButtons(boolean enabled) {
		serverURL.setEnabled(enabled);
		cloudKey.setEnabled(enabled);
		numberOfInstances.setEnabled(enabled);
		maxJobMemory.setEnabled(enabled);
		cloudKeyButton.setEnabled(enabled);
		createClusterButton.setEnabled(enabled);
		submitJobButton.setEnabled(enabled && serverToQuery != null && serverToQuery.startsWith("http://"));
		queryButton.setEnabled(enabled);
		stopQueryingButton.setEnabled(enabled);
	}

	/**
	 * Create panel to display preferences.
	 * @throws BadLocationException 
	 */
	protected void createPane() throws BadLocationException {
		int TEXT_WIDTH = 50;
		int SMALL_TEXT_WIDTH = TEXT_WIDTH - 10;

		this.setLayout(new BorderLayout());
		JPanel small = new JPanel();
		BoxLayout layout = new BoxLayout(small, BoxLayout.PAGE_AXIS);
		small.setLayout(layout);
		this.add(small, BorderLayout.NORTH);
        JScrollPane jsp;
		
		JPanel itemPane = new JPanel(new BorderLayout());
		
		// Use cloud facility
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Use cloud servers? "), BorderLayout.WEST);
		itemPane.add(useCloud = new JCheckBox("",false), BorderLayout.EAST);
		small.add(itemPane);
		
		// show layout path
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Layout "), BorderLayout.WEST);
		currentLayout = new JTextField(SMALL_TEXT_WIDTH);
		currentLayout.setEnabled(false);
		itemPane.add(currentLayout, BorderLayout.EAST);
		small.add(itemPane);
		
		// clean experiment output
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Clean experiment output folder"), BorderLayout.WEST);
		itemPane.add(cleanOutputButton = new JButton("Clean All"), BorderLayout.EAST);
		small.add(itemPane);
		
		small.add(new JPanel());

		// set remote server URL
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Server URL "), BorderLayout.WEST);
		itemPane.add(serverURL = new JTextField(SMALL_TEXT_WIDTH), BorderLayout.EAST);
		serverURL.setText("http://jist.qqinput.com:8080");
		small.add(itemPane);
		
		// upload cloud credential
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Cloud credential "), BorderLayout.WEST);
		itemPane.add(cloudKey = new JTextField(SMALL_TEXT_WIDTH - 8), BorderLayout.CENTER);
		itemPane.add(cloudKeyButton = new JButton("Browse"), BorderLayout.EAST);
		small.add(itemPane);
		
		// add space line
		small.add(new JPanel());
		
		// specify number of VM's
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Total number of virtual machines "), BorderLayout.WEST);
		itemPane.add(numberOfInstances = new JSpinner(new SpinnerNumberModel(3, 1, 100, 1)), BorderLayout.EAST);
		small.add(itemPane);		
		
		// specify max job memory
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Max Job Memory "), BorderLayout.WEST);
		itemPane.add(maxJobMemory = new JSpinner(new SpinnerNumberModel(500, 100, 64000, 200)), BorderLayout.EAST);
		small.add(itemPane);
		
		// specify max heap size
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Max heap size "), BorderLayout.WEST);
		itemPane.add(maxHeapSize = new JSpinner(new SpinnerNumberModel(500, 100, 64000, 200)), BorderLayout.EAST);
		maxHeapSize.setEnabled(false);
		small.add(itemPane);
		
		// add space line
		small.add(new JPanel());

		itemPane = new JPanel(new GridBagLayout());
		GridBagConstraints c = new GridBagConstraints();
		c.fill = GridBagConstraints.HORIZONTAL;
		
		createClusterButton = new JButton("Create Cluster");
		c.gridx = 0;
		c.gridy = 0;
		itemPane.add(createClusterButton, c);

        terminateClusterButton = new JButton("Terminate Cluster");
        c.gridx++;
        itemPane.add(terminateClusterButton, c);
		
        submitJobButton = new JButton("Submit Job");
        c.gridx++;
        itemPane.add(submitJobButton, c);
        
		small.add(itemPane);
		
		// add space line
		small.add(new JPanel());
		
		itemPane = new JPanel(new GridBagLayout());
		c = new GridBagConstraints();
		c.fill = GridBagConstraints.HORIZONTAL;
		
        queryButton = new JButton("Server Status");
        c.gridx++;
        itemPane.add(queryButton, c);
        
        stopQueryingButton = new JButton("Stop Refresh");
        c.gridx++;
        itemPane.add(stopQueryingButton, c);
        
        small.add(itemPane);

        // show server status
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Server status:"), BorderLayout.WEST);
		small.add(itemPane);
		
		itemPane = new JPanel(new BorderLayout());
		serverStatus = new JTextArea(15, TEXT_WIDTH);
		serverStatus.setCaretPosition(serverStatus.getDocument().getLength());
		jsp = new JScrollPane(serverStatus);
		itemPane.add(jsp, BorderLayout.EAST);
		small.add(itemPane);
        
        // Copy download link to clipboard
        itemPane = new JPanel(new BorderLayout());
        itemPane.add(getResultButton = new JButton("Get Result"), BorderLayout.EAST);
        //DEBUG
        //getResultButton.setEnabled(true);
        getResultButton.setEnabled(false);
        small.add(itemPane);
        
		// actions
		// Enable more configs when using cloud
		useCloud.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				setStatesForButtons(useCloud.isSelected());
			}
		});
		setStatesForButtons(useCloud.isSelected());
		cleanOutputButton.addActionListener(this);
		createClusterButton.addActionListener(this);
		terminateClusterButton.addActionListener(this);
		submitJobButton.addActionListener(this);
		queryButton.addActionListener(this);
        stopQueryingButton.addActionListener(this);
        getResultButton.addActionListener(this);
        cloudKeyButton.addActionListener(this);
	}

	/**
	 * Initialize preference panel for layout.
	 * 
	 * @param layout
	 *            the layout
	 * @throws InterruptedException 
	 */
	protected void init(PipeLayout layout) {
		params = layout.getRunParameters();
		//cloudKey.setText((params.getInputURI() != null) ? params.getInputURI().toString() : "");
		currentLayout.setText(layout.getFileLocation().getAbsolutePath());
		
		// load experiment-specific configs
		FileInputStream fis = null;
		Properties prop = new Properties();
		try {
			String propertiesPath = MyUtils.getPropertiesPath(layout);
			MyUtils.preparePropertiesFile(propertiesPath);
			fis = new FileInputStream(propertiesPath);
			prop.load(fis);
			
			Boolean useCloudFlag = (prop.getProperty("use_cloud") == null)? false : Boolean.parseBoolean(prop.getProperty("use_cloud"));
			useCloud.setSelected(useCloudFlag);
			if (prop.getProperty("server_url") != null) {
				serverURL.setText(prop.getProperty("server_url"));
			}
			
			serverToQuery = (prop.getProperty("master_url") == null) ? "" : prop.getProperty("master_url");
			cloudKey.setText( (prop.getProperty("cloud_credential") == null)? "" : prop.getProperty("cloud_credential") );

			int tmpMaxJobMemory = 0;
			if (prop.getProperty("max_job_memory") == null) {
				tmpMaxJobMemory = params.getMaxHeap() + 300;
			} else {
				tmpMaxJobMemory = Integer.parseInt(prop.getProperty("max_job_memory"));
			}
			maxJobMemory.setValue(tmpMaxJobMemory);
			
			maxHeapSize.setValue(params.getMaxHeap());			
			cloudKey.setText(prop.getProperty("cloud_credential"));
			if (prop.getProperty("total_number_of_vms") != null) {
				numberOfInstances.setValue(Integer.parseInt(prop.getProperty("total_number_of_vms")));
			}
			
			setStatesForButtons(useCloudFlag);
			
		} catch (Exception ex) {
			ex.printStackTrace();
			
		} finally {
			if (fis != null) {
				try {
					fis.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		
	}
		
}
