/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JSplitPane;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.SwingWorker;
import javax.swing.WindowConstants;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.DescribeInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeInstancesResult;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.transfer.MultipleFileUpload;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;

import edu.jhu.ece.iacl.jist.io.MipavController;
import edu.jhu.ece.iacl.jist.pipeline.ExecutionContext;
import edu.jhu.ece.iacl.jist.pipeline.ExecutionContext.Status;
import edu.jhu.ece.iacl.jist.pipeline.JistPreferences;
import edu.jhu.ece.iacl.jist.pipeline.PerformanceSummary;
import edu.jhu.ece.iacl.jist.pipeline.PipeLayout;
import edu.jhu.ece.iacl.jist.pipeline.PipeLibrary;
import edu.jhu.ece.iacl.jist.pipeline.PipeScheduler;
import edu.jhu.ece.iacl.jist.pipeline.gui.resources.PlaceHolder;
import edu.jhu.ece.iacl.jist.pipeline.tree.ProcessNode;
import edu.jhu.ece.iacl.jist.utility.FileUtil;
import edu.jhu.ece.iacl.jist.utility.HelpUtil;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.jistcloud.JistAwsEC2;
import edu.vanderbilt.masi.jistcloud.JistAwsS3;
import edu.vanderbilt.masi.jistcloud.JistAwsUtil;
import edu.vanderbilt.masi.jistcloud.JistCloudExtension;
import edu.vanderbilt.masi.jistcloud.MyUtils;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;

/**
 * Process manager GUI to manage the execution of experiments.
 * 
 * @author Blake Lucas (bclucas@jhu.edu) <br>
 *         Muqun Li(muqun.li@vanderbilt.edu)
 */
public class ProcessManager extends JFrame implements Runnable, ActionListener,
		WindowListener, JistPreferences.PreferenceListener {
	protected String PMTitle = "Title Not Set";
	protected boolean useWorker = true;
	protected boolean managerVisible = true;

	public boolean isUsingWorker() {
		return useWorker;
	}

	public void setShowManager(boolean vis) {
		this.managerVisible = vis;
	}

	public void setUseWorker(boolean useWorker) {
		this.useWorker = useWorker;
	}

	/**
	 * Worker to load layout.
	 * 
	 * @author Blake Lucas (bclucas@jhu.edu)
	 */
	protected class LoadLayout extends SwingWorker<Void, Void> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.SwingWorker#doInBackground()
		 */
		public Void doInBackground() {
			try {
				for (JMenuItem item : toggleItems) {
					item.setEnabled(false);
				}
				scheduler.init();
				for (JMenuItem item : toggleItems) {
					item.setEnabled(true);
				}
				// Do not process destinations on load! It takes too long!
				// scheduler.processDestinations();
				stopAllItem.setEnabled(true);
				pauseAllItem.setEnabled(false);
				runAllItem.setEnabled(true);
				closeItem.setEnabled(true);
				reloadItem.setEnabled(true);
				prefItem.setEnabled(true);
				saveAsItem.setEnabled(true);
			} catch (Exception e) {
				e.printStackTrace();
			}
			return null;
		}
	}

	/** The manager. */
	protected static ProcessManager manager = null;

	/** The streams. */
	protected static PrintStream[] streams = null;

	/**
	 * Get singleton reference.
	 * 
	 * @return process manager
	 */
	public static ProcessManager getInstance() {
		if (manager == null) {
			manager = new ProcessManager();
		}
		manager.refresh();
		return manager;
	}

	public void refresh() {
		if (updater == null || !updater.isAlive()) {
			updater = null;
			updater = new Thread(this);
			updater.setName("Process Status Updater");
			updater.start();
		}
	}

	/**
	 * The main method.
	 * 
	 * @param args
	 *            the arguments
	 */
	public static void main(String[] args) {
		MipavController.setQuiet(true);
		MipavController.init();
		if (args.length > 1) {
			Date today;
			String output;
			SimpleDateFormat formatter;
			formatter = new SimpleDateFormat("yyyy-MM-dd-kkmmssz");
			today = new Date();
			output = formatter.format(today);
			File debugOut = new File(args[0] + ".debug." + output + ".out");
			File debugErr = new File(args[0] + ".debug." + output + ".err");
			streams = FileUtil.redirect(debugOut, debugErr);
		}
		ProcessManager frame = getInstance();
		frame.setVisible(frame.managerVisible);
		frame.addWindowListener(frame);
		// frame.library = PipeLibrary.getInstance();
		for (String element : args) {
			File f = new File(element);
			if (f.exists() && f.isDirectory()) {
				PipeLibrary.getInstance().setLibraryPath(f);
			}
		}
		PipeLibrary.getInstance().loadPreferences();
		JistPreferences.getPreferences().addListener(frame);
		frame.historyChange(JistPreferences.getPreferences());
		for (String element : args) {
			File f = new File(element);
			if (f.exists() && !f.isDirectory()) {
				frame.open(f);
			}
		}
	}

	public void run(File f) {
		System.out.println(f.toPath());
		run(PipeLayout.read(f), false);
	}

	public boolean runAndWait(PipeLayout layout, boolean outOfProcessOverride) {

		setExitOnClose(false);
		setUseWorker(false);
		if (layout != null) {
			layout.init();
			initScheduler(layout); // Fix to NITRC Bug 3934
		}
		// if(outOfProcessOverride)
		// scheduler.setOverrideRunOutOfProcess(true);
		run(layout, outOfProcessOverride);
		try {
			while (!scheduler.isCompleted()) {
				Thread.sleep(1000);
			}
		} catch (InterruptedException e) {
			JistLogger.logError(JistLogger.WARNING, getClass()
					.getCanonicalName()
					+ "\t"
					+ "ProcessManager runAndWait interrupted.");
			JistLogger.logError(JistLogger.WARNING, getClass()
					.getCanonicalName() + e.getMessage());
		}
		return (scheduler.getWaitingCount() == 0 && scheduler.getFailedCount() == 0);
	}

	public void forceQuit() {
		quit(true);
	}

	public void run(PipeLayout layout, boolean outOfProcessOverride) {
		if (this.layout != null)
			close();
		this.layout = layout;
		layout.init();
		setVisible(managerVisible);
		addWindowListener(this);
		// library = PipeLibrary.getInstance();
		PipeLibrary.getInstance().loadPreferences();
		JistPreferences.getPreferences().addListener(this);
		historyChange(JistPreferences.getPreferences());
		String title = layout.getTitle();
		if (title != null) {
			this.setTitle(PMTitle + " [" + PipeLibrary.getHostName() + " - "
					+ layout.getTitle() + "]");
		} else {
			this.setTitle(PMTitle + " [" + PipeLibrary.getHostName() + "]");
		}
		if (scheduler != null) {
			scheduler.dispose();
		}
		scheduler = new PipeScheduler(layout, this);

		scheduler.setOverrideRunOutOfProcess(outOfProcessOverride);
		managerPane.setScheduler(scheduler);
		for (JMenuItem item : toggleItems) {
			item.setEnabled(false);
		}
		scheduler.init();
		for (JMenuItem item : toggleItems) {
			item.setEnabled(true);
		}
		// Do not process destinations on load! It takes too long!
		// scheduler.processDestinations();
		stopAllItem.setEnabled(true);
		pauseAllItem.setEnabled(false);
		runAllItem.setEnabled(true);
		closeItem.setEnabled(true);
		reloadItem.setEnabled(true);
		prefItem.setEnabled(true);
		saveAsItem.setEnabled(true);
		scheduler.enqueue();
		if (!scheduler.isRunning())
			scheduler.start();
	}

	/** The manager pane. */
	public ProcessManagerTable managerPane;

	/** The ancestors pane. */
	public ProcessInfoPanel ancestorsPane;

	/** The descendants pane. */
	public ProcessInfoPanel descendantsPane;

	/** The scheduler. */
	PipeScheduler scheduler = null;

	public PipeScheduler getCurrentScheduler() {
		return scheduler;
	}

	/** The layout. */
	PipeLayout layout;

	/** The status bar. */
	JLabel statusBar;

	/** The updater. */
	Thread updater;

	/** The library. */
	// public PipeLibrary library;

	/** The update running. */
	boolean updateRunning;

	/** The recent files. */
	private JMenu recentFiles = null;

	/** The recent items. */
	Vector<JMenuItem> recentItems = new Vector<JMenuItem>();

	/** The toggle items. */
	ArrayList<JMenuItem> toggleItems;

	/** The jre item. */
	JMenuItem runAllItem, destItem, stopAllItem, openItem, pauseAllItem,
			quitItem, prefItem, closeItem, saveAsItem, reloadItem,
			cleanAllItem, cleanOutOfSyncItem, globalItem, aboutItem,
			helpTopicItem, sendProjectInputToCloudItem;

	// Validation tools
	public JMenuItem bugReportItem;

	/** The exit on close. */
	protected boolean exitOnClose = true;

	/**
	 * Default constructor.
	 */
	public ProcessManager() {
		super();

		PMTitle = "JIST Process Manager " + "(v"
				+ license.JISTLicenseAbout.getReleaseID() + "-"
				+ license.JISTLicenseAbout.getReleaseBuildDate() + ")";

		managerPane = new ProcessManagerTable();
		JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
		splitPane.setTopComponent(managerPane);
		splitPane.setOneTouchExpandable(true);
		splitPane.setResizeWeight(0.5);
		ancestorsPane = new ProcessInfoPanel(managerPane,
				ProcessNode.Hierarchy.Ancestors);
		descendantsPane = new ProcessInfoPanel(managerPane,
				ProcessNode.Hierarchy.Descendants);
		JSplitPane infoSplit = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		infoSplit.setLeftComponent(ancestorsPane);
		infoSplit.setRightComponent(descendantsPane);
		infoSplit.setDividerLocation(1024 / 2 - 10);
		splitPane.setBottomComponent(infoSplit);
		splitPane.setDividerLocation(200);
		splitPane.setResizeWeight(0.5);
		setPreferredSize(new Dimension(1024, 500));
		this.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				quit();
			}
		});
		pack();
		this.setTitle(PMTitle);
		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(managerPane.createToolBar(), BorderLayout.NORTH);
		getContentPane().add(statusBar = new JLabel(" "), BorderLayout.SOUTH);
		getContentPane().add(splitPane, BorderLayout.CENTER);
		setJMenuBar(createMenuBar());
		statusBar.setHorizontalAlignment(SwingConstants.RIGHT);
		statusBar.setBorder(BorderFactory.createLoweredBevelBorder());
		updater = new Thread(this);
		updater.setName("Process Status Updater");
		updater.start();

		setIconImage(Toolkit.getDefaultToolkit().getImage(
				PlaceHolder.class.getResource("Logo_ProjectManager.png")));
		/** SET ICON HERE **/

		// System.out.println("FIND INITIAL PROCESSMANAGER");

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	@SuppressWarnings("deprecation")
	public void actionPerformed(ActionEvent evt) {
		if (evt.getSource() == runAllItem) {
			if (scheduler != null) {
				scheduler.enqueue();

				if (!scheduler.isRunning())
					scheduler.start();
			}
			stopAllItem.setEnabled(true);
			pauseAllItem.setEnabled(true);
			cleanAllItem.setEnabled(true);
			cleanOutOfSyncItem.setEnabled(true);
		} else if (evt.getSource() == stopAllItem) {
			if (scheduler != null) {
				scheduler.killAll();
			}
			pauseAllItem.setEnabled(false);
		} else if (evt.getSource() == pauseAllItem) {
			if (scheduler != null) {
				if (scheduler.isRunning())
					scheduler.stop();
			}
			pauseAllItem.setEnabled(false);
			runAllItem.setEnabled(true);
		} else if (evt.getSource() == quitItem) {
			quit();
		} else if (evt.getSource() == openItem) {
			open();
		} else if (evt.getSource() == closeItem) {
			if (layout != null) {
				close();
			}
		} else if (evt.getSource() == reloadItem) {
			if (layout != null) {
				File f = layout.getFileLocation();
				close();
				open(f);
				refresh();
			}
		} else if (evt.getSource() == saveAsItem) {
			if (layout != null) {
				layout.saveAs(false);
			}
		} else if (evt.getSource() == prefItem) {
			if (layout != null) {
				if (LayoutPreferencePanel.showDialog(this, layout)) {
					lastModifiedLock = true;
					layout.save(false);
					lastModifiedLayoutTime = layout.getFileLocation()
							.lastModified();
					lastModifiedLock = false;
					scheduler.killAll();
				}
			}
		} else if (evt.getSource() == destItem) {
			if (scheduler != null) {
				scheduler.processDestinations();
			}
		} else if (evt.getSource() == cleanAllItem) {
			int n = JOptionPane
					.showOptionDialog(
							this,
							"Are you sure you want to delete ALL existing experiment folders in "
									+ layout.getRunParameters()
											.getOutputDirectory()
											.getAbsolutePath()
									+ " and ALL its sub-directories? \n"
									+ "Warning: This may delete data that exists in the output directory that was not created by this JIST layout.",
							null, JOptionPane.YES_NO_OPTION,
							JOptionPane.WARNING_MESSAGE, null, null, null);
			if (n == 0) {
				scheduler.cleanAll();

			}
		} else if (evt.getSource() == cleanOutOfSyncItem) {
			int n = JOptionPane
					.showOptionDialog(
							this,
							"Are you sure you want to delete ALL out-of-sync experiment folders in "
									+ layout.getRunParameters()
											.getOutputDirectory()
											.getAbsolutePath()
									+ " and ALL their sub-directories? \n"
									+ "Warning: This may delete data that exists in the output directory that was not created by this JIST layout.",
							null, JOptionPane.YES_NO_OPTION,
							JOptionPane.WARNING_MESSAGE, null, null, null);
			if (n == 0) {
				Vector<ExecutionContext> outOfSyncContexts = new Vector<ExecutionContext>();
				for (ExecutionContext context : scheduler.getSchedule())
					if (context.getStatus() == Status.OUT_OF_SYNC)
						outOfSyncContexts.add(context);
				scheduler.clean(outOfSyncContexts, true);
			}
		} else if (recentItems.contains(evt.getSource())) {
			open(new File(((JMenuItem) evt.getSource()).getText()));
		} else if (evt.getSource() == globalItem) {
			if (GlobalPreferencePanel.showDialog(this,
					JistPreferences.getPreferences())) {
			}
		} else if (evt.getSource() == aboutItem) {
			license.JISTLicenseAbout.showDialog();
		} else if (evt.getSource() == helpTopicItem) {
			HelpUtil.showHelp(null);
		} else if (evt.getSource() == bugReportItem) {
			JISTValidationGUI report = new JISTValidationGUI();
			report.fileBugReport(managerPane.getSelectedContexts());
		} else if (((JMenuItem) evt.getSource()).getText().equalsIgnoreCase(
				"Save Screen Capture...")) {
			JistLogger.savePrintScreen(this);
		} else if (evt.getSource() == sendProjectInputToCloudItem) {
			try {
				initAWS();
			} catch (AmazonServiceException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			} catch (AmazonClientException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			} catch (InterruptedException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
		}

	}

	/**
	 * Close process manager and clean up resources.
	 */
	public void close() {
		this.setTitle(PMTitle);
		closeItem.setEnabled(false);
		runAllItem.setEnabled(false);
		stopAllItem.setEnabled(false);
		pauseAllItem.setEnabled(false);
		saveAsItem.setEnabled(false);
		prefItem.setEnabled(false);
		reloadItem.setEnabled(false);
		cleanAllItem.setEnabled(false);
		cleanOutOfSyncItem.setEnabled(false);
		if (scheduler != null)
			scheduler.killAll();
		if (!hideOnExit) {
			// layout.dispose();
			scheduler = null;
			layout = null;
		}
		managerPane.removeScheduler();
		ancestorsPane.updateTree(null);
		descendantsPane.updateTree(null);
		managerPane.getOutFrame().setVisible(false);
	}

	/**
	 * Create menubar.
	 * 
	 * @return the j menu bar
	 */
	public JMenuBar createMenuBar() {
		toggleItems = new ArrayList<JMenuItem>();
		JMenuBar menuBar = new JMenuBar();
		JMenu menuFile = new JMenu("File");
		menuFile.setMnemonic(KeyEvent.VK_F);
		menuBar.add(menuFile);
		toggleItems.add(openItem = new JMenuItem("Open"));
		openItem.setMnemonic(KeyEvent.VK_O);
		openItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O,
				ActionEvent.CTRL_MASK));
		openItem.addActionListener(this);
		menuFile.add(openItem);
		recentFiles = new JMenu("Recent Files");
		menuFile.add(recentFiles);
		toggleItems.add(saveAsItem = new JMenuItem("Save As"));
		saveAsItem.setMnemonic(KeyEvent.VK_A);
		saveAsItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,
				ActionEvent.CTRL_MASK));
		saveAsItem.addActionListener(this);
		menuFile.add(saveAsItem);
		toggleItems.add(reloadItem = new JMenuItem("Reload"));
		reloadItem.setMnemonic(KeyEvent.VK_C);
		reloadItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R,
				ActionEvent.CTRL_MASK));
		reloadItem.addActionListener(this);
		menuFile.add(reloadItem);
		toggleItems.add(closeItem = new JMenuItem("Close"));
		closeItem.setMnemonic(KeyEvent.VK_C);
		closeItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_W,
				ActionEvent.CTRL_MASK));
		closeItem.addActionListener(this);
		menuFile.add(closeItem);
		toggleItems.add(prefItem = new JMenuItem("Layout Preferences"));
		prefItem.setMnemonic(KeyEvent.VK_P);
		prefItem.addActionListener(this);
		menuFile.add(prefItem);
		toggleItems.add(globalItem = new JMenuItem("Global Preferences"));
		globalItem.addActionListener(this);
		menuFile.add(globalItem);
		// Shunxing Edit
		toggleItems.add(sendProjectInputToCloudItem = new JMenuItem(
				"Send Project to Cloud"));
		sendProjectInputToCloudItem.addActionListener(this);
		menuFile.add(sendProjectInputToCloudItem);
		// Shunxing Edit
		JMenuItem menuItem;
		toggleItems.add(menuItem = new JMenuItem("Save Screen Capture..."));
		menuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P,
				ActionEvent.CTRL_MASK));
		menuItem.setMnemonic(KeyEvent.VK_P);
		menuItem.addActionListener(this);
		menuFile.add(menuItem);

		toggleItems.add(quitItem = new JMenuItem("Quit"));
		quitItem.setMnemonic(KeyEvent.VK_Q);
		quitItem.addActionListener(this);
		quitItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q,
				ActionEvent.CTRL_MASK));
		quitItem.setMnemonic(KeyEvent.VK_Q);
		menuFile.add(quitItem);
		JMenu menuAction = new JMenu("Scheduler");
		menuAction.setMnemonic(KeyEvent.VK_S);
		menuBar.add(menuAction);
		toggleItems.add(runAllItem = new JMenuItem("Start Scheduler"));
		runAllItem.addActionListener(this);
		menuAction.add(runAllItem);
		toggleItems.add(pauseAllItem = new JMenuItem(
				"Finish Queued / Stop Enqueue"));
		pauseAllItem.addActionListener(this);
		menuAction.add(pauseAllItem);
		toggleItems.add(destItem = new JMenuItem("Process Destinations"));
		destItem.addActionListener(this);
		menuAction.add(destItem);
		toggleItems.add(stopAllItem = new JMenuItem("Stop All"));
		stopAllItem.addActionListener(this);
		menuAction.add(stopAllItem);
		toggleItems.add(cleanAllItem = new JMenuItem("Clean All"));
		cleanAllItem.addActionListener(this);
		menuAction.add(cleanAllItem);
		toggleItems
				.add(cleanOutOfSyncItem = new JMenuItem("Clean Out of Sync"));
		cleanOutOfSyncItem.addActionListener(this);
		menuAction.add(cleanOutOfSyncItem);

		JMenu menuActionHelp = new JMenu("Help");
		menuActionHelp.setMnemonic(KeyEvent.VK_H);
		menuBar.add(menuActionHelp);
		toggleItems.add(aboutItem = new JMenuItem("About"));
		aboutItem.addActionListener(this);
		menuActionHelp.add(aboutItem);

		menuActionHelp.add(this.helpTopicItem = new JMenuItem(
				"JIST help topics"));
		helpTopicItem.addActionListener(this);
		menuActionHelp.add(this.bugReportItem = new JMenuItem(
				"Submit a bug report"));
		bugReportItem.addActionListener(this);

		closeItem.setEnabled(false);
		runAllItem.setEnabled(false);
		stopAllItem.setEnabled(false);
		cleanAllItem.setEnabled(true);
		cleanOutOfSyncItem.setEnabled(true);
		pauseAllItem.setEnabled(false);
		prefItem.setEnabled(false);
		saveAsItem.setEnabled(false);
		reloadItem.setEnabled(false);
		return menuBar;
	}

	/**
	 * Update menus if history changed.
	 * 
	 * @param prefs
	 *            the prefs
	 */
	public void historyChange(JistPreferences prefs) {
		Vector<File> fileHistory = prefs.getFileHistory();
		for (JMenuItem item : recentItems) {
			item.removeActionListener(this);
		}
		recentItems.clear();
		JMenuItem item;
		recentFiles.removeAll();
		int i = 1;
		for (File f : fileHistory) {
			recentItems.add(item = new JMenuItem(f.getAbsolutePath()));
			item.addActionListener(this);
			if (i < 10) {
				item.setAccelerator(KeyStroke.getKeyStroke('0' + i,
						ActionEvent.CTRL_MASK));
			}
			recentFiles.add(item);
			i++;
		}
	}

	/**
	 * Initlaize manager.
	 */
	/*
	 * public void init() { setVisible(managerVisible); addWindowListener(this);
	 * library = PipeLibrary.getInstance(); library.loadPreferences();
	 * JistPreferences.getPreferences().addListener(this);
	 * historyChange(JistPreferences.getPreferences()); }
	 */
	public void initScheduler(PipeLayout layout) {
		this.setTitle(PMTitle + " [" + PipeLibrary.getHostName() + " - "
				+ layout.getTitle() + "]");
		if (scheduler != null) {
			scheduler.dispose();
		}
		lastModifiedLayoutTime = layout.getLastModifiedTime();
		scheduler = new PipeScheduler(layout, this);
		scheduler.setUseWorker(useWorker);
		managerPane.setScheduler(scheduler);
		this.layout = layout;

	}

	/**
	 * Load layout.
	 * 
	 * @param layout
	 *            layout
	 */
	public void load(PipeLayout layout) {
		if (layout == null) {
			return;
		}
		File outputDir = layout.getRunParameters().getOutputDirectory();
		if (!outputDir.exists()) {
			int n = JOptionPane.showOptionDialog(this, "The output directory "
					+ outputDir.getAbsolutePath()
					+ " could not be found. Would you like to change it now?",
					null, JOptionPane.YES_NO_OPTION,
					JOptionPane.WARNING_MESSAGE, null, null, null);
			if (n == 0) {
				if (LayoutPreferencePanel.showDialog(this, layout)) {
					lastModifiedLock = true;
					layout.save(false);
					lastModifiedLayoutTime = layout.getFileLocation()
							.lastModified();
					lastModifiedLock = false;
				}
			}
		}
		initScheduler(layout);
		LoadLayout loader = new LoadLayout();
		if (useWorker) {
			loader.execute();
		} else {
			loader.doInBackground();
		}
	}

	/**
	 * Open layout from file.
	 */
	public void open() {
		PipeLayout layout = PipeLayout.open(this);
		if (layout != null) {
			layout.init();
		}

		load(layout);
		// Shunxing edit
		System.out.println(layout.getFileLocation());
		System.out.println(layout.getFileLocation().getParent());
		File a = new File(layout.getFileLocation().getParent());
		System.out.println(a.getParent());
		// Shunxing edit
	}

	/**
	 * Open specified layout file.
	 * 
	 * @param f
	 *            file
	 */
	public void open(File f) {
		PipeLayout layout = PipeLayout.read(f);
		if (layout != null) {
			layout.init();
			load(layout);
		}
	}

	public boolean quit() {
		return quit(false);
	}

	/**
	 * Quit process manager.
	 * 
	 * @return true, if quit
	 */
	public boolean quit(boolean forceQuit) {

		JLabel msg = new JLabel(
				"<html>Are you sure that you want to close the JIST Process Manager?<br><br>Note: All running jobs will be killed.</html>",
				JLabel.CENTER);
		int n = (!forceQuit) ? JOptionPane.showOptionDialog(this, msg, null,
				JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE, null,
				null, null) : 0;

		if (n == 0) {

			// make this invisible
			this.dispose();

			// make sure we stop the update
			updateRunning = false;

			// try to stop the updater
			try {
				if (updater != null && updater.isAlive()) {
					updater.interrupt();
					updater.join(200);
					if (updater.isAlive()) {
						JistLogger.logError(JistLogger.WARNING, getClass()
								.getCanonicalName()
								+ "ProcessManager join failed.");
					}
				}
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				JistLogger.logError(JistLogger.WARNING, getClass()
						.getCanonicalName()
						+ "ProcessManager join interrupted.");
			}

			// kill all running jobs
			if (scheduler != null)
				scheduler.killAll();

			// kill any open streams
			if (streams != null)
				for (PrintStream stream : streams)
					if (stream != null) {
						stream.flush();
						stream.close();
					}

			if (!forceQuit)
				JistPreferences.getPreferences()
						.write(new File(PipeLibrary.getInstance()
								.getLibraryPath(), JistPreferences
								.getDefaultPreferencesFileName()));
			return true;
		}
		return false;
	}

	protected long lastModifiedLayoutTime;
	protected boolean lastModifiedLock;

	/**
	 * Main process manager thread.
	 */
	public void run() {
		updateRunning = true;
		while (updateRunning) {
			if (scheduler != null) {
				if (scheduler.isRunning()) {
					if (runAllItem.isEnabled()) {
						runAllItem.setEnabled(false);
					}
					if (!pauseAllItem.isEnabled()) {
						pauseAllItem.setEnabled(true);
					}
				} else {
					if (!runAllItem.isEnabled()) {
						runAllItem.setEnabled(true);
					}
					if (pauseAllItem.isEnabled()) {
						pauseAllItem.setEnabled(false);
					}
				}

				if ((scheduler != null) && (scheduler.hasRunningQueue()/*
																		 * getRunningQueue
																		 * () !=
																		 * null
																		 */)) {
					LinkedList<ExecutionContext> rows = scheduler
							.getRunningQueue();
					for (int i = 0; i < rows.size(); i++) {
						managerPane.getTableModel().updateRow(rows.get(i));
						// System.out.println(rows.get(i).toString());
					}
					managerPane.repaint();
				}

				updateStatusBar();
				long lastModified = (layout.getFileLocation() != null) ? layout
						.getFileLocation().lastModified() : 0;
				if ((!lastModifiedLock && lastModified > lastModifiedLayoutTime)
						&& this.managerVisible) {
					int n = JOptionPane
							.showOptionDialog(
									this,
									"The current layout has been modified. Would you like to reload it?",
									null, JOptionPane.YES_NO_OPTION,
									JOptionPane.QUESTION_MESSAGE, null, null,
									null);
					if (n == 0) {
						File f = layout.getFileLocation();
						close();
						open(f);

					} else {
						lastModifiedLayoutTime = lastModified;
					}
				}
			}
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				updateRunning = false;
				JistLogger.logOutput(JistLogger.INFO, getClass()
						.getCanonicalName() + ": stopping via interrupt.");
			}
			refresh();
		}
	}

	/**
	 * Set true if process manager should exit when closed.
	 * 
	 * @param exitOnClose
	 *            true if exit on close
	 */
	public void setExitOnClose(boolean exitOnClose) {
		this.exitOnClose = exitOnClose;
	}

	private static String connectionStatus = "";

	public static void setHypervisorConnectionStatus(boolean isConnected) {
		if (JistPreferences.getPreferences().isUseHypervisor()) {
			connectionStatus = "Hypervisor ";
			connectionStatus += isConnected ? "Connected" : "Not Connected";
		} else
			connectionStatus = "";
		getInstance().updateStatusBar();
	}

	/**
	 * Update status bar.
	 */
	public void updateStatusBar() {

		if (scheduler == null)
			return;

		if (scheduler.isInitializing()) {
			statusBar.setText("Initializing " + scheduler.getInitializedCount()
					+ " experiments");
		} else {
			int totalCompleted = scheduler.getCompletedCount();
			int failed = scheduler.getFailedCount();
			int waiting = scheduler.getWaitingCount();
			int queued = scheduler.getQueuedCount();
			int running = scheduler.getRunningCount();
			boolean isRunning = scheduler.isRunning();
			int total = totalCompleted + waiting + running;
			// int
			// percent=(int)Math.round(100*totalCompleted/(float)(totalCompleted+waiting+running));
			String updateText = "Scheduler "
					+ ((isRunning) ? "Running" : "Stopped")
					+ ": Succeeded ("
					+ (totalCompleted - failed)
					+ ") Failed ("
					+ failed
					+ ") Running ("
					+ running
					+ ") Queued ("
					+ queued
					+ ") Total ("
					+ total
					+ ")  Elapsed Time (Actual: "
					+ PerformanceSummary.formatTime(scheduler
							.getElapsedActualTime())
					+ " / CPU: "
					+ PerformanceSummary.formatTime(scheduler
							.getElapsedCpuTime()) + ")" + " ";
			if (JistPreferences.getPreferences().isUseHypervisor()) {
				updateText += connectionStatus;
			}
			statusBar.setText(updateText);
		}

		scheduler.updateManagerTable();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.WindowListener#windowActivated(java.awt.event.WindowEvent)
	 */
	public void windowActivated(WindowEvent arg0) {
		// TODO Auto-generated method stub
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.WindowListener#windowClosed(java.awt.event.WindowEvent)
	 */
	public void windowClosed(WindowEvent arg0) {
		if (exitOnClose) {
			System.exit(0);
		} else {
			this.setVisible(false);
			this.dispose();
		}
	}

	/**
	 * Create toolbar to manipulate graph.
	 * 
	 * @param arg0
	 *            the arg0
	 */
	public void windowClosing(WindowEvent arg0) {
		// TODO Auto-generated method stub
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.WindowListener#windowDeactivated(java.awt.event.WindowEvent
	 * )
	 */
	public void windowDeactivated(WindowEvent arg0) {
		// TODO Auto-generated method stub
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.WindowListener#windowDeiconified(java.awt.event.WindowEvent
	 * )
	 */
	public void windowDeiconified(WindowEvent arg0) {
		// TODO Auto-generated method stub
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.WindowListener#windowIconified(java.awt.event.WindowEvent)
	 */
	public void windowIconified(WindowEvent arg0) {
		// TODO Auto-generated method stub
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.WindowListener#windowOpened(java.awt.event.WindowEvent)
	 */
	public void windowOpened(WindowEvent arg0) {
		// TODO Auto-generated method stub
	}

	/**
	 * Disable system.exit on exit - causes close window and and exit to be the
	 * same
	 **/
	private boolean hideOnExit = false;

	public void setCloseOnlyOnExit(boolean flag) {
		hideOnExit = flag;
	}

	public PipeLayout getCurrentLayout() {
		return (layout);
	}

	public void cleanAllQuietly(PipeLayout layout) {
		// TODO Auto-generated method stub
		scheduler = new PipeScheduler(layout, this);
		scheduler.cleanAll(false);
	}

	public void processDestinations() {
		// TODO Auto-generated method stub
		scheduler.processDestinations();

	};

	// Shunxing edit
	/**
	 * AWS Properties
	 */
	/**
	 * AWS Property
	 */
	public static JistAwsEC2 jistAwsEC2;
	public static JistAwsS3 jistAwsS3;
	// public static TransferManager transferManager;
	public static AmazonS3 s3Client = null;
	public static AmazonEC2 ec2Client = null;
	// private static AWSCredentials credentials = null;
	private String accessKey = null;
	private String secretKey = null;
	private String region = null;
	private String regionS3 = null;
	private String keyPair = null;
	private boolean ifUseCloud = false;
	public static String bucketName = null;
	public static String layoutLocalPth = "";
	public static String layoutParentLocation = "";
	public static String layoutParentName = "";

	/**
	 * initAWS when init Process Manager, init S3Client, EC2Client, and send zip
	 * project to S3, unzip the project via a EC2 instance.
	 * 
	 * @author Shunxing
	 * @throws AmazonServiceException
	 * @throws AmazonClientException
	 * @throws InterruptedException
	 */
	public void initAWS() throws AmazonServiceException, AmazonClientException,
			InterruptedException {
		//Check if Use AWS cloud
		ifUseCloud = layout.getRunParameters().isUseAwsEC2();
		if (ifUseCloud) {
			JOptionPane.showMessageDialog(null,
					"Sending resource to cloud, please wait:)");

			//READ preferences file from local
			Properties prop = new Properties();
			InputStream inputStream = null;

			// Step 1. Make sure if JistAwsEC2.preferences exits
			File inputFile = new File(System.getProperties().getProperty(
					"user.home")
					+ File.separator+"JistAwsEc2" + File.separator+"JISTAWSEC2.preferences");
			if (!inputFile.exists()) {
				return;
			} else {
				try {

					// IF TRUE, get accesskey, secretkey, region, regionS3,
					// keypair, running instances
					System.out.println("init_Aws_1");
					inputStream = new FileInputStream(System.getProperties()
							.getProperty("user.home")
							+ File.separator+ "JistAwsEc2"
							+ File.separator+"JISTAWSEC2.preferences");
					prop.load(inputStream);
					accessKey = prop.getProperty("accessKeyId");
					secretKey = prop.getProperty("secretAccessKey");
					region = prop.getProperty("region");
					regionS3 = prop.getProperty("regionS3");
					keyPair = prop.getProperty("keyPairPath");

					if (accessKey == null || secretKey == null) {
						return;
					} else {
						jistAwsEC2 = new JistAwsEC2(accessKey, secretKey,
								region);
						jistAwsEC2.setKeyPairAbsolutePath(keyPair);

						jistAwsS3 = new JistAwsS3(accessKey, secretKey,
								regionS3);

						ec2Client = jistAwsEC2.getAmazonEc2Client();
						s3Client = jistAwsS3.getAmazonS3Client();
						// transferManager = new TransferManager(s3Client);

					}

					String runningId = prop
							.getProperty("AllCurrentRunningInstanceId");

					// CREATE RUNNING INSTANCE MAP
					if (runningId.length() > 0) {
						String[] curRunningInstanceId = prop.getProperty(
								"AllCurrentInstanceId").split(":");
						Set<String> tmpCurId = new HashSet<String>();
						HashMap<String, Integer> tmpMap = new HashMap<String, Integer>();
						for (int i = 0; i < curRunningInstanceId.length; ++i) {
							String woca = curRunningInstanceId[i].replace(" ",
									"");
							tmpCurId.add(woca);
							System.out.println(woca);
							tmpMap.put(woca, 1);

						}
						jistAwsEC2.setRunningInstanceUsage(tmpMap);
						jistAwsEC2.setCurRunningInstancesId(tmpCurId);
					} else {
						return;
					}

					bucketName = prop.getProperty("bucket");
					jistAwsS3.setS3Bucket(bucketName);

					if (bucketName == null) {
						return;
					} else {
						// send initial package to s3;
						// UPLOAD WHOLE directory
						this.jistAwsS3.setS3Bucket(bucketName);
						layoutParentLocation = layout.getFileLocation()
								.getParent();

						String targetDirectory = layout.getFileLocation()
								.getParentFile().getName();
						

						layoutParentName = targetDirectory;

						File dirToUpload = new File(layout.getFileLocation()
								.getParent());
						layoutLocalPth = dirToUpload.getParent();

						ObjectListing list1 = s3Client.listObjects(bucketName,
								targetDirectory);
						int objSumSize1 = list1.getObjectSummaries().size();
						if (objSumSize1 > 0) {
							System.out.println("Project exists!");
						} else {

							JistAwsUtil.SendZipProjectInputToCloud(layout
									.getFileLocation().getParent(), layout
									.getFileLocation().getAbsolutePath(),
									layout.getTitle());

							// COMPRESS THE PROJECT
							String tmpZipProject = System.getProperties()
									.getProperty("user.home")
									+ "/JistAwsEc2/tmp/"
									+ targetDirectory
									+ ".zip";
							// CREATE local zip file
							JistCloudExtension.compressDirectory(System
									.getProperties().getProperty("user.home")
									+ "/JistAwsEc2/tmp/" + targetDirectory,
									tmpZipProject);
							// send local zip file to s3
							s3Client.putObject(bucketName, targetDirectory
									+ ".zip", new File(tmpZipProject));
							// get freeRunningInstanceId
							String freeRunningInstanceId = ProcessManager.jistAwsEC2
									.getRunningInstanceIdFromMap();
							DescribeInstancesRequest describeInstanceRequest = new DescribeInstancesRequest()
									.withInstanceIds(freeRunningInstanceId);
							DescribeInstancesResult describeInstanceResult = ProcessManager.jistAwsEC2
									.getAmazonEc2Client().describeInstances(
											describeInstanceRequest);

							Instance runningInstance = describeInstanceResult
									.getReservations().get(0).getInstances()
									.get(0);
							System.out.println("init_Aws_5");
							Session session;
							try {
								session = JistAwsUtil.createSession(
										jistAwsEC2.getKeyPairAbsolutePath(),
										"ec2-user",
										runningInstance.getPublicDnsName(), 22);
								session.setConfig("StrictHostKeyChecking", "no");
								session.connect();

								try {
									while (!JistAwsUtil.execIfComplete(session,
											"./IfFileExists.bin " + "/mnt/s3/"
													+ targetDirectory + ".zip")) {
										try {
											Thread.sleep(1000);
											System.out
													.println("FILE TRANSMIT NOT COMPLETE");
										} catch (InterruptedException e) {
											System.out
													.println("FILE TRANSIMIT FAIL");
										}
									}
									// UNZIP the submission project
									String tmpCmd = "cd /mnt/s3/ && chmod 777 "
											+ targetDirectory + ".zip"
											+ " && unzip -o /mnt/s3/"
											+ targetDirectory + ".zip";
									System.out.println(tmpCmd + "woshinibaba");
									JistAwsUtil.exec(session, tmpCmd);
									// tmpCmd = "chmod 777 /mnt/s3/" +
									// targetDirectory;
									// JistAwsUtil.exec(session, tmpCmd);
								} catch (Exception e) {
									// TODO Auto-generated catch block
									e.printStackTrace();
								}

								session.disconnect();
							} catch (JSchException e) {
								// TODO Auto-generated catch block
								e.printStackTrace();
							}

							jistAwsEC2
									.setRunningInstanceFromBusyToFree(freeRunningInstanceId);

							// delete zip file
							Files.delete(Paths.get(tmpZipProject));
							
							// delete unzip file
							String origindir = System.getProperties().getProperty("user.home")+ "/JistAwsEc2/tmp/" + targetDirectory;	
							String[] dirCheck = origindir.split("/");
							if(dirCheck[dirCheck.length-1].contains(".")){
								dirCheck[dirCheck.length-1] = dirCheck[dirCheck.length-1].replace(".", "");
								origindir = "";
								for(int i = 1; i<dirCheck.length;++i){
									origindir = origindir +  File.separator + dirCheck[i];
								}
							}
							File delUploadFolder = new File(origindir);
							//JistAwsUtil.deleteDir(delUploadFolder);
							
//							 FileUtils.forceDelete(new
//							 File(System.getProperties()
//							 .getProperty("user.home")
//							 + "/JistAwsEc2/tmp/"
//							 + targetDirectory));
//							
//							 JistAwsUtil.DeleteAwsTmpFile(System.getProperties()
//							 .getProperty("user.home")
//							 + "/JistAwsEc2/tmp/"
//							 + targetDirectory);
						}
							JOptionPane.showMessageDialog(null,
									"Upload Complete!");
						
						// // Layout From Souce folder
						// expSrcName = dirToUpload.getName();
						//
						// //
						// ArrayList<String> allExpUniqueName = new
						// ArrayList<String>();
						// ArrayList<String> allOutputPicName = new
						// ArrayList<String>();
						// if (dirToUpload.isDirectory()) {
						// String[] subNote = dirToUpload.list();
						// for (String filename : subNote) {
						// if (filename.contains("exp")) {
						// expName = filename;
						// File expF = new File(layout
						// .getFileLocation().getParent()
						// + "/" + expName);
						// // System.out.println("???????//"+expF);
						// if (expF.isDirectory()) {
						//
						// String[] subsubNote = expF.list();
						//
						// for (String subFileName : subsubNote) {
						// if (!subFileName.contains(".input")) {
						// // System.out.println("sss");
						// allExpUniqueName
						// .add(subFileName);
						// // File fSubFile = new
						// // File(subFileName);
						// // if(fSubFile.isDirectory()){
						// // String[] subsubsubNote =
						// // fSubFile.list();
						// // for(String subsubFileName :
						// // subsubsubNote){
						// //
						// if(!(subsubFileName.contains(".err")||subsubFileName.contains(".out")||subsubFileName.contains(".input")||subsubFileName.contains(".output"))){
						// // allOutputPicName.add(subsubFileName);
						// // }
						// // }
						// // }
						// System.out.println(subFileName);
						// }
						// }
						// }
						// }
						//
						// }
						// }
						//
						// // the tricky part is we have to create dir and then
						// the
						// // result can be mounted from s3 to each ec2,
						// // the dir creation just need once.
						//
						// JistAwsUtil.createExpDir(expSrcName, expName,
						// allExpUniqueName);
						// MultipleFileUpload myUpload = transferManager
						// .uploadDirectory(bucketName, targetDirectory,
						// dirToUpload, true);
						//
						// // You can poll your transfer's status to check its
						// // progress
						// if (myUpload.isDone() == false) {
						// System.out.println("Transfer: "
						// + myUpload.getDescription());
						// System.out.println("  - State: "
						// + myUpload.getState());
						// System.out.println("  - Progress: "
						// + myUpload.getProgress()
						// .getBytesTransferred());
						// }
						//
						// // Transfers also allow you to set a
						// // <code>ProgressListener</code> to receive
						// // asynchronous notifications about your transfer's
						// // progress.
						// myUpload.addProgressListener(new
						// MyProgressListener());
						//
						// // Or you can block the current thread and wait for
						// your
						// // transfer to
						// // to complete. If the transfer fails, this method
						// will
						// // throw an
						// // AmazonClientException or AmazonServiceException
						// // detailing the reason.
						// myUpload.waitForCompletion();
						//
						// // After the upload is complete, call shutdownNow to
						// // release the resources.
						// transferManager.shutdownNow();

						// files need to update

						// JistAwsUtil.arraylistFile = new ArrayList<File>();
						// JistAwsUtil.arraylistFile = JistAwsUtil
						// .updatePathAndSendToS3(dirToUpload);

						// // String line = null;
						// for (File tmpFile : JistAwsUtil.arraylistFile) {
						//
						// Path srcpath = Paths.get(tmpFile.getAbsolutePath());
						// Charset charset = StandardCharsets.UTF_8;
						//
						// String content = new String(
						// Files.readAllBytes(srcpath), charset);
						// String tmpContent = content;
						// content = content.replaceAll(layoutLocalPth,
						// "/mnt/s3");
						//
						// if (!content.equals(tmpContent)) {
						// Path dstpath = Paths.get(System.getProperties()
						// .getProperty("user.home")
						// + "/JistAwsEc2/tmp/"
						// + tmpFile.getName());
						// Files.write(dstpath, content.getBytes(charset));
						//
						// File needToSend = dstpath.toFile();
						//
						// // it is a kind of messy since the dst file is
						// // saved to a different path,
						// // we need to find original path via tmpFile.
						// String needToSendFileName = tmpFile.getName();
						// String needToSendFileParent = tmpFile
						// .getParent();
						// String needToSendFileS3Dir = needToSendFileParent
						// .replace(layoutLocalPth, "");
						// String s3Location = bucketName
						// + needToSendFileS3Dir;
						// System.out.println(s3Location);
						// System.out.println(needToSendFileName);
						// System.out
						// .println(needToSend.getAbsolutePath());
						// // Thread.sleep(1000);
						// jistAwsS3.getAmazonS3Client().putObject(new
						// PutObjectRequest(s3Location,needToSendFileName,new
						// File(needToSend.getAbsolutePath())));
						// }
						//
						// }
					}

				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}

		}
	}

	static BitSet flag = new BitSet(3);

	static class MyProgressListener implements ProgressListener {
		/**
		 * Start progress.
		 */
		public void progressStart(ProgressEvent evt) {
			System.out.println("start: received progressevent " + evt);
			if (flag.nextSetBit(0) == -1)
				flag.set(0);
		}

		/**
		 * Update progress.
		 */
		public void progressUpdate(ProgressEvent evt) {
			System.out.println("update: received progressevent " + evt);
			if (flag.nextSetBit(1) == -1)
				flag.set(1);
		}

		/**
		 * Finish progress.
		 */
		public void progressFinish(ProgressEvent evt) {
			System.out.println("finish: received progressevent " + evt);
			if (flag.nextSetBit(2) == -1)
				flag.set(2);
		}

		@Override
		public void progressChanged(ProgressEvent progressEvent) {
			// TODO Auto-generated method stub

		}
	}
	// Shunxing edit

}