/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.parameter;

import java.io.File;
import java.io.IOException;
import java.io.ObjectOutputStream;

import javax.swing.JPanel;
import javax.swing.ProgressMonitor;
import javax.swing.tree.MutableTreeNode;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import edu.jhu.ece.iacl.jist.pipeline.PipePort;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamInputView;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamViewObserver;
import edu.jhu.ece.iacl.jist.pipeline.view.output.ParamOutputView;
import edu.jhu.ece.iacl.jist.utility.JistXMLUtil;
import gov.nih.mipav.model.scripting.ParserException;
import gov.nih.mipav.model.scripting.parameters.ParameterException;
import gov.nih.mipav.model.scripting.parameters.ParameterTable;
import gov.nih.mipav.view.dialogs.AlgorithmParameters;

/**
 * Generic Parameter with a bridge to interface with methods in the parameter
 * getFactory(). The parameter factory is not serialized when exporting the
 * parameter to XML
 * 
 * @author Blake Lucas
 */
public abstract class ParamModel<T> extends PipePort<T> implements Comparable<ParamModel>, Cloneable {

	/** The factory. */
	protected transient ParamFactory factory = null;

	/** The shortLabel --- "command line flag" */
	protected String shortLabel = null;

	/** The label. */
	protected String label = null;

	/** The name. */
	private String name = "";

	/** The hidden. */
	protected boolean hidden = false;

	/** Description of parameter. */
	protected String description;
	
	/** Should the parameter be validated on load. True = default behavior. **/ 
	protected transient boolean loadAndSaveOnValidate = true; 

	/**
	 * Default constructor.
	 */
	public ParamModel() {
		super(null, null);
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.PipePort#clone()
	 */
	public abstract ParamModel<T> clone();

	/**
	 * Compare specificity of parameters. Zero indicates both parameters have
	 * the same level of specificity. Negative one indicates this parameter is
	 * less specific than the foreign one. Positive one indicates this parameter
	 * is more specific than the foreign one.
	 * 
	 * @param mod
	 *            the mod
	 * @return the int
	 */
	public int compareTo(ParamModel mod) {
		return (this.getClass().equals(mod.getClass())) ? 0 : 1;
	}

	/**
	 * Create MIPAV parameter that represents this parameter.
	 * 
	 * @param scriptParams
	 *            script parameter
	 * @throws ParserException
	 *             the parser exception
	 */
	public void createMipavParameter(AlgorithmParameters scriptParams) throws ParserException {
		getFactory().createMipavParameter(scriptParams);
	}

	/**
	 * Create tree node that represents this parameter.
	 * 
	 * @return the mutable tree node
	 */
	public final MutableTreeNode createTreeNode() {
		return getFactory().createTreeNode();
	}
	/**
	 * Clean resources that maybe out-of-date with the system
	 */
	public void clean(){

	}
	/**
	 * Replace path in all file type objects with this path. This is used for JUnit testing
	 * @param replaceDir
	 */
	public void replacePath(File originalDir,File replaceDir){

	}

	/**
	 * Get parameter factory.
	 * 
	 * @return parameter factory
	 */
	public ParamFactory getFactory() {
		if (factory == null) {
			init();
		}
		return factory;
	}

	/**
	 * Get the input view to specify the parameter value.
	 * 
	 * @return input view
	 */
	public final ParamInputView getInputView() {
		return getFactory().getInputView();
	}

	/**
	 * Get the short text label for this parameter. Useful for command line flag.
	 * 
	 * @return suggested command line flag for parameter 
	 */
	public String getShortLabel() {
		if (shortLabel == null )  {
			return getLabel();
		} else  {
			return shortLabel;
		}
	}

	/**
	 * Get the text label for this parameter. If none is specified, the text
	 * name is used
	 * 
	 * @return label
	 */
	public String getLabel() {
		if (label == null) {
			if(name==null)
				return "NULL-NO-NAME";
			return name;
		} else {
			return label;
		}
	}

	/**
	 * Get the parameter's name, which should be unique.
	 * 
	 * @return name
	 */
	public String getName() {
		if ((name == null) && (label != null)) {
			return label;
		}		
		return name;
	}

	/**
	 * Get the output view to specify the output parameter.
	 * 
	 * @return output view
	 */
	public final ParamOutputView getOutputView() {
		return getFactory().getOutputView();
	}

	/**
	 * Get the value stored in the parameter.
	 * 
	 * @return this the default value stored by the parameter. Although,
	 *         parameters could potentially store more than one type of value.
	 */
	public abstract T getValue();

	/**
	 * Get panel that represents this parameter, depending on whether it is an
	 * input or output parameter.
	 * 
	 * @return the view
	 */
	public JPanel getView() {
		return (isInputPort()) ? getFactory().getInputView() : getFactory().getOutputView();
	}

	/**
	 * Hide this parameter.
	 */
	public void hide() {
		hidden = true;
	}

	/**
	 * Import value from MIPAV script parameter.
	 * 
	 * @param paramTable
	 *            parameter table
	 * @throws ParameterException
	 *             the parameter exception
	 */
	public void importMipavParameter(ParameterTable paramTable) throws ParameterException {
		getFactory().importMipavParameter(paramTable);
	}

	/**
	 * Import the contents of a foreign parameter into this parameter.
	 * 
	 * @param model
	 *            foreign parameter
	 * @return true if success
	 */
	public final boolean importParameter(ParamModel model) {
		return getFactory().importParameter(model);
	}

	/**
	 * Check for compatibility between parameters.
	 * 
	 * @param model
	 *            foreign parameter
	 * @return true if compatible
	 */
	public boolean isCompatible(PipePort model) {
		return (this.getClass().equals(model.getClass()));
	}

	/**
	 * returns true if connector can connect to this parameter.
	 * 
	 * @return true, if checks if is connectible
	 */
	public boolean isConnectible() {
		return (connectible && !hidden);
	}

	/**
	 * Returns true if parameter is hidden.
	 * 
	 * @return true if hidden
	 */
	public boolean isHidden() {
		return hidden;
	}

	/**
	 * Load resources.
	 * 
	 * @param param
	 *            the param
	 * @return true, if successful
	 */
	public final boolean loadResources(ParamModel param) {
		return getFactory().loadResources(param, null);
	}

	/**
	 * Load external resources specified in foreign parameter.
	 * 
	 * @param param
	 *            foreign parameter
	 * @param monitor
	 *            TODO
	 * @return true if resources successfully loaded or not specified
	 */
	public final boolean loadResources(ParamModel param, ProgressMonitor monitor) {
		return getFactory().loadResources(param, monitor);
	}

	/**
	 * Read parameter information from file.
	 * 
	 * @param f
	 *            XML file
	 * @return true if parameter information imported correctly
	 */
	public final boolean read(File f) {
		return getFactory().read(f);
	}

	/**
	 * Read parameter information from string.
	 * 
	 * @param text
	 *            the text
	 * @param loadResources
	 *            the load resources
	 * @return true if parameter information imported correctly
	 */
	public final boolean read(String text, boolean loadResources) {
		return getFactory().read(text, loadResources);
	}

	/**
	 * Save external resources associated with the parameter to the specified
	 * directory.
	 * 
	 * @param dir
	 *            save directory
	 * @return true if success
	 */
	public final boolean saveResources(File dir, boolean saveSubDirectoryOverride) {		
		if(loadAndSaveOnValidate)
			return getFactory().saveResources(dir, saveSubDirectoryOverride);
		else {
			return true;
		}
	}

	/**
	 * Use a different factory for manipulating this parameter model.
	 * 
	 * @param factory
	 *            the factory
	 */
	public void setFactory(ParamFactory factory) {
		this.factory = factory;
	}

	/**
	 * Set hidden status.
	 * 
	 * @param b
	 *            true if hidden
	 */
	public void setHidden(boolean b) {
		hidden = b;
	}

	/**
	 * Set the input view used to specify the parameter value.
	 * 
	 * @param view
	 *            the view
	 */
	public final void setInputView(ParamInputView view) {
		for(ParamViewObserver obs:getFactory().getInputView().getObservers()){
			view.addObserver(obs);
		}
		getFactory().setInputView(view);

	}

	/**
	 * Set the label to be displayed in the input and output views.
	 * 
	 * @param label
	 *            the label
	 */
	public void setLabel(String label) {
		this.label = label;
	}

	/**
	 * Set the shortLabel to be used as a command line switch.
	 * 
	 * @param shortLabel
	 *            the shortLabel
	 */
	public void setShortLabel(String shortLabel) {
		this.shortLabel = shortLabel;
	}


	/**
	 * Set the name of the parameter, which will appear as the label if the
	 * label is not specified.
	 * 
	 * @param name
	 *            the name
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Set the output view used to display the parameter value.
	 * 
	 * @param view
	 *            the view
	 */
	public final void setOutputView(ParamOutputView view) {
		getFactory().setOutputView(view);
	}

	/**
	 * Set the parameter value. A parameter maybe able to support more than one
	 * type of value
	 * 
	 * @param value
	 *            the value
	 * @throws InvalidParameterValueException
	 *             object type is not an acceptable value for this parameter
	 * @throws parameter
	 *             assigned value of an improper type
	 */
	public abstract void setValue(T value) throws InvalidParameterValueException;

	/**
	 * Get description of parameter.
	 * 
	 * @return the string
	 */
	public abstract String toString();

	/**
	 * Serialize parameter as XML.
	 * 
	 * @return the string
	 */
	public final String toXML() {
		return getFactory().toXML();
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.PipePort#validate()
	 */
	public abstract void validate() throws InvalidParameterException;

	/**
	 * Write this parameter to an XML file.
	 * 
	 * @param f
	 *            file
	 * @return true if success
	 */
	public final boolean write(File f) {
		return getFactory().write(f);
	}

	/**
	 * Write parameter to byte serialized output stream.
	 * 
	 * @param out
	 *            the out
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public final void write(ObjectOutputStream out) throws IOException {
		getFactory().write(out);
	}

	/**
	 * @return the description
	 */
	public String getDescription() {
		if(description==null){
			return getLabel();
		} else {
			return description;
		}
	}

	/**
	 * @param description the description to set
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * 
	 */
	public boolean equals(ParamModel<T> model){
		T value1=this.getValue();
		T value2=model.getValue();
		if(value1==null&&value2==null)return true;
		if(value1==null||value2==null)
			return false;
		if(value1.equals(value2)){
			return true;
		} else {
			//System.err.println(getClass().getCanonicalName()+this.getLabel()+": "+value1+" NOT EQUAL TO "+model.getLabel()+": "+value2);
			return false;
		}
	}
	
	public abstract String getHumanReadableDataType();
	protected String cliTag=null;
	public void setCliTag(String tag) {cliTag=tag;};
	public String getCliTag() {return cliTag;}

	abstract public void setXMLValue(String arg);
	abstract public String getXMLValue();
	abstract public String probeDefaultValue(); 
	
	/**
	 * Allow memory saving operations by reducing the level of validation performed. 
	 * 
	 * @param flag - false = do not perform extra validation, true (default) = perform standard valdiation
	 */
	public void setLoadAndSaveOnValidate(boolean flag) {
		loadAndSaveOnValidate = flag;
	}
	
	/**
	 * Release as much memory as possible from this object without
	 * setting the entire object to null. If this is an input parameter, 
	 * the data should be able to be retrieved. By default, does nothing. 
	 */
	public void dispose() {
		
	}
	
	/**
	 * Attempt to export the current parameter now, and, if possible, free
	 * associated memory. Does not apply in cases where the algorithm is being
	 * run in process. By default, does nothing. Only possibly active for parameters
	 * with setLoadAndSaveOnValidate(false)
	 * 
	 * @param src - Parent ProcessingAlgorithm needed to determine output location
	 * 
	 */
	public void writeAndFreeNow(ProcessingAlgorithm src) {
		// default = do nothing
	}
	
	public boolean xmlEncodeParam(Document document, Element parent) {

		Element em;
		
		/** Label to display for this parameter. */
		em = document.createElement("classname");		
		em.appendChild(document.createTextNode(this.getClass().getCanonicalName()));
		parent.appendChild(em);
		
		/** Label to display for this parameter. */
		em = document.createElement("label");		
		em.appendChild(document.createTextNode(getLabel()));
		parent.appendChild(em);
		
		/** Unique name to identify this parameter. */		
		em = document.createElement("name");		
		em.appendChild(document.createTextNode(getName()));
		parent.appendChild(em);
		
		/** The shortLabel --- "command line flag" */
		em = document.createElement("shortLabel");		
		em.appendChild(document.createTextNode(getShortLabel()));
		parent.appendChild(em);

		/** The hidden. */
		em = document.createElement("hidden");		
		em.appendChild(document.createTextNode(""+hidden));
		parent.appendChild(em);		

		/** The mandatory. */
		em = document.createElement("mandatory");		
		em.appendChild(document.createTextNode(""+mandatory));
		parent.appendChild(em);				

		/** Description of parameter. */
		em = document.createElement("description");		
		em.appendChild(document.createTextNode(""+description));
		parent.appendChild(em);		
		
		return true;
				
	}

	public void xmlDecodeParam(Document document, Element el) {
		label = JistXMLUtil.xmlReadTag(el, "label");
		name = JistXMLUtil.xmlReadTag(el, "name");
		description = JistXMLUtil.xmlReadTag(el, "description");
		shortLabel = JistXMLUtil.xmlReadTag(el, "shortLabel");
		hidden = Boolean.valueOf(JistXMLUtil.xmlReadTag(el, "hidden"));
		mandatory = Boolean.valueOf(JistXMLUtil.xmlReadTag(el, "mandatory"));

	}
	
	public static ParamModel abstractXmlDecodeParam(Document document, Element parent) {
		String className = JistXMLUtil.xmlReadTag(parent, "classname");
		ParamModel param;
		try {
			param = (ParamModel)Class.forName(className).newInstance();
		} catch (InstantiationException e) {
			return null;
		} catch (IllegalAccessException e) {
			return null;
		} catch (ClassNotFoundException e) {
			return null;
		}
		param.xmlDecodeParam(document, parent);
		return param;
	}

	public boolean isVolume() { return false; }
	public boolean isSurface() { return false; }
}
