/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.parameter;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import edu.jhu.ece.iacl.jist.pipeline.PipePort;
import edu.jhu.ece.iacl.jist.pipeline.factory.ParamSurfaceCollectionFactory;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Surface collection stores a collection of files. The restrictions on the
 * files are set to be the same as the collection.
 * 
 * @author Blake Lucas
 */
public class ParamSurfaceCollection extends ParamSurfaceLocationCollection {
	
	/**
	 * Default constructor.
	 */
	public ParamSurfaceCollection() {
		super();
		this.factory = new ParamSurfaceCollectionFactory(this);
	}

	/**
	 * Constructor.
	 * 
	 * @param name
	 *            parameter name
	 */
	public ParamSurfaceCollection(String name) {
		this();
		this.setName(name);
	}

	/**
	 * Add a new value to the collection.
	 * 
	 * @param value
	 *            the value
	 * @return surface parameter
	 */
	public void add(Object value) {
		ParamSurface param;
		if (value instanceof ParamSurface) {
			fileParams.add(param = (ParamSurface) value);
		} else {
			param = create(value);
			fileParams.add(param);
		}
//		return param;
	}

	/**
	 * Remove all files from collection.
	 */
	public void clear() {
		this.fileParams.clear();
	}

	/**
	 * Clone object.
	 * 
	 * @return the param surface collection
	 */
	public ParamSurfaceCollection clone() {
		ParamSurfaceCollection param = new ParamSurfaceCollection();
		param.setName(this.getName());
		param.label=this.label;
		param.setHidden(this.isHidden());
		param.setMandatory(this.isMandatory());
		param.fileParams = new Vector<ParamFile>(fileParams.size());
		for (ParamFile p : fileParams) {
			param.fileParams.add(p.clone());
		}
		param.mandatory = mandatory;
		param.readerWriter = readerWriter;
		return param;
	}

	/**
	 * Compare restriction of one volume collection to another.
	 * 
	 * @param model
	 *            the model
	 * @return the int
	 */
	public int compareTo(ParamModel model) {
		return (model instanceof ParamSurfaceCollection) ? 0 : 1;
	}
	
	public ParamSurface getParamSurface(int i) {
		if (i < fileParams.size()) {
			return (ParamSurface) fileParams.get(i);
		} else {
			return null;
		}
	}

	/**
	 * Create a new ParamSurface with the same restrictions as the collection
	 * and the specified value.
	 * 
	 * @param value
	 *            the value
	 * @return surface parameter
	 */
	protected ParamSurface create(Object value) {
		ParamSurface param = new ParamSurface(getName());
		if (value instanceof String) {
			param.setValue((String) value);
			param.setName((String) value);
		} else if (value instanceof File) {
			param.setValue((File) value);
			param.setName(((File) value).getName());
		} else if (value instanceof EmbeddedSurface) {
			param.setValue((EmbeddedSurface) value);
			param.setName(((EmbeddedSurface) value).getName());
		}
		param.setReaderWriter(readerWriter);
		param.setMandatory(mandatory);
		param.shortLabel=shortLabel;
		param.cliTag=cliTag;
		return param;
	}

	/**
	 * Get surfaces.
	 * 
	 * @return list of surface
	 */
	public List<EmbeddedSurface> getSurfaceList() {
		ArrayList<EmbeddedSurface> surfs = new ArrayList<EmbeddedSurface>();
		for (ParamFile param : fileParams) {
			surfs.add(((ParamSurface) param).getSurface());
		}
		return surfs;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection#isCompatible(edu.jhu.ece.iacl.jist.pipeline.PipePort)
	 */
	public boolean isCompatible(PipePort model) {
		return ((model instanceof ParamSurfaceLocationCollection) || (model instanceof ParamSurface)||(model instanceof ParamSurfaceLocation));
	}

	/**
	 * Set the volume collection. This method accepts ArrayLists with any of the
	 * valid types of ParamFile
	 * 
	 * @param value
	 *            parameter value
	 */
	public void setValue(List value) {
		clear();
		for (Object obj : value) {
			this.add(obj);
		}
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection#size()
	 */
	public int size() {
		return this.fileParams.size();
	}
	
	public boolean isSurface() { return true; }
}
