/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.src;

import java.util.Random;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import edu.jhu.ece.iacl.jist.pipeline.PipeSource;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;


/**
 * Generate N random doubles within a range.
 * 
 * @author Aaron Carass
 */
public class PipeRandomDoubleSource extends PipeSource {

	/** Min */
	transient protected double min;

	/** Max */
	transient protected double max;

	/** N = Number to generate */
	/** n = Number generated thus far */
	transient protected int N;
	transient protected int n = 0;

	/** Current Number */
	transient protected double val;

	/** Java RNG Object */
	Random generator = new Random();


	/** Params */
	protected ParamDouble minParam, maxParam;
	protected ParamDouble valParam;
	protected ParamInteger nParam;


	protected boolean xmlEncodeModule(Document document, Element parent) {
		boolean val = super.xmlEncodeModule(document, parent);		
		return val;
	}


	public void xmlDecodeModule(Document document, Element el) {
		super.xmlDecodeModule(document, el);
		minParam = (ParamDouble) inputParams.getFirstChildByName("Minimum Value");
		maxParam = (ParamDouble) inputParams.getFirstChildByName("Maximum Value");

		nParam = (ParamInteger) inputParams.getFirstChildByName("Number of Random Numbers to Generate");

		valParam = (ParamDouble) outputParams.getFirstChildByName("Double");
		getParentPort().setParameter(valParam);
	}


	/**
	 * Default constructor.
	 */
	public PipeRandomDoubleSource() {
		super();
		getParentPort().setParameter(valParam);
	}


	/**
	 * Create input parameters.
	 * 
	 * @return the param collection
	 */
	public ParamCollection createInputParams() {
		ParamCollection group = new ParamCollection();
		group.setLabel("Random Float Generator");
		group.setName("randomDoubleGenerator");


		group.add(minParam = new ParamDouble("Minimum Value"));
		group.add(maxParam = new ParamDouble("Maximum Value"));
		group.add(nParam = new ParamInteger("Number of Random Numbers to Generate"));


		group.setCategory("Number.Float");
		return group;
	}


	/**
	 * Create ouptut parameters.
	 * 
	 * @return the param collection
	 */
	public ParamCollection createOutputParams() {
		ParamCollection group = new ParamCollection();
		group.setLabel("Float");
		group.add(valParam = new ParamDouble("Float"));
		return group;
	}


	/**
	 * Get output parameter.
	 * 
	 * @return the output param
	 */
	public ParamModel getOutputParam() {
		return valParam;
	}


	/**
	 * Return true if iterator has more elements.
	 * 
	 * @return true, if checks for next
	 *
	 * As long as the value is changing every iteration
	 * (ie. prevVal < val) then we proceed.
	 */
	public boolean hasNext() {
		return (super.hasNext() || (n < N));
	}


	/**
	 * Iterate through double values.
	 * 
	 * @return true, if iterate
	 */
	public boolean iterate() {
		if (hasNext()) {
			if (!super.iterate()) {
				if (N < 1) {
					/**
					 * Catch some sillyness.
					 */
					return false;
				}

				/**
				 * generator.nextDouble() returns [0, 1] so we 
				 * never need to test val < min OR val > max.
				 */

				if (n < N) {
					val = (max - min) * generator.nextDouble() + min;
					valParam.setValue(val);
					push();
					n++;
				} else {
					reset();
					isReset = true;
					return false;
				}
			}
			return true;
		} else {
			reset();
			isReset = true;
			return false;
		}
	}


	/**
	 * Reset iterator.
	 */
	public void reset() {
		super.reset();

		min = minParam.getDouble();
		max = maxParam.getDouble();
		val = (max - min) * generator.nextDouble() + min;

		N = nParam.getInt();
		n = 0;

		valParam.setValue(val);
		push();
	}
}
