/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.src;

import java.io.File;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import javax.swing.ProgressMonitor;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import edu.jhu.ece.iacl.jist.io.ArrayObjectTxtReaderWriter;
import edu.jhu.ece.iacl.jist.io.FileReaderWriter;
import edu.jhu.ece.iacl.jist.pipeline.PipeSource;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamCollectionPipelineInputView;
import edu.jhu.ece.iacl.jist.utility.JistLogger;

/**
 * Iterate through list of parameters.
 * 
 * @author Blake Lucas
 */
@SuppressWarnings("unchecked")
public abstract class PipeSourceSet extends PipeSource {
	
	/** the selector **/
	protected ParamOption typeOption;
	
	/** The File Collection Based Parameters */
	protected ParamFileCollection inFileCollection;
	
	/** the Text File Based Parameters **/
	protected ParamFile txtFileParam;
	protected ParamInteger offset;
	protected ParamInteger startEntry;
	protected ParamInteger endEntry;
	
	/** the Directory Search Based Parameters **/
	protected ParamFile dirParam;
	protected ParamBoolean includeParam;
	protected ParamOption filterRule;
	protected ParamString filterExpression;
	protected ParamString fileExtension;
	protected ParamInteger searchDepth;
	
	/** The data. */
	transient protected Vector<File> fileList;
	transient protected Object[][] data;
	transient protected List<File> selection;
	
	/** The index. */
	transient protected int index;
	
	/** The off. */
	transient int st, end, off;
	
	public void xmlDecodeModule(Document document, Element el) {
		super.xmlDecodeModule(document, el);
		typeOption = (ParamOption) inputParams.getFirstChildByName("Set Specification Type");
		txtFileParam = (ParamFile) inputParams.getFirstChildByName("List File");
		startEntry = (ParamInteger) inputParams.getFirstChildByName("Start Entry");
		endEntry = (ParamInteger) inputParams.getFirstChildByName("End Entry");
		offset = (ParamInteger) inputParams.getFirstChildByName("Offset Index");
		dirParam =(ParamFile) inputParams.getFirstChildByName("Directory");
		includeParam =(ParamBoolean) inputParams.getFirstChildByName("Include Sub-directories");
		filterRule = (ParamOption) inputParams.getFirstChildByName("Filter Rule");
		filterExpression = (ParamString) inputParams.getFirstChildByName("Filter Expression");
		fileExtension = (ParamString) inputParams.getFirstChildByName("File Extension");
		searchDepth = (ParamInteger) inputParams.getFirstChildByName("Max Directory Depth");
	}

	/**
	 * Create input parameters.
	 * 
	 * @return the param collection
	 */
	public ParamCollection createInputParams() {
		ParamCollection outGroup = new ParamCollection();

		
		outGroup.add(typeOption = new ParamOption("Set Specification Type", new String[] { "Manual", "Text File", "Directory Search"}));
		
		// set the manual options
		ParamCollection manualGroup = new ParamCollection("Manual");
		manualGroup.add(inFileCollection = getInputCollection());
		System.out.println("Created inFileCollection");
		
		// set the text file options
		ParamCollection txtGroup = new ParamCollection("Text File");
		txtGroup.add(txtFileParam = new ParamFile("List File"));
		txtFileParam.setExtensionFilter(ArrayObjectTxtReaderWriter.getInstance().getExtensionFilter());
		txtGroup.add(offset = new ParamInteger("Offset Index", 0, 1000000, 0));
		txtGroup.add(startEntry = new ParamInteger("Start Entry", 0, 100000, 0));
		txtGroup.add(endEntry = new ParamInteger("End Entry", -1, 100000, -1));
		
		// set the directory file options
		ParamCollection dirGroup = new ParamCollection("Directoy Search");
		dirGroup.add(dirParam = new ParamFile("Directory", ParamFile.DialogType.DIRECTORY));
		dirGroup.add(includeParam = new ParamBoolean("Include Sub-directories", true));
		includeParam.setValue(false);
		dirGroup.add(searchDepth = new ParamInteger("Max Directory Depth", 0, 100000, 1));
		dirGroup.add(fileExtension = new ParamString("File Extension"));
		fileExtension.setValue("txt");
		dirGroup.add(filterRule = new ParamOption("Filter Rule", new String[] { "Contains", "Matches","Begins With", "Ends With" }));
		dirGroup.add(filterExpression = new ParamString("Filter Expression"));
		
		outGroup.add(manualGroup);
		outGroup.add(txtGroup);
		outGroup.add(dirGroup);
		
		return outGroup;
	}
	
	protected abstract ParamFileCollection getInputCollection();
	
	public void setSelectedTab() {
		ParamCollectionPipelineInputView view = (ParamCollectionPipelineInputView)this.getInputParams().getFactory().getInputView();
		view.getTabPane().setSelectedIndex(typeOption.getIndex());
	}
	
	@SuppressWarnings("rawtypes")
	public abstract ParamModel getOutputParam();

	/**
	 * Get File value.
	 * 
	 * @param i
	 *            the i
	 * @return the value
	 */
	protected File getValue(int i) {
		if (off < data[i].length) {
			return new File((String) data[i][off]);
		} else {
			return null;
		}
	}
	
	protected boolean getSourceHasNext() { return(super.hasNext()); }

	protected void sourceReset() { super.reset(); }
	protected boolean sourceIterate() { return(super.iterate()); }
	
	/**
	 * Returns true if iterator has more values.
	 * 
	 * @return true, if checks for next
	 */
	public boolean hasNext() {
		if (typeOption.getIndex() == 0)
			return (super.hasNext() || (index < selection.size()));
		else if (typeOption.getIndex() == 1)
			return (super.hasNext() || (((end == -1) || (index < end)) && (index < data.length)));
		else
			return (super.hasNext() || (index < fileList.size()));
	}

	/**
	 * Iterate.
	 * 
	 * @return true, if iterate
	 */
	public boolean iterate() {
		if (typeOption.getIndex() == 0)
			return(iterate_manual());
		else if (typeOption.getIndex() == 1)
			return(iterate_txt());
		else
			return(iterate_dir());
	}

	private boolean iterate_manual() {
		if (hasNext()) {
			if (!super.iterate()) {
				if (index < selection.size()) {
					getOutputParam().setValue(selection.get(index++));
					push();
				} else {
					reset();
					isReset = true;
					return false;
				}
			}
			return true;
		} else {
			reset();
			isReset = true;
			return false;
		}
	}
	private boolean iterate_txt() {
		if (hasNext()) {
			if (!super.iterate()) {
				if (index < data.length) {
					getOutputParam().setValue(getValue(index++));
					push();
				} else {
					reset();
					isReset = true;
					return false;
				}
			}
			return true;
		} else {
			reset();
			isReset = true;
			return false;
		}
	}
	private boolean iterate_dir() {
		if (hasNext()) {
			if (!super.iterate()) {
				if (index < fileList.size()) {
					getOutputParam().setValue(fileList.get(index++));
					push();
				} else {
					reset();
					isReset = true;
					return false;
				}
			}
			return true;
		} else {
			reset();
			isReset = true;
			return false;
		}
	}

	/**
	 * Reset iterator.
	 */
	public void reset() {
		if (typeOption.getIndex() == 0)
			reset_manual();
		else if (typeOption.getIndex() == 1)
			reset_txt();
		else
			reset_dir();
	}
	private void reset_manual() {
		super.reset();
		selection = inFileCollection.getValue();
		index = 0;
		if (selection.size() > 0) {
			getOutputParam().setValue(selection.get(index++));
		} else {
			index++;
		}
		push();
	}
	private void reset_txt() {
		super.reset();
		data = ArrayObjectTxtReaderWriter.getInstance().read(txtFileParam.getValue());
		if(data == null) {
			JistLogger.logError(JistLogger.INFO, "Cannot Read: "+txtFileParam.getValue());
			data = new Object[0][0];
		} 
		index = 0;
		st = startEntry.getInt();
		end = endEntry.getInt();
		off = offset.getInt();
		index = st;
		if (((end == -1) || (index < end)) && (index < data.length)) {
			getOutputParam().setValue(getValue(index++));
		} else {
			index++;
		}
		push();
	}
	public void reset_dir() {
		super.reset();
		findMatchingFiles();
		index = 0;
		if (fileList.size() > 0) {
			getOutputParam().setValue(fileList.get(index++));
			push();

		}
	}
	
	protected void findMatchingFiles(){
		fileList = new Vector<File>();
		File dir = dirParam.getValue();
		if (dir == null) {
			return;
		}
		boolean recurse = includeParam.getValue();
		int rule = filterRule.getIndex();
		String exp = filterExpression.getValue();
		LinkedList<File> dirs = new LinkedList<File>();
		LinkedList<Integer> depths = new LinkedList<Integer>();
		String extension = fileExtension.getValue();
		int maxDepth = searchDepth.getInt();
		ProgressMonitor monitor= new ProgressMonitor(null, "Searching directory "+dir.getAbsolutePath(),"Searching...", 0,2);
		monitor.setMillisToDecideToPopup(100);
		if (recurse) {
			dirs.add(dir);
			depths.add(0);
			File d;
			int depth;
			while (dirs.size() > 0) {
				d = dirs.remove();
				depth = depths.remove();
				if ((d == null) || (depth > maxDepth)) {
					continue;
				}
				File[] files = d.listFiles();
				if (files == null) {
					continue;
				}
				if(monitor.isCanceled())break;
				monitor.setProgress(1);
				monitor.setNote(d.getAbsolutePath());
				for (File f : files) {
					if (f.isDirectory()) {
						dirs.add(f);
						depths.add(depth + 1);
					} else {
						String ext = FileReaderWriter.getFileExtension(f);
						String filename = FileReaderWriter.getFileName(f);
						String full_filename = filename + "." + ext;
						boolean accept = true;
						if (exp.length() > 0) {
							switch (rule) {
							case 0:
								if (filename.indexOf(exp) < 0 && full_filename.indexOf(exp) < 0) {
									accept = false;
								}
								break;
							case 1:
								if (!filename.matches(exp) && !full_filename.matches(exp)) {
									accept = false;
								}
								break;
							case 2:
								if (!filename.startsWith(exp) && !full_filename.startsWith(exp)) {
									accept = false;
								}
								break;
							case 3:
								if (!filename.endsWith(exp) && !full_filename.endsWith(exp)) {
									accept = false;
								}
								break;
							}
						}
						if (accept && (ext.equals(extension) || (extension.length() == 0) || full_filename.endsWith(extension))&&f.isFile()) {
							fileList.add(f);
						}
					}
				}
			}
		} else {
			File[] files = dir.listFiles();
			for (File f : files) {
				
				// skip directories
				if (f.isDirectory())
					continue;
				
				String ext = FileReaderWriter.getFileExtension(f);
				String filename = FileReaderWriter.getFileName(f);
				String full_filename = filename + "." + ext;
				boolean accept = true;
				if (exp.length() > 0) {
					switch (rule) {
					case 0:
						if (filename.indexOf(exp) < 0 && full_filename.indexOf(exp) < 0) {
							accept = false;
						}
						break;
					case 1:
						if (!filename.matches(exp) && !full_filename.matches(exp)) {
							accept = false;
						}
						break;
					case 2:
						if (!filename.startsWith(exp) && !full_filename.startsWith(exp)) {
							accept = false;
						}
						break;
					case 3:
						if (!filename.endsWith(exp) && !full_filename.endsWith(exp)) {
							accept = false;
						}
						break;
					}
				}
				if (accept && (ext.equals(extension) || (extension.length() == 0) || full_filename.endsWith(extension))&&f.isFile()) {
					fileList.add(f);
				}
			}
		}
		monitor.close();
		Collections.sort(fileList);
	}

}
