/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.view.input;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.ScrollPaneConstants;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;

import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamStringArea;

/**
 * Input View creates a text field to enter a numerical value. The value is
 * validated against the specified min and max value. This view is default for
 * double values.
 * 
 * @author Blake Lucas
 */
public class ParamTextAreaInputView extends ParamInputView implements ActionListener, CaretListener {
	
	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = 5522230314641027357L;
	
	/** The field. */
	private JTextArea field;
	private JScrollPane fieldScrollPane;
	private JPanel fieldPane;

	public static final Dimension defaultTextPaneDimension = new Dimension(200, 250);

	/**
	 * Construct text field to enter numerical value.
	 * 
	 * @param param
	 *            the param
	 */
	public ParamTextAreaInputView(ParamStringArea param) {
		super(param);
		buildLabel(BorderLayout.NORTH);
		
		
		field = new JTextArea();
		field.setText(param.getValue());
		field.addCaretListener(this);
		field.setLineWrap(true);
		field.setWrapStyleWord(true);

		fieldScrollPane = new JScrollPane(field);
		fieldScrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
		fieldScrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		
		fieldPane = new JPanel(new BorderLayout());
		fieldPane.add(fieldScrollPane);
		fieldPane.setPreferredSize(defaultTextPaneDimension);
		add(fieldPane, BorderLayout.SOUTH);
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		if (event.getSource().equals(field)) {
			try {
				getParameter().setValue(field.getText());
				notifyObservers(param, this);
			} catch (NumberFormatException e) {
				System.err.println(getClass().getCanonicalName()+e.getMessage());
			}
		}
	}

	/**
	 * A change has occurred to the input field.
	 * 
	 * @param event
	 *            text input changed
	 */
	public void caretUpdate(CaretEvent event) {
		if (event.getSource().equals(field)) {
			if (!getParameter().getValue().equals(field.getText())) {
				// Set parameter value to textfield value
				getParameter().setValue(field.getText());
				notifyObservers(param, this);
			}
		}
	}

	/**
	 * Commit changes to text field.
	 */
	public void commit() {
		// Set parameter value to textfield value
		getParameter().setValue(field.getText());
		notifyObservers(param, this);
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.view.input.ParamInputView#getParameter()
	 */
	public ParamStringArea getParameter() {
		return (ParamStringArea) param;
	}

	/**
	 * Update field with parameter value.
	 */
	public void update() {
		if (!field.getText().equals(getParameter().getValue()))
			field.setText(getParameter().getValue());
	}
	/**
	 * Get field used to enter this value
	 */
	public JComponent getField() {
		return field;
	}
}
