/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.view.input;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.Dialog.ModalityType;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableModel;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.io.MipavController;
import edu.jhu.ece.iacl.jist.pipeline.JistPreferences;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamXnatFileCollection;
import edu.vanderbilt.masi.xnatJist.XnatFileObject;
import edu.vanderbilt.masi.xnatJist.XnatLoginPanel;
import edu.vanderbilt.masi.xnatJist.XnatMain;
import edu.vanderbilt.masi.xnatJist.XnatSearchController;


/**
 * Input view to select multiple files.
 * 
 * @author Blake Lucas
 */
public class ParamXnatFileCollectionInputView extends ParamInputView implements ListSelectionListener, ActionListener,
		ListCellRenderer {
	
	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -4483721450186968746L;
	
	/** The add button. */
	private JButton addButton;
	
	/** The remove button. */
	private JButton removeButton;
	
	/** The up button. */
	private JButton upButton;
	
	/** The down button. */
	private JButton downButton;
	
	private JButton viewButton;
	
	/** The field. */
	protected JList field;
	
	/** The image list size. */
	protected int imageListSize = -1;
	
	/** The list box entries. */
	protected DefaultListModel listBoxEntries;
	
	/** The list pane. */
	protected JPanel listPane;
	
	/** The scroll pane. */
	protected JScrollPane scrollPane;
	
	public XnatFileObject obj;
	
	

	/**
	 * Default constructor.
	 * 
	 * @param param
	 *            parameters
	 */
	public ParamXnatFileCollectionInputView(ParamXnatFileCollection param) {
		super(param);
		buildLabel(BorderLayout.NORTH);
		// Create list of entries that contain ParamFile
		listBoxEntries = new DefaultListModel();
		// Create listbox entry field
		field = new JList(listBoxEntries);
		// Use custom entry renderer
		field.setCellRenderer(this);
		field.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		// Create browse button to select images
		addButton = new JButton("Add");
		addButton.addActionListener(this);
		upButton = new JButton("Up");
		upButton.addActionListener(this);
		downButton = new JButton("Down");
		downButton.addActionListener(this);
		removeButton = new JButton("Remove");
		removeButton.addActionListener(this);
		viewButton = new JButton("View Query");
		viewButton.addActionListener(this);
		// Create scroll pane to display entries
		scrollPane = new JScrollPane();
		scrollPane.setMinimumSize(new Dimension(100, 30));
		scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
		scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		// Create list pane to layout listbox entries
		listPane = new JPanel(new BorderLayout());
		listPane.add(field, BorderLayout.CENTER);
		// Create pane to layout list pane
		JPanel smallPane = new JPanel(new BorderLayout());
		smallPane.add(listPane, BorderLayout.NORTH);
		scrollPane.setViewportView(smallPane);
		// Create list box
		// Create pane to layout scroll pane and browse button
		smallPane = new JPanel(new BorderLayout(5, 5));
		smallPane.add(scrollPane, BorderLayout.CENTER);
		JPanel optButtons = new JPanel(new GridLayout(0, 2));
		optButtons.add(addButton);
		optButtons.add(removeButton);
		optButtons.add(upButton);
		optButtons.add(downButton);
		optButtons.add(viewButton);
		smallPane.add(optButtons, BorderLayout.SOUTH);
		smallPane.setPreferredSize(new Dimension(200, 150));
		add(smallPane, BorderLayout.CENTER);
		setMinimumSize(new Dimension(100, 150));
	}

	/**
	 * Select files to load when the browse button is clicked.
	 * 
	 * @param event
	 *            browse button clicked
	 */
	public void actionPerformed(ActionEvent event) {
		if (event.getSource() == addButton) {
			
			if(XnatLoginPanel.loggedInStatus) {
				File[] files = openFileChooser();
				
				if (files != null)
					if (files.length > 0) {
						for (File f : files) {
							
							listBoxEntries.addElement(f);
						
						}
						commit();
					}
			} else {
				JOptionPane.showMessageDialog(new JFrame(), "First login by selecting Project > Xnat Login/Logout");
			}
		} else if (event.getSource() == removeButton) {
			Object[] selected = field.getSelectedValues();
			if (selected != null)
				if (selected.length > 0) {
					for (Object obj : selected)
						listBoxEntries.removeElement(obj);
					commit();
				}
		} else if (event.getSource() == upButton) {
			int index1 = field.getSelectedIndex();
			if (index1 != -1) {
				int index2 = (index1 - 1 + listBoxEntries.size()) % listBoxEntries.size();
				Object obj1 = listBoxEntries.elementAt(index1);
				Object obj2 = listBoxEntries.elementAt(index2);
				listBoxEntries.setElementAt(obj1, index2);
				listBoxEntries.setElementAt(obj2, index1);
				field.setSelectedIndex(index2);
				commit();
			}
		} else if (event.getSource() == downButton) {
			int index1 = field.getSelectedIndex();
			if (index1 != -1) {
				int index2 = (index1 + 1) % listBoxEntries.size();
				Object obj1 = listBoxEntries.elementAt(index1);
				Object obj2 = listBoxEntries.elementAt(index2);
				listBoxEntries.setElementAt(obj1, index2);
				listBoxEntries.setElementAt(obj2, index1);
				field.setSelectedIndex(index2);
				commit();
			}
		} else if (event.getSource() == viewButton) {
			if(XnatLoginPanel.loggedInStatus) {
				File f = JistPreferences.getPreferences().getLastLayoutDirectory();
				File file = new File(f.getAbsolutePath() + File.separator + "XnatQuery.txt");
				BufferedReader br = null;
				
				ArrayList<String> queryResults = new ArrayList<String>();
				queryResults.clear();
				
				for(int i=0; i<listBoxEntries.size(); i++) {
					File fil = (File) listBoxEntries.get(i);
					
					String line = null;
					ArrayList<String> temp = new ArrayList<String>();
					temp.clear();
					try {
						br = new BufferedReader(new FileReader(file));
					} catch (FileNotFoundException e1) {
						JOptionPane.showMessageDialog(new JFrame(), "Xnat Query file not found.");
					}
					try {
						while ((line = br.readLine()) != null) {
							
							String[] parts = line.split(";");
							
							
							if(parts[0].equals(fil.getName())) {
								
								temp.add(line);
							} 
						}
						
					} catch (IOException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
					if(temp.size() != 0) {
						queryResults.add(temp.get(0));
					}
				}
				if(queryResults.size() != 0) {
					displayQuery(queryResults);
				}
				
			} else {
				JOptionPane.showMessageDialog(new JFrame(), "First login by selecting Project > Xnat Login/Logout");
			}
		}
	}

	private void displayQuery(final ArrayList<String> queryResults) {
		JFrame frame = new JFrame("Query Results");
		String[] col = {"Project","Subjects", "Experiments", "Scans", "Resources"};
		DefaultTableModel tableModel = new DefaultTableModel(col, 0);
		for(String row : queryResults) {
			String[] parts = row.split(";");
			String s = parts[1];
			String e = parts[2];
			String sc = parts[3];
			String rs = parts[4];
			String p = parts[5];
			Object[] obj = {p, s, e, sc, rs};
			tableModel.addRow(obj);
		}
		
		JTable table = new JTable(tableModel);
				
		ListSelectionModel selectionModel = table.getSelectionModel();
		selectionModel.setSelectionInterval(0, 0);
		table.setPreferredScrollableViewportSize(new Dimension(600, 200));
		JScrollPane tableScrollPane = new JScrollPane(table);
		Object[] options = {"Re-Download", "Cancel"};
		int response = JOptionPane.showOptionDialog(frame, tableScrollPane, 
	               "Query Results", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE,
	               null, options, null);
		
		if(response == 0) {
			SwingWorker<Void, Void> mySwingWorker = new SwingWorker<Void, Void>(){
		         @Override
		         protected Void doInBackground() throws Exception {
		        	 
		        	 for (String row : queryResults) {
		     			String[] parts = row.split(";");
		     			String s = parts[1];
		     			String e = parts[2];
		     			String sc = parts[3];
		     			String rs = parts[4];
		     			String project = parts[5];
		     			String jsessionID = XnatLoginPanel.jsessionID;
		     			String host = XnatLoginPanel.host;
		     			ArrayList<String> filename = new ArrayList<String>();
		     			filename.clear();
		     			//String filePath = null;
		     			
		     			try {
		     				filename = XnatMain.download(s, e, sc, rs, project, jsessionID, host);
		     				
		     				
		     			} catch (IOException e1) {
		     				
		     			}
		     			
		     			
		     		}
		     			
		     			
		        	 
		        	return null;
		         }
		      };

		      Window win = SwingUtilities.getWindowAncestor(frame);
		      final JDialog dialog = new JDialog(win, "Dialog", ModalityType.APPLICATION_MODAL);

		      mySwingWorker.addPropertyChangeListener(new PropertyChangeListener() {

		         public void propertyChange(PropertyChangeEvent evt) {
		            if (evt.getPropertyName().equals("state")) {
		               if (evt.getNewValue() == SwingWorker.StateValue.DONE) {
		                  dialog.dispose();
		               }
		            }
		         }
		      });
		      mySwingWorker.execute();

		      JProgressBar progressBar = new JProgressBar();
		      progressBar.setIndeterminate(true);
		      JPanel panel = new JPanel(new BorderLayout());
		      panel.add(progressBar, BorderLayout.CENTER);
		      panel.add(new JLabel("Please wait......."), BorderLayout.PAGE_START);
		      dialog.add(panel);
		      dialog.pack();
		      dialog.setLocationRelativeTo(win);
		      dialog.setVisible(true);
			
		      
		}
		
	}

	/**
	 * Commit changes to this parameter view.
	 */
	public void commit() {
		
		updateParameter();
		
	}

	/**
	 * Use a custom cell renderer that can interpret ParamFiles.
	 * 
	 * @param list
	 *            list box
	 * @param value
	 *            list box entry
	 * @param index
	 *            selected index
	 * @param isSelected
	 *            is selected
	 * @param cellHasFocus
	 *            has focus
	 * @return the list cell renderer component
	 */
	public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected,
			boolean cellHasFocus) {
		// Get the selected index. (The index parameter isn't
		// always valid, so just use the value.)
		File f = (value instanceof ParamFile) ? ((ParamFile) value).getValue() : (File) value;
		
		JLabel listLabel ;
		if(f!=null)
			listLabel = new JLabel(f.getName());
		else
			listLabel = new JLabel("null");
		JPanel pane = new JPanel(new BorderLayout());
		if (isSelected) {
			pane.setBackground(list.getSelectionBackground());
			pane.setForeground(list.getSelectionForeground());
		} else {
			pane.setBackground(list.getBackground());
			pane.setForeground(list.getForeground());
		}
		pane.add(listLabel,BorderLayout.CENTER);
		return pane;
	}

	/**
	 * Get file parameter.
	 * 
	 * @return the parameter
	 */
	public ParamXnatFileCollection getParameter() {
		return (ParamXnatFileCollection) param;
	}

	/**
	 * Open file chooser to select file with specific extension.
	 * 
	 * @return absolute path of the file
	 */
	private File[] openFileChooser() {
		/*
		JFileChooser openDialog = new JFileChooser();
		FileExtensionFilter filter = (getParameter().getExtensionFilter() != null) ? getParameter()
				.getExtensionFilter() : null;
		if (filter != null) {
			openDialog.setFileFilter(filter);
		}
		openDialog.setSelectedFile(MipavController.getDefaultWorkingDirectory());
		openDialog.setMultiSelectionEnabled(true);
		openDialog.setDialogTitle("Select File(s)");
		openDialog.setFileSelectionMode(JFileChooser.FILES_ONLY);
		openDialog.setDialogType(JFileChooser.OPEN_DIALOG);
		openDialog.setCurrentDirectory(JistPreferences.getPreferences().getLastDirectory());
		int returnVal = openDialog.showOpenDialog(this);
		*/
		File[] files = null;
		obj = new XnatFileObject();
		XnatSearchController xnat = new XnatSearchController();
		obj = xnat.getFiles();
		files = obj.files;
		//selectedResults.clear();
		
		
		/*
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			files = openDialog.getSelectedFiles();
			if (files.length > 0)
				JistPreferences.getPreferences().setLastDirectory(files[0].getParentFile());
			return files;
		} else {
			return null;
		}*/
		
		return files;
	}

	/**
	 * Update pane with new value from parameter.
	 */
	public void update() {
		ParamXnatFileCollection p = getParameter();
		// Get current volumes in volume collection
		List<ParamFile> vols = p.getParameters();
		listBoxEntries.clear();
		for (ParamFile vol : vols) {
			listBoxEntries.addElement(vol);
		}
	}

	/**
	 * Update parameter value with selected items from list box.
	 */
	protected void updateParameter() {
		ParamXnatFileCollection p = getParameter();
		p.clear();
		Enumeration en = listBoxEntries.elements();
		while (en.hasMoreElements()) {
			Object obj = en.nextElement();
			if (obj instanceof ParamFile) {
				p.add(((ParamFile) obj).getValue());
			} else {
				p.add((File) obj);
			}
		}
		
		notifyObservers(p, this);
	}

	/**
	 * Update parameter when the list box selection changes.
	 * 
	 * @param event
	 *            selection changed
	 */
	public void valueChanged(ListSelectionEvent event) {
		updateParameter();
	}
	/**
	 * Get field used to enter this value
	 */
	public JComponent getField() {
		return field;
	}
}
