/**
 * JIST Extensions for Computer-Integrated Surgery
 *
 * Center for Computer-Integrated Surgical Systems and Technology &
 * Johns Hopkins Applied Physics Laboratory &
 * The Johns Hopkins University
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 * @author Blake Lucas
 */
#if ATI
	#pragma OPENCL EXTENSION cl_khr_3d_image_writes : enable 
#endif 
#define WORLD_RADIUS 1000.f
#define WORLD_CENTER ((float4)(0.f, -WORLD_RADIUS - 2.f, 0.f, 0.f))
#define BOUNDING_RADIUS_2 4.f
#define ESCAPE_THRESHOLD 1e1f
#define DELTA 1e-4f
typedef struct {
    float x, y, z; // position, also color (r,g,b)
} Vec;

typedef struct {
    Vec orig, target;
    Vec dir, x, y;
} Camera;

typedef struct {
    unsigned int width, height;
    int superSamplingSize;
    int actvateFastRendering;
    int enableShadow;

    unsigned int maxIterations;
    float epsilon;
    float color[4];
    float light[3];
    Camera camera;
} RenderingConfig;

typedef struct{
	float2 particle;
	float2 mapping;
    float2 vertexes[2];
    float phi;
    float psi; //not used, but it makes the number of values even!
} Springl2D;

inline int getIndex(int i, int j) {
	return (j * ROWS) + i;
}
inline void getRowCol(uint index,int* i, int* j) {
	(*j)=index/ROWS;
	(*i)=index-(*j)*ROWS;
}
inline float getImageValue(__global float* image,int i,int j){
	int r = clamp((int)i,(int)0,(int)(ROWS-1));
	int c = clamp((int)j,(int)0,(int)(COLS-1));
	return image[getIndex(r,c)];
}
kernel void copyCapsulesToMesh(

		global int* labels,
		global Springl2D* capsules,
		global float2* vertexBuffer,
		global float2* particleBuffer,
		global float2* normalSegmentBuffer,
		global float2* mapPointBuffer,
		uint elements){
	uint id=get_global_id(0);
	if(id>=elements)return;
	capsules+=id;
	vertexBuffer+=id*2;
	float2 particle=SCALE_UP*capsules->particle;
	float2 mapping=SCALE_UP*capsules->mapping;
	particleBuffer[id]=particle;
	//float2 tanget=capsules->vertexes[1]-capsules->vertexes[0];
	//tanget=0.5f*normalize(tanget);
	vertexBuffer[0]=SCALE_UP*capsules->vertexes[0];
	vertexBuffer[1]=SCALE_UP*capsules->vertexes[1];
	float2 norm;
	norm.x=vertexBuffer[0].y-vertexBuffer[1].y;
	norm.y=vertexBuffer[1].x-vertexBuffer[0].x;
	norm=normalize(norm);
	
	normalSegmentBuffer[2*id]=particle;
	normalSegmentBuffer[2*id+1]=particle+0.5f*norm;
	
	mapPointBuffer[2*id]=particle;
	mapPointBuffer[2*id+1]=mapping;

}
kernel void springlsContourRenderer(const global float* levelset,global float4* pixelBuffer,float r,float g,float b,float alpha){
    uint id = get_global_id(0);
   	int i,j;
	getRowCol(id,&i,&j);
	float level=levelset[id];
	float weight=tanh(fabs(level));
	float4 color=(float4)(r,g,b,alpha);
	if(level<=0){
		pixelBuffer[id]=weight*color+(1-weight)*(float4)(0,0,0,alpha);
	} else {
		pixelBuffer[id]=(1-weight)*(float4)(0,0,0,alpha);
	}
}