function ssfmri(varargin)
%------------------------------------------------------------------------
%    Spatio-Spectral Mixed Effects Model for fMRI (ssfmri) Toolbox 
%-------------------------------------------------------------------------
%                           ssfmri
% work with spm8
% For a detailed description, refer to:
%
% Spatio-Spectral Mixed Effects Model for Functional Magnetic Resonance
% Imaing Data
% 
% Hakmook Kang, Hernando Ombao, Crystal Linkletter, Nicole Long, David
% Badre
%
% and
% --my paper
%-------------------------------------------------------------------------
%  
%  Medical-image Analysis and Statistical Interpretation Lab (MASI) &
%  Biostatistics
%
%  Vanderbilt University
%  Nashville, TN, USA
%--------------------------------------------------------------------------
%            ssfmri can be executed from the command line 
%--------------------------------------------------------------------------  
%   Format: 
%   ssfmri(SSF)
%--------------------------------------------------------------------------
%  Examples:
%    >> ssfmri; %-launches GUI  
%    >> ssfmri(SSF);
%   
% -------------------------------------------------------------------------
%  Input Parameters:
%
%  SSF
%--------------------------------------------------------------------------
%             ssfmri can be executed from the command line 
%   SSF structure:
%   SSF.result_dir  - the path and name of the directory where results will be
%                   stored.
%
%   SSF.type        - type of fMRI analysis
%                   'REST' : resting state fMRI
%                   'EVENT': task fMRI (may not work for this version yet)
%
%   SSF.imgname     - a cell array containing image names. each cell is the 
%                   full path and name of one scan image
%
%   SSF.RT          - fMRI repetition time
%
%   SSF.seed        - The seed file name. It stores the seed time course
%
%   SSF.titles      - a cell array of names assigned to the contrast
%
%   SSF.conf        - the file name of other confounds. One column
%                   represents one confound
%
%   SSF.ROI         - ROI structure
%                   ROI.Label   : The label image corresponding to fMRI
%                   ROI.LabelVal: The labeling intensity value of the ROIs we are
%                   interested in
%
%   SSF.band        - a cell array for frequency index. 
%                   e.g., SSF.band{1} = 2:99; (1 is the constant part index)
%
%   SSF.mask        - path and file name of the file containing the brain
%                   mask. set to '' if do not provide user defined mask
%
%   SSF.nband       - number of frequency bands
%
%   SSF.Dpattern    - optional. The priors for between ROI covariance
%                   estimation. It will be set to '' if no this field in
%                   input.
%
%   SSF.Type_STAT   - 'T', only t-test has been implemented for this version
%
%   SSF.test        - a cell array containing test structure. 
%                   SSF.test{i}.con    : contrast vector for coeffiecients
%                   SSF.test{i}.H0     : 'ONE' for one sided t-test or 'TWO'
%                   for two sided t-test
%                   SSF.test{i}.correct: multiple correction. only
%                   impelmented 'FDR' for this version
%                   SSF.test{i}.alpha  : the FDR threshold
%--------------------------------------------------------------------------

global startdir

startdir = pwd;

if nargin < 1 % launch GUI
    ssftitlebar_text = 'Spatio-Spectral Model (fMRI) Toolbox';
    color1 = [.663, .663, .675];
    color2 = [.5, .5, .5];
    color3 = [1 1 1];
    buttoncolor = [0 0 0];
    ssfversion = 'Version 1';
    version_color = [0, 0, 0];
%
% create FIGURE based on SPM Interactive Window size
%
    I = spm('WinSize', 'Interactive');

    B = [I(1), I(2)+I(4)+50, I(3), I(4)/2];
    f = figure('Position', B, ...
               'Name', ssftitlebar_text, ...
               'Color', color1,...
               'HandleVisibility', 'on',...
               'IntegerHandle', 'off',...
               'Renderer', 'painters',...
               'NumberTitle', 'off',...
               'MenuBar', 'none', ...
               'Toolbar', 'none' ... 
               );
%
% create PANEL to hold buttons
%
	edge = B(4)/10;
	panelw = B(3) - edge * 2;
	panelh = B(4) - edge * 2;
	p = uicontrol(f, 'Style', 'frame', ...
        'BackgroundColor', color2, ...
        'Position', [edge, edge, panelw, panelh], ...
        'Parent', f);
%
% create version label
%
    version_pos = [edge * 2, panelh - edge * 1.5, panelw - edge * 2, edge * 2];
    versionLabel = uicontrol(f, 'Style', 'text', ...
        'ForegroundColor', version_color, ...
        'BackgroundColor', color2, ...
        'HorizontalAlignment', 'c', ...
        'Position', version_pos, ...
        'String', ssfversion, ...
        'Parent', f);
%
% create BUTTONS
%
    nbutton = 3;
    buttonmargin = edge * 1.5;
    buttonw = (panelw - buttonmargin*(nbutton+1))/nbutton;
    buttonh = edge * 2.25;
    buttonx = edge + buttonmargin;
    buttony = edge + panelh/2 - buttonh * .75;
    button1 = uicontrol(f, 'Style', 'pushbutton', ...
        'Position', [buttonx, buttony, buttonw, buttonh], ...
        'BackgroundColor', buttoncolor, ...
        'ForegroundColor', color3, ...
        'String', 'fMRI Analysis', ...
        'Parent', f, ...
        'Callback', @button1Callback);
	
    buttonx = buttonx + buttonw + buttonmargin;
    button2 = uicontrol(f, 'Style', 'pushbutton', ...
        'Position', [buttonx, buttony, buttonw, buttonh], ...
        'BackgroundColor', buttoncolor, ...
        'ForegroundColor', color3, ...
        'String', 'Inference', ...
        'Parent', f, ...
        'Callback', @button2Callback);
    
    buttonx = buttonx + buttonw + buttonmargin;
    button3 = uicontrol(f, 'Style', 'pushbutton', ...
        'Position', [buttonx, buttony, buttonw, buttonh], ...
        'BackgroundColor', buttoncolor, ...
        'ForegroundColor', color3, ...
        'String', 'Results', ...
        'Parent', f, ...
        'Callback', @button3Callback);

%
% set FONT for buttons
%
    if isunix
        uihandles = findall(f, 'Type', 'uicontrol');
        set(uihandles, 'FontName', 'Helvetica', 'FontSize', 10);
     end
else % INVOKE COMMAND LINE OPTIONS
    ssfmri_analysis(varargin{:});
end
%
% BUTTON CALLBACKS
%
function button1Callback(~, ~)
global startdir
if isempty(startdir), startdir = pwd; end
disp('calling ssfmri_analysis ...');
cd(startdir);
ssfmri_analysis;
disp('all done ... analysis');
%
function button2Callback(~, ~)
disp('calling contrast inference ...');
ssfmri_con_man;
disp('all done ... contrast inference');

function button3Callback(~, ~)
disp('calling results viewer ...');
ssfmri_results;
disp('all done ... ');