function boost_file = run_adaboost_training(est_seg, json_fnames, ...
                                             feature_vols, out_dir, ...
                                             runtype, varargin)
% RUN_ADABOOST_CORRECTION - runs the adaboost segmentation correction
%
% json_fnames = run_adaboost_training(est_seg, json_fnames, ...
%                                     feature_vols, out_dir, ...
%                                     runtype, varargin)
%
% Input: est_seg - estimated segmentation filename
%        json_fnames - cell array of json filenames
%        feature_vols - a cell array of feature volumes (can be empty)
%        out_dir - the output directory
%        runtype - either 'cluster' or 'single'
%        opts - (optional) a struct of options
%
% Output: json_fnames
%

namestr = 'AdaBoost-Correction';

% do some error checking
if ~exist(out_dir, 'dir')
    error(sprintf('Output directory %s does not exist', out_dir));
end

% set the opts struct
if length(varargin) == 0
    opts = struct;
else
    opts = varargin{1};
end

% set the default options
if ~isfield(opts, 'use_xvfb')
    if strcmp(runtype, 'cluster')
        opts.use_xvfb = true;
    else
        opts.use_xvfb = false;
    end
end
if ~isfield(opts, 'memval'), opts.memval = {'5G', '8G'}; end
if ~isfield(opts, 'pbsfile')
    opts.pbsfile = sprintf('%s-pbsfile.pbs', namestr);
end
if ~isfield(opts, 'txtout')
    opts.txtout = sprintf('%s-txtout.txt', namestr);
end
if ~isfield(opts, 'waitdone')
    opts.waitdone = true;
end
if ~isfield(opts, 'mipav'), opts.mipav = [getenv('HOME'), '/mipav/']; end
if ~isfield(opts, 'jvmmemval'), opts.jvmmemval = '5000m'; end
if ~isfield(opts, 'plugindir'), opts.plugindir = [opts.mipav, '/plugins/']; end
if ~isfield(opts, 'debuglvl'), opts.debuglvl = 2; end

% set the output directories
boost_dir = sprintf('%s/%s/', out_dir, namestr);
temp_dir = [out_dir, '/temp-out/'];

% first make sure the output directories exist
if ~exist(boost_dir, 'dir')
    mkdir(boost_dir);
end
if ~exist(temp_dir, 'dir')
    mkdir(temp_dir);
end

done_list = zeros([1 1]);
chk_files = cell([1 1]);

% set the command prefix
mipavjava_cmd = sprintf(['%s -Xms%s -XX:+UseSerialGC -Xmx%s -classpath %s:%s:', ...
                         '`find %s -name \\*.jar | grep -v Uninstall | ', ...
                         'sed "s#%s#:%s#" | tr -d "\\n" | sed "s/^://"`'], ...
                         javaloc, opts.jvmmemval, opts.jvmmemval, opts.plugindir, ...
                         opts.mipav, opts.mipav, opts.mipav, opts.mipav);
run_str = 'edu.jhu.ece.iacl.jist.cli.run';
class_str = 'edu.vanderbilt.masi.plugins.adaboost.PluginAdaBoostApply';
full_run_str = sprintf('%s %s %s', mipavjava_cmd, run_str, class_str);

featurevol_str = '';
if length(feature_vols) > 0
    for i = 1:length(feature_vols)
        featurevol_str = sprintf('%s%s;', featurevol_str, feature_vols{i});
    end
    featurevol_str = featurevol_str(1:end-1);
end
json_str = '';
for i = 1:length(json_fnames)
    json_str = sprintf('%s%s;', json_str, json_fnames{i});
end
json_str = json_str(1:end-1);

% create the output directory
out_temp_dir = sprintf('%s/%s-%s/', temp_dir, namestr, get_basename(est_seg));
if ~exist(out_temp_dir, 'dir')
    mkdir(out_temp_dir);
end

% set the output training file
boost_file = sprintf('%s%s_AdaBoost.nii.gz', boost_dir, get_basename(est_seg));
if ~strcmp(boost_file(1), '/'), boost_file = fullfile(pwd, boost_file); end
j = 1;

% let the user know what is going on
tprintf('-> Running AdaBoost Correction\n');
tprintf('Mipav Location: %s\n', opts.mipav);
tprintf('Estimated Segmentation: %s\n', est_seg);
tprintf('Number of JSON Files: %d\n', length(json_fnames));
tprintf('Number of Feature Volumes: %d\n', length(feature_vols));
tprintf('Output Parameter File: %s\n', boost_file);

% make sure it needs to be run
chk_files{1}{1} = boost_file;
if files_exist(chk_files{j})
    tprintf('Skipping AdaBoost Correction: (output files exist)\n');
    tprintf('\n');
    pause(0.01);
    done_list(j) = 1;
    return;
end

opts_str = '';
opts_str = sprintf('%s -inEstimated "%s"', opts_str, est_seg);
if length(feature_vols) > 0
    opts_str = sprintf('%s -inFeature "%s"', opts_str, featurevol_str);
end
opts_str = sprintf('%s -inJSON "%s"', opts_str, json_str);
opts_str = sprintf('%s -xDebugLvl %d', opts_str, opts.debuglvl);
cmds{1} = sprintf('%s %s -outCorrected %s\n', ...
                  full_run_str, opts_str, boost_file);

% enable running jist on the cluster
if opts.use_xvfb && length(cmds) == 1
    bname = sprintf('%s_%s', namestr, get_basename(est_seg));
    cmds = prepend_jist_cluster_prefix(cmds, 0.5*j, '/tmp', bname);
end

% run the commands
if (length(cmds) > 0)
    txtout = [out_temp_dir, opts.txtout];
    if strcmp(runtype, 'cluster')
        pbsout = [out_temp_dir, opts.pbsfile];
        run_cmd_cluster(cmds, opts.memval, pbsout, txtout);
        done_list(j) = 0;
    else
        run_cmd_single(cmds, txtout);
        tprintf('Correction finished (see %s)\n', txtout);
        done_list(j) = 1;
    end
else
    done_list(j) = 1;
end

tprintf('\n');

% iterate until done
if (opts.waitdone)
    while (min(done_list) == 0)
        for i = 1:length(done_list)
            if (done_list(i) == 0)
                if files_exist(chk_files{i})
                    done_list(i) = 1;
                end
            end
        end
        pause(1);
    end
end

