function json_fnames = run_adaboost_training(man_segs, est_segs, ...
                                             feature_vols, out_dir, ...
                                             runtype, varargin)
% RUN_ADABOOST_TRAINING - runs the adaboost training for segmentation correction
%
% json_fnames = run_adaboost_training(man_segs, est_segs, ...
%                                     feature_vols, out_dir, ...
%                                     runtype, varargin)
%
% Input: man_segs - a cell array of manual segmentation filenames
%        est_segs - a cell array of estimated segmentation filenames
%        feature_vols - a cell array of feature volumes (can be empty)
%        out_dir - the output directory
%        runtype - either 'cluster' or 'single'
%        opts - (optional) a struct of options
%
% Output: json_fnames
%

namestr = 'AdaBoost-Training';

% do some error checking
if ~exist(out_dir, 'dir')
    error(sprintf('Output directory %s does not exist', out_dir));
end
if length(est_segs) ~= length(man_segs)
    error('Manual and estimated segmentations are differing lengths');
end

% set the opts struct
if length(varargin) == 0
    opts = struct;
else
    opts = varargin{1};
end

% set the default options
if ~isfield(opts, 'use_xvfb')
    if strcmp(runtype, 'cluster')
        opts.use_xvfb = true;
    else
        opts.use_xvfb = false;
    end
end
if ~isfield(opts, 'memval'), opts.memval = {'5G', '8G'}; end
if ~isfield(opts, 'pbsfile')
    opts.pbsfile = sprintf('%s-pbsfile.pbs', namestr);
end
if ~isfield(opts, 'txtout')
    opts.txtout = sprintf('%s-txtout.txt', namestr);
end
if ~isfield(opts, 'waitdone')
    opts.waitdone = true;
end
if ~isfield(opts, 'labels')
    tprintf('-> Determining Number of Labels to Classify\n');
    opts.labels = [];
    for i = 1:length(est_segs)
        a = load_untouch_nii_gz(est_segs{i});
        opts.labels = union(opts.labels, unique(a.img(:)));
    end
    clear a;
    tprintf('Number of Labels: %d\n', length(opts.labels));
end
if ~isfield(opts, 'dilation'), opts.dilation = 1; end
if ~isfield(opts, 'featurerad'), opts.featurerad = [2 2 2]; end
if ~isfield(opts, 'numiters'), opts.numiters = 50; end
if ~isfield(opts, 'maxsamples'), opts.maxsamples = 500000; end
if ~isfield(opts, 'equalclass'), opts.equalclass = 'true'; end
if ~isfield(opts, 'equalexample'), opts.equalexample = 'true'; end
if ~isfield(opts, 'mipav'), opts.mipav = [getenv('HOME'), '/mipav/']; end
if ~isfield(opts, 'jvmmemval'), opts.jvmmemval = '5000m'; end
if ~isfield(opts, 'plugindir'), opts.plugindir = [opts.mipav, '/plugins/']; end
if ~isfield(opts, 'debuglvl'), opts.debuglvl = 2; end

% set the output directories
boost_dir = sprintf('%s/%s/', out_dir, namestr);
temp_dir = [out_dir, '/temp-out/'];

% first make sure the output directories exist
if ~exist(boost_dir, 'dir')
    mkdir(boost_dir);
end
if ~exist(temp_dir, 'dir')
    mkdir(temp_dir);
end

done_list = zeros([length(opts.labels) 1]);
chk_files = cell([length(opts.labels) 1]);

% set the command prefix
mipavjava_cmd = sprintf(['%s -Xms%s -XX:+UseSerialGC -Xmx%s -classpath %s:%s:', ...
                         '`find %s -name \\*.jar | grep -v Uninstall | ', ...
                         'sed "s#%s#:%s#" | tr -d "\\n" | sed "s/^://"`'], ...
                         javaloc, opts.jvmmemval, opts.jvmmemval, opts.plugindir, ...
                         opts.mipav, opts.mipav, opts.mipav, opts.mipav);
run_str = 'edu.jhu.ece.iacl.jist.cli.run';
class_str = 'edu.vanderbilt.masi.plugins.adaboost.PluginAdaBoostTrain';
full_run_str = sprintf('%s %s %s', mipavjava_cmd, run_str, class_str);

manseg_str = '';
estseg_str = '';
featurevol_str = '';
for i = 1:length(est_segs)
    manseg_str = sprintf('%s%s;', manseg_str, man_segs{i});
    estseg_str = sprintf('%s%s;', estseg_str, est_segs{i});
end
manseg_str = manseg_str(1:end-1);
estseg_str = estseg_str(1:end-1);
if length(feature_vols) > 0
    for i = 1:length(feature_vols)
        featurevol_str = sprintf('%s%s;', featurevol_str, feature_vols{i});
    end
    featurevol_str = featurevol_str(1:end-1);
end

% do all of the labels separately
for j = 1:length(opts.labels)

    % create the output directory
    out_temp_dir = sprintf('%s/%s-%04d/', temp_dir, namestr, opts.labels(j));
    if ~exist(out_temp_dir, 'dir')
        mkdir(out_temp_dir);
    end

    % set the output training file
    boost_file = sprintf('%s%s-%04d.json', boost_dir, namestr, opts.labels(j));
    if ~strcmp(boost_file(1), '/'), boost_file = fullfile(pwd, boost_file); end
    json_fnames{j} = boost_file;

    % let the user know what is going on
    tprintf('-> Processing Label %d\n', opts.labels(j));
    tprintf('Mipav Location: %s\n', opts.mipav);
    tprintf('Number of Examples: %d\n', length(man_segs));
    tprintf('Number of Feature Volumes: %d\n', length(feature_vols));
    tprintf('Output Parameter File: %s\n', boost_file);

    % make sure it needs to be run
    chk_files{j}{1} = boost_file;
    if files_exist(chk_files{j})
        tprintf('Skipping AdaBoost Training: %d (output files exist)\n', ...
                opts.labels(j));
        tprintf('\n');
        pause(0.01);
        done_list(j) = 1;
        continue;
    end

    opts_str = '';
    opts_str = sprintf('%s -inManual "%s"', opts_str, manseg_str);
    opts_str = sprintf('%s -inEstimated "%s"', opts_str, estseg_str);
    if length(feature_vols) > 0
        opts_str = sprintf('%s -inFeature "%s"', opts_str, featurevol_str);
    end
    opts_str = sprintf('%s -inTarget %d', opts_str, opts.labels(j));
    opts_str = sprintf('%s -inDilation %d', opts_str, opts.dilation);
    opts_str = sprintf('%s -inFeature2 %d', opts_str, opts.featurerad(1));
    opts_str = sprintf('%s -inFeature3 %d', opts_str, opts.featurerad(2));
    opts_str = sprintf('%s -inFeature4 %d', opts_str, opts.featurerad(3));
    opts_str = sprintf('%s -inNumber %d', opts_str, opts.numiters);
    opts_str = sprintf('%s -inMaximum %d', opts_str, opts.maxsamples);
    opts_str = sprintf('%s -inUse %s', opts_str, opts.equalclass);
    opts_str = sprintf('%s -inUse2 %s', opts_str, opts.equalexample);
    opts_str = sprintf('%s -xDebugLvl %d', opts_str, opts.debuglvl);
    cmds{1} = sprintf('%s %s -outAdaBoost %s\n', ...
                      full_run_str, opts_str, boost_file);

    % enable running jist on the cluster
    if opts.use_xvfb && length(cmds) == 1
        bname = sprintf('%s_%04d', namestr, opts.labels(j));
        cmds = prepend_jist_cluster_prefix(cmds, 0.5*j, '/tmp', bname);
    end

    % run the commands
    if (length(cmds) > 0)
        txtout = [out_temp_dir, opts.txtout];
        if strcmp(runtype, 'cluster')
            pbsout = [out_temp_dir, opts.pbsfile];
            run_cmd_cluster(cmds, opts.memval, pbsout, txtout);
            done_list(j) = 0;
        else
            run_cmd_single(cmds, txtout);
            tprintf('Fusion finished (see %s)\n', txtout);
            done_list(j) = 1;
        end
    else
        done_list(j) = 1;
    end

    tprintf('\n');

end

% iterate until done
if (opts.waitdone)
    while (min(done_list) == 0)
        for i = 1:length(done_list)
            if (done_list(i) == 0)
                if files_exist(chk_files{i})
                    done_list(i) = 1;
                end
            end
        end
        pause(1);
    end
end

