function [cmds chk_files] = ...
                run_ants_strip_registration(target, atlas, labels, ...
                                            out_prefix, out_type, ...
                                            out_warp_dir, out_seg_dir, ...
                                            out_atlas_dir, out_temp_dir,...
                                            regloc, opts);
% RUN_ANTS_STRIP_REGISTRATION - runs the ANTS-strip registration algorithm
%                               between a given target and an atlas with labels
%
% [cmds chk_files] = run_ants_strip_registration(target, atlas, labels, ...
%                                                out_prefix, out_type, ...
%                                                out_warp_dir, out_seg_dir, ...
%                                                out_atlas_dir, ...
%                                                out_temp_dir, regloc, opts);
%
% Input: target - the target filename
%        atlas - the atlas filename
%        labels - the atlas labels filename
%        out_prefix - the output prefix for the filenames
%        out_type - the output filetype (1 - .nii, 2 - .nii.gz)
%        out_warp_dir - the output directory for the warp files
%        out_seg_dir - the output directory for the segmentation files
%        out_atlas_dir - the output directory for the registred atlas files
%        out_temp_dir - the output directory to store the temporary files
%        regloc - the location of the registration application
%        opts - the registration-specific options
%
% Output: cmds - a cell array of commands to run
%         chk_files - a cell array of files that should be there when done
%

% set some filenames
wrp_pfix = sprintf('%s/%s_', out_warp_dir, out_prefix);
aff_name = sprintf('%s/%s_Affine.txt', out_warp_dir, out_prefix);
wrp_name = sprintf('%s/%s_Warp.nii.gz', out_warp_dir, out_prefix);
if out_type == 1
    seg_name = sprintf('%s/%s_seg.nii', out_seg_dir, out_prefix);
    reg_name = sprintf('%s/%s.nii', out_atlas_dir, out_prefix);
else
    seg_name = sprintf('%s/%s_seg.nii.gz', out_seg_dir, out_prefix);
    reg_name = sprintf('%s/%s.nii.gz', out_atlas_dir, out_prefix);
end

chk_files{1} = seg_name;
chk_files{2} = reg_name;
chk_files{3} = aff_name;
chk_files{4} = wrp_name;

% set the default options
if ~isfield(opts, 'simmetric')
    opts.simmetric = 'CC';
end
if ~isfield(opts, 'simopts')
    opts.simopts = '1,4';
end
if ~isfield(opts, 'addopts')
    opts.addopts = ['-r Gauss[3,0] -t SyN[0.25] -i 100x100x100x20 ', ...
                    '--use-Histogram-Matching '];
end
if ~isfield(opts, 'target_strip_dir')
    error('opts.target_strip_dir must be defined');
end
if ~isfield(opts, 'atlas_strip_dir')
    error('opts.atlas_strip_dir must be defined');
end

opts.newopts = ['-r Gauss[3,0] -t SyN[0.25] -i 0 ', ...
                '--use-Histogram-Matching ', ...
                '--number-of-affine-iterations ', ...
                '10000x10000x10000x10000x10000'];

% set the locations of the ART executables
ANTS_reg = [regloc, '/bin/ANTS'];
ANTS_apply = [regloc, '/bin/WarpImageMultiTransform'];

[a b c] = fileparts(atlas);
temp_atlas = [opts.atlas_strip_dir, '/', b, c];
[a b c] = fileparts(target);
temp_target = [opts.target_strip_dir, '/', b, c];

cmds = {};

% run the initial registration
cmds{1} = sprintf('%s 3 -m %s[%s,%s,%s] -o %s %s\n', ANTS_reg, ...
                  opts.simmetric, temp_target, temp_atlas, opts.simopts, ...
                  wrp_pfix, opts.newopts);

% run the registration
cmds{2} = sprintf(['%s 3 -m %s[%s,%s,%s] -o %s -a %s --continue-affine ', ...
                   'false %s\n'], ANTS_reg, opts.simmetric, target, atlas, ...
                  opts.simopts, wrp_pfix, aff_name, opts.addopts);

% apply to the atlas
cmds{3} = sprintf('%s 3 %s %s -R %s %s %s\n', ANTS_apply, atlas, reg_name, ...
                  target, wrp_name, aff_name);

% apply to the labels
cmds{4} = sprintf('%s 3 %s %s -R %s %s %s --use-NN\n', ANTS_apply, labels, ...
                  seg_name, target, wrp_name, aff_name);


