function obs = add_obs(obs, data, varargin)
% ADD_OBS - Adds an observation to a previously created observation struct
%
% There are two possible forms for this function:
% - For slice-type observation structs
% obs = add_obs(obs, data, slice, rater);
%
% Input: obs - The previous observation struct
%        data - a 2D matrix of observed labels
%        slice - the associated slice number
%        rater - the associated rater number
%
% - For volume-type observation structs
% obs = add_obs(obs, data);
%
% Input: obs - The previous observation struct
%        data - A 3D matrix of observed labels
%
% Output: obs - The output observation struct with the added observation

if ~isstruct(obs)
    error('The input argument "obs" should be an observation struct (see create_obs.m');
end

% if the observation struct is of slice-type
if strcmp(obs.type, 'slice')

    % make sure that they used the right form
    if length(varargin) ~= 2
        error('Should use the input form obs = add_obs(obs, data, slice, rater)');
    end

    % make sure that the slice number makes sense
    if (varargin{1} < 1 | varargin{1} > obs.dims(3))
        error('The slice should be a number between 1 and the number of slices');
    end

    % make sure the data is the right size
    if min(size(data) == obs.dims(1:2)) ~= 1
        error('The input data should be same size as the dimensions for the observations');
    end

    obs.num_obs = obs.num_obs + 1;
    obs.data{obs.num_obs} = uint8(data);
    obs.slices(obs.num_obs) = varargin{1};
    obs.raters(obs.num_obs) = varargin{2};
elseif strcmp(obs.type, 'volume')

    % make sure they used the right form
    if length(varargin) ~= 0
        error('Should use the input form obs = add_obs(obs, data)');
    end

    % make sure the data is the right size
    sd = size(data);
    if length(sd) == 2
        sd = [sd 1];
    end
    if min(sd == obs.dims) ~= 1
        error('The input data should be same size as the dimensions for the observations');
    end

    % add the actual observation
    obs.num_obs = obs.num_obs + 1;
    obs.data{obs.num_obs} = uint8(data);
else

    % something strange is going on
    error('The observations struct should be of type "slice" or "volume"');

end

