function [fraccorrect] = fraction_correct(truth, estimate, varargin)
% FRACTION_CORRECT - calculate the Fraction voxels correct
%
% [fraccorrect] = fraction_correct(truth, estimate)
%
% [fraccorrect] = fraction_correct(truth, estimate, obs)
%
% [fraccorrect] = fraction_correct(truth, estimate, obs, 'confusion')
%
% Assumes: truth, estimate are the same dimension, orientation, etc...
%          only interested in the labels that are unique to the truth
% Input: truth - truth volume/vector/matrix
%        estimate - estimate volume/vector/matrix
%        (optional) obs - the observations used to construct the estimate
%                         returns fraction correct over the actual "volume"
%        (optional) 'confusion' - indicator flag that indicates that you only
%                                 want the fraction correct over non-consensus
%                                 voxels
%
% Output: fraccorrect - The fraction of total voxels that are the same

if (nargin == 2)
    err = abs(double(truth) - double(estimate));
elseif length(varargin) == 1

    vals = zeros(size(truth));
    obs = varargin{1};
    if strcmp(obs.type, 'slice')

        for o = 1:obs.num_obs
            vals(:, :, obs.slices(o)) = vals(:, :, obs.slices(o)) + ...
                                        abs(double(obs.data{o}));
        end
        err = abs(double(truth(vals>0)) - double(estimate(vals>0)));

    elseif strcmp(obs.type, 'volume')

        for o = 1:obs.num_obs
            vals = vals + abs(double(obs.data{o}));
        end
        err = abs(double(truth(vals > 0)) - double(estimate(vals > 0)));

    else
        error('Unrecognized Observation type');
    end
else

    obs = varargin{1};

    W = zeros([size(truth), length(unique(truth))]);
    if strcmp(obs.type, 'volume')
        for o = 1:obs.num_obs
            for x = 1:obs.dims(1)
                for y = 1:obs.dims(2)
                    for z = 1:obs.dims(3)
                        W(x, y, z, obs.data{o}(x, y, z)+1) = ...
                                    W(x, y, z, obs.data{o}(x, y, z)+1) + 1;
                    end
                end
            end
        end

        W = max(W, [], 4);
        err = abs(double(truth(W < obs.num_obs)) - ...
                  double(estimate(W < obs.num_obs)));
    end

end

fraccorrect = length(err(err == 0)) / numel(err);

