function final = canny3D(im, filterprops, thresholds)
% CANNY3D - a simple function to detect edges in a 3D matrix
%   Format: final = canny3D(im, filterprops, thresholds)
%   
% arguments:
%   im - 3D matrx
%   filterprops - 2x1 array of Gaussian filter properties [size sigma]        
%       size - filter size (diameter) in voxels (default: 5)
%       sigma - standard dev of Gaussian filter size (default: 1.5)
%   thresholds - 2x1 array of edge thresholds [uppper lower]
%       upper - upper filter boundary 
%       lower - lower filter boundary 
%
%   If th_up and th_low aren't chosen, they are set automatically
%-------------------------------------------------------------------------%

%-------------------------------------------------------------------------%
%   Created by:
%   University of Illinois at Urbana Champaign
%   Department of Mechanical Science and Engineering
%   Laboratory of Photonics Research on Bio/Nano Environment
%   Written by : Tung Yuen Lau
%   Advisor: Prof. Kimani Toussaint
%   Start date: FEB 16 2012
%      
%   Modified by: Frederick Bryan, Vanderbilt, July 2013
%-------------------------------------------------------------------------%


% account for optional vars
if nargin < 2 || ~isequal(size(filterprops), [1 2]);
    filterprops = [5 1.5];
end
if nargin < 3 || ~isequal(size(thresholds), [1 2]);
    thresholds = 'auto'; 
end


% cast  & rename variables
im = double(im);
filsize = filterprops(1);
sigma = filterprops(2);

% Gaussian filter
hfil = floor(filsize/2);
[x y z] = meshgrid(-hfil:hfil,-hfil:hfil,-hfil:hfil);
fil_x = exp(-x.^2/(2*sigma^2))./(sigma*sqrt(2*pi)); clear x;
fil_y = exp(-y.^2/(2*sigma^2))./(sigma*sqrt(2*pi)); clear y;
fil_z = exp(-z.^2/(2*sigma^2))./(sigma*sqrt(2*pi)); clear z;
f = fil_x .* fil_y .* fil_z; clear fil_x; clear fil_y; clear fil_z;
f = f/sum(abs(f(:)));
imfil = imfilter(im,f,'replicate'); %clear f; clear im;

% find xyz gradients (could be re-implemented using sobel, but why?)
[imfil_x , imfil_y, imfil_z] = gradient(imfil); %clear imfil;

% Thinning (non-maximum suppression)
im_thin = nonmax_sup(imfil_x,imfil_y,imfil_z);

% find magnitude
imfil_mag = sqrt(imfil_x.^2+imfil_y.^2+imfil_z.^2);

% hysterisis
im_sub = hysteresis(im_thin,imfil_mag,thresholds);

% output
final = im_sub;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% nonmaximag suppression (thinning) 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function im_thin = nonmax_sup(gx,gy,gz)

% find gradient magnitude 
gmag = sqrt(gx.^2+gy.^2+gz.^2);

% sort of non-maximal suppression? Seems right.enough
[w,h,d] = size(gx);
[x,y,z] = meshgrid(1:h,1:w,1:d);

xi = x - gx./gmag;
yi = y - gy./gmag;
zi = z - gz./gmag;
imtemp = interp3(x,y,z,gmag,xi,yi,zi);

xi = x + gx./gmag;
yi = y + gy./gmag;
zi = z + gz./gmag;
imtemp2 = interp3(x,y,z,gmag,xi,yi,zi);

im_thin = gmag & ((imtemp<gmag) & (imtemp2<gmag));

% 
% THIS DIDNT WORK AND I GAVE UP
%
% % find angles in spherical coords
% [th, phi, r] = cart2sph(gx, gy, gz);
% 
% % the matrix way of thinking and the xyz way of thinking break down
% % because the x-axis increases "down", so we have to flip our angles;
% th = -th; 
% 
% % half range to only represent 1/2 of all angles : 
% % th:[-pi pi]->[0 pi], phi:[-pi/2 p/2]->[0 pi/2]
% th=th+pi*(th<0);
% % phi=phi+pi/2*(phi<0); 
% 
% % round to nearest pi/2 (both rh & phi)
% % 0, pi/4, pi/2, 3pi/4
% th((th<=pi/8)|(th>7*pi/8))=0;
% th((th>1*pi/8)&(th<=3*pi/8))=pi/4;
% th((th>3*pi/8)&(th<=5*pi/8))=pi/2;
% th((th>5*pi/8)&(th<=7*pi/8))=3*pi/4; 
% 
% phi((phi<-3*pi/8)|(phi>=3*pi/8))=pi/2; %top and bottom quadrants
% phi((phi>=pi/8)&(phi<3*pi/8))=pi/4; %45deg
% phi((phi>=-pi/8)&(phi<pi/8))=0; % 0 deg
% phi((phi>=-3*pi/8)&(phi<-pi/8))=-pi/4;
% 
% % preallocate output as no local maxima
% im_thin = false(size(gx));
% 
% % go through all 13 (angle) cases and determine if maximal or not?
% % ii=1;
% for th_ang = [0 pi/4 pi/2 3*pi/4];
%     for phi_ang = [-pi/4 0 pi/4 pi/2];
%         inds = (th==th_ang) & (phi==phi_ang); 
%         
% %         % testing for complete coverage of voxels
% %         a(ii)=sum(inds(:));
% %         ii=ii+1;
% %         im_thin = im_thin+inds; % should be all ==1 when done.
% %         phi = phi(:,:,10); th= th(:,:,10); 
% %         gmag=gmag(:,:,10); phi_ang = 0; th_ang = 0;
% %         phi=zeros(size(phi)); 
% 
% 
%         % handle xy-plane shifting
%         if th_ang == 0; % want East-West
%             shift = [0 1 0];
%             indsa = circshift(inds,shift);
%             indsb = circshift(inds,-shift); 
%         elseif th_ang == pi/4; % NE-SW
%             shift = [-1 1 0];
%             indsa = circshift(inds,shift);
%             indsb = circshift(inds,-shift);
%         elseif th_ang == pi/2; % N-S
%             shift = [-1 0 0];
%             indsa = circshift(inds,shift);
%             indsb = circshift(inds,-shift);
%         elseif th_ang == 3*pi/4; % NW-SE 
%             shift = [-1 1 0]; 
%             indsa = circshift(inds,shift);
%             indsb = circshift(inds,-shift);
%         end
%         
%         %handle z-shifting
%         if phi_ang == pi/2; % "above" (in the 3rd dimension sense)
%             shift = [0 0 1];
%             indsa = circshift(inds,shift); 
%             indsb = circshift(inds,-shift); 
%         elseif phi_ang == pi/4; % "above" at an angle
%             shift = [0 0 1];
%             indsa = circshift(indsa,shift);
%             indsb = circshift(indsb, -shift); 
%         elseif phi_ang == 0; % in xy plane
%             % nothing
%         elseif phi_ang == -pi/4;
%             shift = [0 0 -1];
%             indsa = circshift(indsa,shift);
%             indsb = circshift(indsb, -shift);
%         end
%         
%         % find max of these 3 neighbors
%         mag = gmag.*inds; 
%         maga = gmag.*indsa;
%         magb = gmag.*indsb; 
%         tmp = cat(4,maga,mag,magb); 
%         [stuff,I]=max(tmp,[],4); 
%         
%         % this will put a 1 wherever we were wondering AND it was maximum
%         im_thin(inds&(I==2))=true;
%         
%         figure; 
%         imagesc(inds(:,:,15)); 
%         title(sprintf('th = %3.2fpi, phi = %3.2fpi',th_ang/pi, phi_ang/pi));
%         
%     end
% end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% hysteresis function
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function maskout = hysteresis(maskin,im_mag,thresholds)

% im - binary image of possible edges
% im_mag - gradient magnitude image
% thresholds - [high & low] thresholds for hysteresis
keyboard; 

% create thresholds, if missing
if strcmp(thresholds, 'auto');                                            
    % normalize gradient image (& remove outliers)
    im_mag = img_normalize(im_mag);
    
    % convert to probability
    im_mag = exp(-1*(max(im_mag(:))-im_mag)); 
    
    % set thresholds
    thresholds = [0.5 0.99];
end

th_up = thresholds(2); 
th_low = thresholds(1); 

% high threshold (will definitely be edges)
maskout = (im_mag > th_up);

[w h d] = size(maskout);
index = find(maskout);
old_index = [];

while length(old_index)~=length(index)
    old_index = index;
    x = mod(mod(index,w*h),h);
    y = ceil(mod(index,w*h)/h);
    z = ceil(index/w/h);
    % for 3x3x3 neighborhood
    for i = -1:1
        for j = -1:1
            for k = -1:1
                xtemp = x+i;
                ytemp = y+j;
                ztemp = z+k;
                ind = find(xtemp>=1 & xtemp<=w & ytemp>=1 & ytemp<=h & ...
                    ztemp>=1 & ztemp<=d);
                xtemp = xtemp(ind);
                ytemp = ytemp(ind);
                ztemp = ztemp(ind);
                mask(xtemp+(ytemp-1)*w+(ztemp-1)*h*w) = ...
                    im_mag(xtemp+(ytemp-1)*w+(ztemp-1)*h*w)>th_low;
            end
        end
    end
    index = find(mask);
end



function out = img_normalize(in)
% out = IMG_NORMALIZE(in)
% utility function used to normalize intensity to between 0.5 and 99.5
% percentile and remap everyting [0 1]

% cast to double
in = double(in);

% saturate btwen .5 and 99.9 percentile
lowval = prctile(in(:), .5);
highval = prctile(in(:), 99.9);
out = in;
out(in<lowval) = lowval;
out(in>highval) = highval;

% scale to one;
out = (out-lowval)/(highval-lowval);



