function [obs_raw obs_seg] = apply_2D_affine_model(img, seg, res, num_obs, ss);
% APPLY_2D_AFFINE_MODEL - Applies a random 2D affine transformation to an input
%                         observation (containing an image and segmentation).
%
% [obs_raw obs_seg] = apply_2D_affine_model(img, seg, res, num_obs, ss);
%
% Input: img - the reference image
%        seg - the reference segmentation
%        res - the resolution of the reference image
%        num_obs - the number of observations to generate
%        ss - the standard deviation of the various parameters
%           - ss(1) standard deviation of the rotation (in degrees)
%           - ss(2) standard deviation of the x translation (in mm)
%           - ss(3) standard deviation of the y translation (in mm)
%           - ss(4) standard deviation of the x scaling (in mm)
%           - ss(5) standard deviation of the y scaling (in mm)
% Output: obs_raw - the simulated image observations
%         obs_seg - the simulated segmentation observations
%

dims = size(img);
if length(dims) ~= 2
    error('This only works for 2D images');
end

xl = linspace(-dims(1)/2, dims(1)/2, dims(1));
yl = linspace(-dims(2)/2, dims(2)/2, dims(2));
[X Y] = ndgrid(xl, yl);

% pre-allocate
obs_raw = zeros([dims num_obs]);
obs_seg = zeros([dims num_obs]);
rand_vals = randn(num_obs, 5);

for i = 1:num_obs

    % construct the affine matrix
    theta = ss(1) * rand_vals(i, 1);
    tx = ss(2) * (1/res(1)) * rand_vals(i, 2);
    ty = ss(3) * (1/res(2)) * rand_vals(i, 3);
    sx = ss(4) * (1/res(1)) * rand_vals(i, 4) + 1;
    sy = ss(5) * (1/res(2)) * rand_vals(i, 5) + 1;
    mat = [sx * cosd(theta), sy * -sind(theta), tx; ...
           sx * sind(theta), sy * cosd(theta), ty; ...
           0, 0, 1];

    % apply the transformation to the coordinates
    vec = [X(:)'; Y(:)'; ones(size(X(:)'))];
    vals = mat * vec;
    Xp = reshape(vals(1, :)', size(X));
    Yp = reshape(vals(2, :)', size(Y));

    % interpolate the new image/segmentation
    obs_raw(:, :, i) = interpn(X, Y, img, Xp, Yp, 'linear');
    obs_seg(:, :, i) = interpn(X, Y, seg, Xp, Yp, 'nearest');
end

obs_raw(isnan(obs_raw)) = -1024;
obs_seg(isnan(obs_seg)) = 0;
