function [obs tobs cms ppvs truth] = label_variation_simulation(...
                                                         num_raters, dims, ...
                                                         label_size, ...
                                                         ppv_min, ppv_max)
% LABEL_VARIATION_SIMULATION - create observations using a user defined label
%                              size for any number of labels
% - This is mainly meant to be used to compare the STAPLE and STAPLER models
%
% [obs tobs cms ppvs truth] = label_variation_simulation(num_raters, dims, ...
%                                                        label_size, ...
%                                                        ppv_min, ppv_max);
%
% Input: num_raters - number of raters to observe
%        dims - dimensions of the volume of the form [x y z]
%        label_size - A vector of relative fractions for a series of labels
%                     NOTE: sum(label_size) must equal 1, if not, then they will
%                     be normalized to 1.
%        ppv_min - Positive Predictive Value for worst rater
%        ppv_max - Positive Predictive Value for best rater
%
% Output: obs - the observation struct
%         tobs - the training observation struct
%         cms - the true confusion matrices
%         ppvs - the PPV matrices
%         truth - the true "segmentation"

% handle the label size vector
if min(label_size < 0)
    error('All elements in label_size must be greater than 0');
end
label_size = label_size / sum(label_size);
num_labels = length(label_size);

%
% create the confusion matrices
%
diag_vals = linspace(ppv_min, ppv_max, num_raters);

% allocate space for the confusion matrices
cms = zeros([num_labels num_labels num_raters]);
ppvs = zeros([num_labels num_labels num_raters]);

% iterate over all of the raters
for i = 1:num_raters
    % set the diagonal values
    ppv = diag(diag_vals(i) * ones([1 num_labels]));

    % fill in the off-diagonal values
    for j = 1:num_labels
        % generate (num_labels - 1) values to fill in the off-diagonals
        nums = rand([1 (num_labels-1)]);

        % calculate how much probability is available
        ava = 1 - diag_vals(i);

        % normalize these values so that we can just slap them into the cm
        nums = ava * (nums / sum(nums));

        count = 0;
        for k = 1:num_labels
            if (j ~= k)
                count = count + 1;
                ppv(j, k) = nums(count);
            end
        end
    end
    ppvs(:,:,i) = ppv;
end
% iterate over all raters
cms = ppvs;
for i = 1:num_raters
    cm = cms(:, :, i);
    cm = cm ./ repmat(sum(cm, 1), [num_labels 1]);
    cms(:, :, i) = cm;
end

% generate the truth volume
truth = zeros(dims(1), dims(2), dims(3));
for z = 1:dims(3)

    % allocate the current slice
    truth_slice = zeros(dims(1), dims(2));

    % get a random permutation of label numbers
    rp = randperm(num_labels);

    % take the cumulative sum
    cumul_label_size = cumsum(label_size(rp));

    for l = 1:length(rp)

        % starting index
        vs = 1;
        if (l > 1)
            vs = ceil(cumul_label_size(l-1) * prod(dims(1:2)));
        end

        % ending index
        ve = prod(dims(1:2));
        if (l < length(rp))
            ve = round(cumul_label_size(l) * prod(dims(1:2)));
        end

        % apply to the slices
        truth_slice(vs:ve) = rp(l) - 1;
    end

    truth(:, :, z) = truth_slice;
end

%
% Testing data

% iterate over all raters
data = zeros(dims(1), dims(2), dims(3), num_raters);
for r = 1:num_raters
    % apply the confusion matrix to the truth
    data_vol = apply_voxelwise_random_model(truth+1, squeeze(cms(:,:,r))) - 1;
    data(:,:,:,r) = data_vol;
end

% create the observations
obs = create_obs('slice', dims);
for r = 1:num_raters
    for s = 1:dims(3)
        obs = add_obs(obs, data(:, :, s, r), s, r);
    end
end

%
% Training Data
%

% iterate over all raters
data = zeros(dims(1), dims(2), dims(3), num_raters);
for r = 1:num_raters
    % apply the confusion matrix to the truth
    data_vol = apply_voxelwise_random_model(truth+1, squeeze(cms(:,:,r))) - 1;
    data(:,:,:,r) = data_vol;
end

% create the observations
tobs = create_obs('slice', dims);
for r = 1:num_raters
    for s = 1:dims(3)
        tobs = add_obs(tobs, data(:, :, s, r), s, r);
    end
end

