function [obs truth] = multi_COLLATE_simulation(alphas, num_raters)
% MULTI_COLLATE_SIMULATION - Multi-COLLATE model observation simulation
% - Meant to compare the COLLATE family of algorithms
%
% [obs truth] = multi_COLLATE_simulation(alphas, num_raters)
%
% Input: alphas - the alpha values used in Multi-COLLATE (4 element array)
%        num_raters - the number of unique observations
%
% Output: obs - the observation struct
%         truth - the true "segmentation"
%

% main settings
num_labels = 5;
dims = [90 90 60];
truth = zeros(dims);

%
% first set the truth model
%
rp1 = [0 1]; %randperm(num_labels-3)-1;
rp2 = [2 0 1 3 4]; %randperm(num_labels)-1;
rp3 = [0 1 2 3]; %randperm(num_labels-1)-1;
rp4 = [2 1 0]; %randperm(num_labels-2)-1;

% set the first quadrant
xs = round(1);
xe = round(dims(1) / 2);
shift = floor((dims(2) / 2) / (length(rp1)));
for l = 0:(length(rp1)-1)
    ys = round(shift * (l) + 1);
    ye = round(ys + shift - 1);
    truth(xs:xe, ys:ye, :) = rp1(l+1);
end

% set the second quadrant
ys = round(1);
ye = floor(dims(2) / 2) - 1;
shift = floor((dims(1) / 2) / length(rp2));
for l = 0:(length(rp2)-1)
    xs = floor(dims(2) / 2 + shift * (l) + 1);
    xe = ceil(xs + shift - 1);
    truth(xs:xe, ys:ye, :) = rp2(l+1);
end

% set the third quadrant
ys = ceil(dims(1) / 2);
ye = round(dims(1));
shift = ((dims(1) / 2) / length(rp3));
for l = 0:(length(rp3)-1)
    xs = floor(shift * (l) + 1);
    xe = ceil(xs + shift - 1);
    truth(xs:xe, ys:ye, :) = rp3(l+1);
end

% set the fourth quadrant
xs = round(dims(1) / 2 + 1);
xe = round(dims(1));
shift = floor((dims(2) / 2) / length(rp4));
for l = 0:(length(rp4)-1)
    ys = round(dims(2) / 2 + shift * (l));
    ye = round(ys + shift - 1);
    truth(xs:xe, ys:ye, :) = rp4(l+1);
end
truth = uint8(truth);

x1s = 1;
x1e = round(dims(1) / 2);
y1s = 1;
y1e = round(dims(1) / 2);
t1 = truth(x1s:x1e, y1s:y1e, :);

x2s = round(dims(1) / 2) + 1;
x2e = dims(1);
y2s = 1;
y2e = round(dims(1) / 2);
t2 = truth(x2s:x2e, y2s:y2e, :);

x3s = 1;
x3e = round(dims(1) / 2);
y3s = round(dims(2) / 2) + 1;
y3e = dims(2);
t3 = truth(x3s:x3e, y3s:y3e, :);

x4s = round(dims(1) / 2) + 1;
x4e = dims(1);
y4s = round(dims(2) / 2) + 1;
y4e = dims(2);
t4 = truth(x4s:x4e, y3s:y4e, :);

main_diag = linspace(0.2, 0.4, num_raters);

% observe the data in this wild manner
obs = create_obs('volume', dims);

for r = 1:num_raters

     cm = generate_random_confusion_matrix(num_labels, main_diag(r), ...
                                           ones([num_labels]));
     cm1 = (alphas(1) * eye(length(cm)) + cm) / (1 + alphas(1));
     cm2 = (alphas(2) * eye(length(cm)) + cm) / (1 + alphas(2));
     cm3 = (alphas(3) * eye(length(cm)) + cm) / (1 + alphas(3));
     cm4 = (alphas(4) * eye(length(cm)) + cm) / (1 + alphas(4));

     data(x1s:x1e, y1s:y1e, :) = apply_voxelwise_random_model(t1, cm4);
     data(x2s:x2e, y2s:y2e, :) = apply_voxelwise_random_model(t2, cm1);
     data(x3s:x3e, y3s:y3e, :) = apply_voxelwise_random_model(t3, cm2);
     data(x4s:x4e, y4s:y4e, :) = apply_voxelwise_random_model(t4, cm3);

     obs = add_obs(obs, data);
end
