/*=========================================================================
 
 Program:   Insight Segmentation & Registration Toolkit
 Module:    $RCSfile: itkComputeSHCoefficientsFilter.txx,v $
 Language:  C++
 Date:      $Date: 2006/01/11 19:43:31 $
 Version:   $Revision: 1.21 $
 
 Copyright (c) Insight Software Consortium. All rights reserved.
 See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.
 
 This software is distributed WITHOUT ANY WARRANTY; without even 
 the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the above copyright notices for more information.
 
 =========================================================================*/
#ifndef _itkComputeSHCoefficientsFilter_txx
#define _itkComputeSHCoefficientsFilter_txx

#include "itkComputeSHCoefficientsFilter.h"
#include "itkImageRegionIterator.h"
#include "math.h"
#include "vnl/algo/vnl_matrix_inverse.h"

namespace itk
{
    template< class TInputImage, class TOutputImage >
    ComputeSHCoefficientsFilter< TInputImage, TOutputImage >
    ::ComputeSHCoefficientsFilter()
    {
        if( TInputImage::ImageDimension != 3 )
            itkExceptionMacro( << "This filter is only supported for input dimension 3" );
        m_List.clear();
        m_BValues.resize(0);
        m_Lambda = 0.006f;
        m_DecompositionType = cOPDT;
        m_L = 6;;
        m_Baselines.clear();
        m_Gradients.clear();
    }
    
    template< class TInputImage, class TOutputImage >
    void ComputeSHCoefficientsFilter< TInputImage, TOutputImage >
    ::GenerateOutputInformation( void )
    {
        Superclass::GenerateOutputInformation();
        this->GetOutput()->SetVectorLength( shmaths::getNumberOfEvenAssociatedLegendrePolynomials( this->GetL() ) );
    }
    
    template< class TInputImage, class TOutputImage >
    void ComputeSHCoefficientsFilter< TInputImage, TOutputImage >
    ::BeforeThreadedGenerateData( void )
    {
        Superclass::BeforeThreadedGenerateData();
        // We have to configure and precompute all the matrixes involved in the computation
        // of SH coefficients. Note that this is done only once, so the performance of the
        // algorithm is not compromised.
        
        // Check if the number of b-values matches the number of DWI channels:
        if( m_List.size() != m_BValues.size() )
            itkExceptionMacro( << "The number of channels does not match the number of b-values provided" );
        // First, determine which channels are actually gradients:
        unsigned int total = m_List.size();
        m_Baselines.clear();
        m_Gradients.clear();
        for( unsigned int k=0; k<total; ++k ){
            double norm = vcl_sqrt( m_List[k][0]*m_List[k][0] + m_List[k][1]*m_List[k][1] + m_List[k][2]*m_List[k][2] );
            if( norm>1e-4 )
                m_Gradients.push_back( k );
            else
                m_Baselines.push_back( k );
        }
        // So the actual number of gradient directions is:
        unsigned int N = m_Gradients.size();
        // Now, generate gx, gy, and gz for each gradient direction:
        double* gx = new double[N];
        double* gy = new double[N];
        double* gz = new double[N];
        for( unsigned int k=0; k<N; ++k ){
            gx[k] = m_List[ m_Gradients[k] ][0];
            gy[k] = m_List[ m_Gradients[k] ][1];
            gz[k] = m_List[ m_Gradients[k] ][2];
        }
        // Now, compute the spherical coordinates corresponding to these gradient directions:
        double* theta = new double[N];
        double* phi   = new double[N];
        shmaths::computeShericalCoordsFromCartesian( gx, gy, gz, theta, phi, N );
        // From the spherical coordinates, we can compute all required matrixes (standard convention):
        MatrixType L;
        shmaths::computeSHMatrixSymmetric( N, theta, phi, this->GetL(), m_B  ); // B has size N x vectorLengthOfTheOutput
        shmaths::computeSHEigMatrixSymmetric( this->GetL(), L );                // L has size vectorLengthOfTheOutput x vectorLengthOfTheOutput
        // Generate the matrix solving the regularized LS problem:
        m_LS  = ( m_B.transpose() * m_B + (L*L)*m_Lambda );
        // Note that the inversion of m_LS is done only once, so performance is secundary.
        // In particular, we can use the SVD-based vnl_matrix_inverse fnction instead of
        // vnl_inverse
        m_LS  = vnl_matrix_inverse<double>( m_LS );
        // And, finally:
        m_LS *= ( m_B.transpose() );
        // If only the ADC is needed, it is enough to compute this matrix; however, we
        // consider the case when either Q-Balls, the cOPDT, the pOPDT, or MTIME need to be
        // computed. In these cases, it is necessary to compute the FRT matrix for SH:
        if( m_DecompositionType != ADC   &&   m_DecompositionType != ATT   &&   m_DecompositionType != ATTforOPDT ){
            MatrixType F;
            shmaths::computeSHFRTMatrixSymmetric( this->GetL(), F );
            if( m_DecompositionType == QBall )
                m_LS = F*m_LS;
            else
                m_LS = ( F*(L*m_LS) )*( 0.0625/PI/PI );
        }
        // Delete the memory previously allocated:
        delete[] gx;
        delete[] gy;
        delete[] gz;
        delete[] theta;
        delete[] phi;
    }
    
    template< class TInputImage, class TOutputImage >
    void ComputeSHCoefficientsFilter< TInputImage, TOutputImage >
#if ITK_VERSION_MAJOR < 4
    ::ThreadedGenerateData( const OutputImageRegionType& outputRegionForThread, int itkNotUsed(threadId) )
#else
    ::ThreadedGenerateData( const OutputImageRegionType& outputRegionForThread, ThreadIdType itkNotUsed(threadId) )
#endif
    {
        // Iterators:
        ImageRegionConstIterator<InputImageType>        bit;  // Iterator for the input image
        ImageRegionIterator<OutputImageType>            it;   // Iterator for the output image
        // Input and output
        InputImageConstPointer   input   =  this->GetInput();
        OutputImagePointer       output  =  this->GetOutput();
        
        bit = ImageRegionConstIterator<InputImageType>( input,  outputRegionForThread );
        it  = ImageRegionIterator<OutputImageType>(     output, outputRegionForThread );
        
        // Precompute a bunch of parameters of interest:
        unsigned int R    = shmaths::getNumberOfEvenAssociatedLegendrePolynomials( this->GetL() );
        unsigned int N    = m_Gradients.size();
        unsigned int B    = m_Baselines.size();
        // The following three parameters are used to deal with scenarios for
        // which different b-values are present in each DWI channel. We track
        // the minimum and maximum b-values in the data set, and care about
        // this situation only if these values are very different:
        double max_b_value   = itk::NumericTraits<double>::min();
        double min_b_value   = itk::NumericTraits<double>::max();
        bool needToNormalize = false;
        // Compute the minimum and maximum b-values, and precompute
        // their inverses to accelerate computations:
        double* inverseB  = new double[ m_BValues.size() ];
        for( unsigned int k=0; k<m_BValues.size(); ++k ){
            // Check for different b-values in each channel:
            if( m_BValues[k] > max_b_value )
                max_b_value = m_BValues[k];
            if( m_BValues[k] < min_b_value && m_BValues[k]>1e-6 ) // Skip baselines!
                min_b_value = m_BValues[k];
            // Compute the inverse of the b-value:
            inverseB[k] = ( m_BValues[k]>1e-6 ? 1.0f/m_BValues[k] : 0 );
        }
        // Do we really need to care about scenarios with different b-values?
        if( max_b_value-min_b_value > 0.01f*max_b_value )
            needToNormalize = true;
        
        // Auxiliar matrices to use with the vnl matrix multiplication methods:
        MatrixType vi(  N, 1 );
        MatrixType vo;
        
        for( bit.GoToBegin(),it.GoToBegin(); !bit.IsAtEnd(); ++bit,++it ){
            InputPixelType  ip = bit.Get();
            OutputPixelType op;
            op.SetSize( R );
            // Average all available baselines:
            double baseline = 0.0f;
            for( unsigned int k=0; k<B; ++k )
                baseline += ip[ m_Baselines[k] ];
            // Check for nans:
            if( baseline>1e-6 )
                baseline        = B/baseline;
            else{
                // This is a background point, so attenuation hould be close to 0
                // Hence, we put a very high value of the baseline to force such
                // attenuation. Note that "baseline" is in fact the inverse of
                // of S_0, so we use: 1/S_0 = 1/1e6 = 1e-6
                baseline        = 1e-6;
            }
            // This variable is only used in the MEANTIME case to compute the mean
            // attenuation and, from it, the diffusivity of the basic tensor:
            double mean_att = itk::NumericTraits<double>::Zero;
            // Compute the auxiliar vector to multiply:
            for( unsigned int k=0; k<N; ++k ){
                // Normalize to compute the attenuation signal, E(q) = S_i/S_0:
                double lval = ip[ m_Gradients[k] ] * baseline;
                // Depending on the type of SH expansion to compute, we need
                // to perform different kinds of operations on E(q).
                if( lval<1e-6 )  // Avoid numerical issues
                    lval = 1e-6;
                if( lval>1 )     // Avoid numerical issues
                    lval = 1;
                // This code is necessary to deal with different b-values; assuming
                // a mono-exponential model, we convert all the attenuations to the
                // maximum b-value. This way, all the attenuations are directly 
                // comparable:
                if( needToNormalize ){
                    lval = ::exp( ::log(lval) * inverseB[ m_Gradients[k] ] * max_b_value );
                    if( lval<1e-6 )  // Avoid numerical issues
                        lval = 1e-6;
                    if( lval>1 )     // Avoid numerical issues
                        lval = 1;
                }
                // Note: in the "ADC" case, max_b_value is exactly the same as all other
                // b-values if needToNormalize is false, and we don't have to care about.
                // Otherwise, the right thing to do is divide by max_b_value, since we have
                // normalized.
                switch( m_DecompositionType ){
                    case ADC:
                        vi( k, 0 ) = -vcl_log( lval ) / max_b_value;
                        break;
                    case QBall:
                        vi( k, 0 ) = lval;
                        break;
                    case MTIME:
                        // We have to compute the cOPDT like in the next case,
                        // but we also need to compute the mean attenuation
                        mean_att  += lval;
                        vi( k, 0 ) = -shmaths::Ein(  -vcl_log( lval )  );
                        break;
                    case cOPDT:
                        vi( k, 0 ) = -shmaths::Ein(  -vcl_log( lval )  );
                        break;
                    case pOPDT:
                        lval = -vcl_log( lval );
                        if ( lval<1e-6 )
                            lval = 1e-6;
                        vi( k, 0 ) = vcl_log( lval );
                        break;
                    case ATT:
                    case ATTforOPDT:
                        vi( k, 0 ) = lval;
                        break;
                    default:
                        break;
                }
            }
            // Multiply by the LS matrix to perform LS fitting:
            vo = m_LS * vi; 
            // If we are computing either the cOPDT or the pOPDT, it is necessary to correct
            // the first element of vi, corresponding to the integral of the radial part of
            // the Laplacian (or, alternatively, the normalization constant so that the 
            // orientation function has integral 1 and hence it is a true PDF).
            if( m_DecompositionType==cOPDT || m_DecompositionType==pOPDT )
                vo( 0, 0 ) = 0.079577471545948; // 1/4/pi
            if( m_DecompositionType==MTIME ){
                // Normalize to obtain the mean_diffusivity:
                mean_att /= N;
                // The mean diffusivity is coded in the DC component (that is simply 1/4/pi):
                vo( 0, 0 ) = mean_att;
            }
            // In case we have computed the attenuation signal for
            // further computation of the OPDT, the DC value will be just
            // ignored (due to the computation of the Laplacian). Hence,
            // we use the first component to encode the mean diffusivity
            // for this voxel:
            if( m_DecompositionType==ATTforOPDT ){
                double mean_diffusivity = 0.0f;
                for( unsigned int k=0; k<N; ++k ){
                    // Normalize to compute the attenuation signal, E(q) = S_i/S_0:
                    double lval = ip[ m_Gradients[k] ] * baseline;
                    if( lval<1e-6 )  // Avoid numerical issues
                        lval = 1e-6;
                    if( lval>1 )     // Avoid numerical issues
                        lval = 1;
                    mean_diffusivity -= ::log( lval )/N;
                }
                vo( 0, 0 ) = ( mean_diffusivity>1e-6 ? mean_diffusivity : 1e-6 );
            }
            // Place these values in the output pixel:
            for( unsigned int k=0; k<R; ++k )
                op[k] = (float)( vo[k][0] );
            it.Set( op );
        }
        
        delete[] inverseB;
    }
    
    
} // end namespace itk


#endif
