#!/usr/bin/perl

use strict;
use warnings;
use Cwd;
use File::Basename;
use File::Find;
use Getopt::Long;

# based on asl_prep bash script

my $FS_DIR = "/apps/freesurfer/bin";
my $AFNI_DIR = "/data/apps/afni/bin";
my $FSL_DIR = "/data/apps/fsl/bin";

$ENV{'FREESURFER_HOME'} = "/apps/freesurfer";
$ENV{'FSLOUTPUTTYPE'} = "NIFTI";

# Set variables
my $LOG="asl_prep.log";
my $log_file = undef;

sub writeLog {
   my ($msg, $consoleLog) = @_;
   $consoleLog ||= 1;
   if ($log_file) {
        if (open(LOG,">>$log_file")) {
            print LOG "$msg\n";
	    close(LOG);
	}
   }
   print "$msg\n" if ($consoleLog);
}

sub error {
   my ($msg,$rc) = @_;
   print STDERR "$msg\n";
   writeLog("ERROR:${msg}",0);
   exit $rc;
}

sub extract_series_name {
   my ($dir) = @_;
   if ($dir =~ /\/(asl[^\/]*)\//) {
       return $1;
   }
   if ($dir =~ /\/(t1)\//) {
       return $1;
   }
}

sub get_first_dicom {
   my ($dir) = @_;
   my @files = ();
   find(sub {push @files, $File::Find::name if (-f) }, $dir);
   foreach my $f (@files) {
      next unless (-f $f);
      if (is_dicom_file($f)) {
         return $f;
      }
   }
   return undef;
}

sub is_dicom_file {
  my ($file) = @_;
  my $rs = `/usr/bin/file $file`;
  chomp($rs);
  my $flag = ($rs =~ /DICOM medical imaging data/);
  if (!$flag && $file =~ /\.dcm$/) {
    $flag = 1;
  }
  return $flag;

}

sub do_dicom_series_to_NIFTI_conversion {
   my ($dicom_dir, $dest_dir) = @_;
   my $cur_dir = getcwd;
   change_dir($dest_dir);
   my $series_name = basename($dicom_dir);
   writeLog(" Creating NIFTI files from DICOM files with mri_convert: $dicom_dir \n");
   my $first_dicom = get_first_dicom($dicom_dir);
   #print "$first_dicom  series_name:${series_name}\n";
   my $cmd = "${FS_DIR}/mri_convert $first_dicom ${series_name}.nii";
   writeLog($cmd);
   `$cmd >> $log_file 2>&1`;
   change_dir($cur_dir);
   return $series_name;
}

sub do_NIFTI_conversion {
   my ($src_dir,$dest_dir) = @_;
   my @dicom_dirs = ();
   find(sub {push @dicom_dirs, $File::Find::name if (-d && basename($File::Find::name) eq 'DICOM') }, $src_dir);
   my @filtered_dirs = ();
   foreach my $dir (@dicom_dirs) {
       push @filtered_dirs, $dir if ($dir =~ /\/asl\w*\// || $dir =~ /\/t1\//); 
   }
   if (@filtered_dirs != 4) {
      error("Incomplete data set:Missing DICOM directories required 4 found " . scalar(@filtered_dirs) . "!",3);
   }
   my $cur_dir = getcwd;
   change_dir($dest_dir);
   
   for my $dicom_dir (@filtered_dirs) {
       my $series_name = extract_series_name($dicom_dir);
       writeLog(" Creating NIFTI files from DICOM files with mri_convert: $dicom_dir \n");
       my $first_dicom = get_first_dicom($dicom_dir);
       my $cmd = "${FS_DIR}/mri_convert $first_dicom ${series_name}.nii";
       writeLog($cmd);
       `$cmd >> $log_file 2>&1`;
       if ($?) {
         error("mri_convert failed!  Please review $log_file for details! Error code:" . ($? & 127), 1);
      }
   }
   change_dir($cur_dir);
}

sub check_oblique {
  my ($nifti_file) = @_;
  my $zero_count = 0;
  foreach my $dir1 (1,2,3) {
    foreach my $dir2 (1,2,3) {
      my $column = $dir2 + 1;
      my @oarr = `${FSL_DIR}/avwhd $nifti_file | grep "qto_xyz:${dir1}" | awk -v col=$column '{ print \$col}'`;
      my $off_diagonal = $oarr[0];
      # print "$dir1 $column $off_diagonal\n";
      if (abs($off_diagonal) == 0.0) {
        $zero_count++;
      }
    }
  }
  print "zero count: $zero_count \n";
  return ($zero_count < 6);
}

sub change_dir {
    my ($dir) = @_;
    writeLog("changing current working dir to:$dir");
    if (!chdir($dir)) {
        error("cannot cd to directory:${dir}!", 1);
    }
}

sub is_rx_same {
   my ($t1_nii,$asl_nii) = @_;
   my @t1_out = `${FSL_DIR}/avwhd $t1_nii | grep "qto_xyz" 2>/dev/null`;
   my @asl_out = `${FSL_DIR}/avwhd $asl_nii | grep "qto_xyz" 2>/dev/null`;
  # print join("",@t1_out) . "\n";
  # print join("",@asl_out) . "\n";
   if (scalar(@t1_out) == scalar(@asl_out)) {
      for(my $i = 0; $i < @t1_out; $i++) {
          return 0 if ($t1_out[$i] ne $asl_out[$i]);
      }
      return 1;
   }
   return 0;
}

sub handle_oblique {
   my ($dest_dir, $asl_series_name, $t1_nifti_fname, $anat_series_name) = @_;
   $t1_nifti_fname ||= "t1.nii";
   my $cur_dir = getcwd;
   change_dir($dest_dir);    
   # Determine if the T1 dataset is oblique
   my $t1_oblique = check_oblique($t1_nifti_fname);
   my $asl_oblique = check_oblique("${asl_series_name}.nii");
   print "\n";
   if (!$t1_oblique && !$asl_oblique) {
      print "  Neither the T1 nor ASL datasets have oblique axes!\n";
      print "  - Copying the t1.nii to anat_ASL.nii ...\n";
      my $cmd = "cp -a ${dest_dir}/$t1_nifti_fname ${dest_dir}/${anat_series_name}.nii";
      writeLog($cmd);
      `$cmd >> $log_file 2>&1`;
      if ($?) {
          error("Linux cp failed!  Please review $log_file for details! Error code:" . ($? >> 8), 1);
      }
   } elsif (!$t1_oblique && $asl_oblique) {
# If the T1 is NOT oblique and the ASL is oblique, then perform a single 3dWarp on the T1
       writeLog("");
       writeLog("  Only the ASL dataset has oblique axes!");
       writeLog("  - Running 3dWarp to warp the non-oblique T1 to the oblique ASL dataset ...");
      my $cmd = "${AFNI_DIR}/3dWarp -card2oblique ${asl_series_name}.nii -prefix  ${anat_series_name}.nii $t1_nifti_fname";
      writeLog($cmd);
      `$cmd >> $log_file 2>&1`;
      if ($?) {
          error("3dWarp failed!  Please review $log_file for details! Error code:" . ($? & 127), 1);
      }
   } elsif ($t1_oblique && !$asl_oblique) {
 # If the T1 is oblique and the ASL is NOT oblique, then perform a single 3dWarp on the T1
      writeLog("");
      writeLog("  Only the T1 dataset has oblique axes!");
      writeLog("  - Running 3dWarp to warp the oblique T1 to the non-oblique ASL dataset ...");
      my $cmd="${AFNI_DIR}/3dWarp -oblique2card -prefix ${anat_series_name}.nii $t1_nifti_fname";
      writeLog($cmd);
      `$cmd >> $log_file 2>&1`;
      if ($?) {
          error("3dWarp failed!  Please review $log_file for details! Error code:" . ($? >> 8), 1);
      }
   } else {
      if (!is_rx_same($t1_nifti_fname,"${asl_series_name}.nii") ) {
         writeLog(""); 
         writeLog("  The T1 and ASL datasets have different oblique axes!");
         writeLog("  - Running 3dWarp to warp the oblique T1 to cardinal axes ...");
         my $cmd = "${AFNI_DIR}/3dWarp -oblique2card -prefix t1_cardinal.nii $t1_nifti_fname";
         writeLog($cmd);
         `$cmd >> $log_file 2>&1`;
         if ($?) {
           error("3dWarp failed!  Please review $log_file for details! Error code:" . ($? >> 8), 1);
         }

         writeLog(""); 
         writeLog("  - Running 3dWarp to warp the cardinal T1 to the oblique ASL dataset ...");
         $cmd="${AFNI_DIR}/3dWarp -card2oblique ${asl_series_name}.nii -prefix ${anat_series_name}.nii t1_cardinal.nii";
         writeLog($cmd);
         `$cmd >> $log_file 2>&1`;
         if ($?) {
           error("3dWarp failed!  Please review $log_file for details! Error code:" . ($? >> 8), 1);
         }
      } else {
         # Rx same 
         writeLog(""); 
         writeLog("  The T1 and ALS datasets have the same oblique axes!");
         writeLog("  - Copying the t1.nii to anat_ASL.nii ...");
         my $cmd="cp -a  $t1_nifti_fname  $anat_series_name}.nii";
         writeLog($cmd);
         `$cmd >> $log_file 2>&1`;
         if ($?) {
           error("Linux cp failed!  Please review $log_file for details! Error code:" . ($? >> 8), 1);
         }
      }
   }
   change_dir($cur_dir);
}

sub runIt {
   my ($cmd) = @_;
   writeLog($cmd);
   `$cmd >> $log_file 2>&1`;
}


sub create_brik {
   my ($dest_dir, $nifti_in, $brik_prefix) = @_;
   my $cur_dir = getcwd;
   change_dir($dest_dir);    
   runIt("${AFNI_DIR}/3dcopy $nifti_in $brik_prefix"); 
   change_dir($cur_dir);
}

sub create_briks {
  my ($dest_dir) = @_;
   my $cur_dir = getcwd;
   change_dir($dest_dir);    
   if (-e "anat_ASL.nii") {
     runIt("${AFNI_DIR}/3dcopy anat_ASL.nii anat_ASL");
   }
   if (-e "asl1.nii") {
    runIt("${AFNI_DIR}/3dcopy asl1.nii ASLbrik");
   }
   if (-e "aslcal1.nii" ) {
     runIt("${AFNI_DIR}/3dcopy aslcal1.nii ASL_CSFbrik");
   }

   if (-e "aslmincon1.nii" ) {
      runIt("${AFNI_DIR}/3dcopy aslmincon1.nii ASL_MinConbrik");
   }

   change_dir($cur_dir);
}

sub usage2() {
die<<EOS
   
   Usage: $0 --dest <dest-dir> --asl-src <asl-dicom-src-dir> --csf-src <csf-dicom-src-dir> --mincon-src <mincon-dicom-src-dir> [--t1-src <t1-dicom-src-dir>] [--anat <anat-series-name>] [--help]
   where 
     --anat <anat-series-name>  (default: anat_ASL) 
EOS
}

sub main2() {
   my $dest_dir = undef;
   my $asl_src_dir = undef;
   my $csf_src_dir = undef;
   my $mincon_src_dir = undef;
   my $t1_src_dir = undef;
   my $anat_series_name = "anat_ASL";
   my $show_usage = 0;
   GetOptions("dest=s" => \$dest_dir,
              "asl-src=s" => \$asl_src_dir,
	      "csf-src=s" => \$csf_src_dir,
	      "mincon-src=s" => \$mincon_src_dir,
	      "t1-src=s" => \$t1_src_dir,
	     "help" => \$show_usage);
   usage2() if ($show_usage);
   usage2() unless($dest_dir);
   mkdir $dest_dir if (! -d $dest_dir);
   $log_file = "$dest_dir/$LOG";

   usage2() unless($asl_src_dir && $csf_src_dir && $mincon_src_dir && 
       -d $asl_src_dir && -d $csf_src_dir && -d $mincon_src_dir);

   my $asl_series_name = basename($asl_src_dir);
   my $csf_series_name = basename($csf_src_dir);
   my $mincon_series_name = basename($mincon_src_dir);
    
   do_dicom_series_to_NIFTI_conversion($asl_src_dir, $dest_dir);
   do_dicom_series_to_NIFTI_conversion($csf_src_dir, $dest_dir);
   do_dicom_series_to_NIFTI_conversion($mincon_src_dir, $dest_dir);
   
   if ($t1_src_dir && -d $t1_src_dir) {
      my $t1_series_name = basename($t1_src_dir);
      do_dicom_series_to_NIFTI_conversion($t1_src_dir, $dest_dir);
      my $t1_nifti_fname = "${t1_series_name}.nii";
      handle_oblique($dest_dir, $asl_series_name, $t1_nifti_fname, $anat_series_name);
      create_brik($dest_dir, "${anat_series_name}.nii",$anat_series_name);
   }
   create_brik($dest_dir, "${asl_series_name}.nii",$asl_series_name);
   create_brik($dest_dir, "${csf_series_name}.nii",$csf_series_name);
   create_brik($dest_dir, "${mincon_series_name}.nii",$mincon_series_name);
   


}


main2();
