#!/usr/bin/perl

use strict;
use warnings;
use File::Find;
use File::Basename;
use Cwd;
use Getopt::Long;

my $AFNI_DIR = "/data/apps/afni/bin";
my $FSL_DIR = "/apps/fsl4/bin";
my $FS_DIR = "/apps/freesurfer/bin";
my $log_file = undef;

$ENV{'PATH'} = $ENV{'PATH'} . ":/apps/freesurfer/bin";

print "PATH=" . $ENV{'PATH'} . "\n";

sub writeLog {
   my ($msg, $consoleLog) = @_;
   $consoleLog ||= 1;
   if ($log_file) {
        if (open(LOG,">>$log_file")) {
            print LOG "$msg\n";
            close(LOG);
        }
   }
   print "$msg\n" if ($consoleLog);
}

sub error {
   my ($msg,$rc) = @_;
   print STDERR "$msg\n";
   writeLog("ERROR:${msg}",0);
   exit $rc;
}

sub change_dir {
    my ($dir) = @_;
    if (!chdir($dir)) {
        error("cannot cd to directory:${dir}!", 1);
    }
}

sub seq_B0_dicoms {
   my ($dicom_dir, $dest_dir, $discard_slices) = @_;
   my @files = ();
   find(sub {push @files, $File::Find::name if (-f && dirname($File::Find::name) eq $dicom_dir)}, $dicom_dir);
   my @dicom_files = ();
   foreach my $file (@files) {
      next if ($file =~/BRIK$/ || $file =~ /HEAD$/ || $file =~ /\.xml$/ || $file =~ /\.gz$/);
      push @dicom_files, $file;
   }
   @files = ();
   my @rel_inst_arr = ();
   my @new_files = ();
   my %seen_map = ();
   foreach my $file (@dicom_files) {
       chomp(my $str = `$AFNI_DIR/dicom_hdr $file | grep "REL Instance"`);
       #print "$str\n";
       if ($str =~ /Number\/\/(\d+)/) {
         my $instance = $1;  
	 push @rel_inst_arr, $instance;
         #print "$file\n";
         #print "rel_instance:$instance\n";
	 my $new_fname = sprintf("%04d.dcm",$instance);
	 unless (exists $seen_map{$new_fname}) {
	   $seen_map{$new_fname}++;
           my $cmd = "cp $file ${dest_dir}/$new_fname";  
	   #print "$cmd\n";
	   `$cmd`;
	   push @new_files, "${dest_dir}/$new_fname"
	 }
       }
   }
   print "rel_instances: " . join(" ",@rel_inst_arr) . "\n";
   print "# dicom files:" . scalar(@dicom_files) . "\n";
   if ($discard_slices) {
     # discard top and bottom three slices (assuming 30 slices)
     my $no_slices = scalar(@new_files);
     my $min_slice_num = 10000;
     my $max_slice_num = -1;
     foreach my $dcm_file (@new_files) {
       my $name = basename($dcm_file);
       $name =~ s/\.dcm$//;
       my $sn = int($name);
       $min_slice_num = $sn if ($sn < $min_slice_num);
       $max_slice_num = $sn if ($sn > $max_slice_num);
     }
     
     foreach my $dcm_file (@new_files) {
       my $name = basename($dcm_file);
       $name =~ s/\.dcm$//;
       my $sn = int($name);
       if ($sn >= $min_slice_num && $sn <= ($min_slice_num + 2)) {
           print "removing $dcm_file\n";
           system("rm -f $dcm_file");
       }
       if ($sn >= ($max_slice_num - 2) && $sn <= $max_slice_num) {
           print "removing $dcm_file\n";
           system("rm -f $dcm_file");
       }
       
  #     if ($name eq "0001" || $name eq "0002" ||
  #        $name eq "0003" || $name eq "0028" ||
  #        $name eq "0029" || $name eq "0030") {
  #          system("rm -f $dcm_file");
  #     }
     }
   }
}

sub runIt {
   my ($cmd) = @_;
   writeLog($cmd);
   `$cmd >> $log_file 2>&1`;
}

sub delete_dir {
   my ($dir) = @_;
   my @files = <$dir/*>;
   foreach my $file (@files) {
      if (-f $file)  {
         unlink($file);
      }
   }
   rmdir $dir;	
}

sub prep4_B0_correction {
   my ($B0_mag_dir,$B0_phase_dir,$dest_dir) = @_;
   my $cur_dir = getcwd;
   mkdir $dest_dir if (! -d $dest_dir);
   change_dir($dest_dir);
   my $cmd = "$AFNI_DIR/to3d -prefix magnitude.nii.gz -assume_dicom_mosaic ${B0_mag_dir}/*.dcm";
   writeLog($cmd);
   `$cmd`;
   $cmd = "$AFNI_DIR/to3d -prefix phase.nii.gz -assume_dicom_mosaic ${B0_phase_dir}/*.dcm";
   writeLog($cmd);
   `$cmd`;
   chomp(my $dphmin=`${FSL_DIR}/fslstats phase.nii.gz -r | awk -F. '{print \$1}'`);
   chomp(my $dphmax=`${FSL_DIR}/fslstats phase.nii.gz -r | awk -F. '{print \$2}' | awk '{print \$2}'`);
   my $dph_diff = $dphmax - $dphmin;
   print "dph_diff:$dph_diff dphmax:$dphmax dphmin:$dphmin \n";
   `${FSL_DIR}/fslmaths phase.nii.gz -sub $dphmin -div $dph_diff -mul 4095 dph_scaled`;
   change_dir($cur_dir);
}

sub dicom_series_2niftigz {
    my ($dicom_dir, $nifti_path, $series_type) = @_;
    my $dest_dir = dirname($nifti_path);
    my $prefix = basename($nifti_path);
    my $cur_dir = getcwd;
    my $tzt = undef;
    if ($series_type eq "asl") {
       $tzt = "24 104 4000";
    } elsif ($series_type eq "csf") {
       $tzt =  "24 1 4000";
    } elsif ($series_type eq "mincon") {
       $tzt =  "24 4 2000";
    } else {
        error("Not a supported series type:$series_type",2);
    } 
    change_dir($dest_dir);
    print "dest_dir:$dest_dir\n";
    my $cmd = "$AFNI_DIR/to3d  -epan -assume_dicom_mosaic -prefix ${prefix} -time:zt ${tzt} seq-z $dicom_dir/*.dcm"; 
   writeLog($cmd);
   `$cmd`;
   change_dir($cur_dir); 
}

sub do_B0_correction {
   my ($src_niftigz_path, $dest_dir, $te_diff, $esp) = @_;
   $te_diff ||= "2.46";
   $esp ||= "0.5";
   my $cur_dir = getcwd;
   error("do_B0_correction:: Missing NIFTI file to apply B0 correction:$src_niftigz_path",1) if (! -f $src_niftigz_path);
   error("do_B0_correction:: No $dest_dir!",1) if (! -d $dest_dir);
   print "cd to $dest_dir\n";
   change_dir($dest_dir);
   my $target_name = basename($src_niftigz_path); 
   if ($target_name =~ /\.nii\.gz/) {
     $target_name =~ s/\.nii\.gz$/_B0.nii.gz/;
   } else {
     $target_name =~ s/\.nii$/_B0.nii.gz/;
   }
   my $target_path = "$dest_dir/$target_name";
   my $cmd = "${FS_DIR}/epidewarp.fsl --mag  magnitude.nii.gz --dph dph_scaled.nii.gz --epi $src_niftigz_path --tediff $te_diff --esp $esp --vsm vsm.nii.gz --epidw $target_path";
   writeLog($cmd);
   `$cmd`;
   change_dir($cur_dir);
   return $target_path;
}


sub has_siemens_B0 {
   my ($visit_dir) = @_;
   my @files = <$visit_dir/*>;
   my @B0_dirs = ();
   my $B0_mag_found = 0;
   my $B0_phase_found = 0;
   foreach my $f (@files) {
       if (-d $f && basename($f) =~ /^B0_/) {
          push @B0_dirs, $f;
	  my $dname = basename($f);
	  $B0_phase_found = 1 if ($dname eq "B0_phase1");
	  $B0_mag_found = 1 if ($dname eq "B0_mag1");
       }
   }
   return ($B0_mag_found && $B0_phase_found);
}

sub already_prepared_4B0 {
   my ($dir) = @_;
   my @files = <$dir/*.nii.gz>;
   my $has_mag = 0;
   my $has_phase = 0;
   my $has_dph_scaled = 0;
   foreach my $f (@files) {
      next unless (-f $f);
      my $name = basename($f);
      $has_mag = 1 if ($name eq "magnitude.nii.gz");
      $has_phase = 1 if ($name eq "phase.nii.gz");
      $has_dph_scaled = 1 if ($name eq "dph_scaled.nii.gz");
   }
   return ($has_mag && $has_phase && $has_dph_scaled);
}

sub create_brik {
   my ($dest_dir, $nifti_in, $brik_prefix) = @_;
   my $cur_dir = getcwd;
   change_dir($dest_dir);  
   my $cmd = "${AFNI_DIR}/3dcopy $nifti_in $brik_prefix";
   writeLog($cmd);
   `$cmd`;
   change_dir($cur_dir);
}

sub usage() {
   die<<EOS
       $0 --visit <visit-dir> --dest <dest-dir> --asl <ASL-NIFTI> |--series <SRC-SERIES-DIR> --type <asl|csf|mincon> [--mag <B0-mag-dir> --phase <B0-phase-dir> ] [--tediff <TE diff> --esp <ESP> --force-prep ]
       where 
       default TE diff = 2.46 and default esp is 0.5
EOS
}
sub main() {
   my $visit_dir = undef;
   my $asl_nifti = undef;
   my $asl_dicom_series = undef;
   my $force_prep = 0;
   my $te_diff = 2.46;
   my $esp = 0.5;
   my $series_type = undef;
   my $dest_dir = undef;
   my $mag_dir = undef;
   my $phase_dir = undef;
   GetOptions("visit=s" => \$visit_dir,
              "dest=s" => \$dest_dir,
              "asl=s" => \$asl_nifti,
	      "series=s" => \$asl_dicom_series,
	      "mag=s" => \$mag_dir,
	      "phase=s" => \$phase_dir,
	      "type=s" => \$series_type,
	      "force-prep" => \$force_prep,
	      "tediff=f" => \$te_diff,
	      "esp=f" => \$esp);
   usage() unless (($visit_dir && -d $visit_dir) || ($mag_dir && -d $mag_dir && $phase_dir && -d $phase_dir));

   usage() unless ($dest_dir && -d $dest_dir);

   if (!$asl_dicom_series) {
      usage() unless ($asl_nifti && -f $asl_nifti && ($asl_nifti =~ /\.nii$/ || $asl_nifti =~ /\.nii.gz$/));
   } 
   usage() unless ($asl_dicom_series || $asl_nifti);
   usage() if ($asl_dicom_series && !$series_type);
   
   if ($visit_dir) {
     unless (has_siemens_B0($visit_dir)) {
        writeLog("No Siemens B0 directories for visit:$visit_dir");
        usage();
     }
   }
   print "esp:$esp TE diff:$te_diff\n";
   $ENV{'FSLOUTPUTTYPE'} = 'NIFTI_GZ';
   
   mkdir $dest_dir unless(-d $dest_dir);

   $mag_dir = "$visit_dir/B0_mag1" unless(defined($mag_dir));
   $phase_dir = "$visit_dir/B0_phase1" unless (defined($phase_dir));
   my $mag_dest_dir = "$dest_dir/B0_mag1";
   my $phase_dest_dir = "$dest_dir/B0_phase1";
   
   mkdir $mag_dest_dir unless (-d $mag_dest_dir);
   mkdir $phase_dest_dir unless (-d $phase_dest_dir);
  
   seq_B0_dicoms($mag_dir, $mag_dest_dir, 1);
   seq_B0_dicoms($phase_dir, $phase_dest_dir, 1);
   
   my $out_dir = "$dest_dir/B0_corrected";
   mkdir $out_dir unless(-d $out_dir);
   if ($force_prep) {
       if (-d $out_dir) {
           delete_dir($out_dir);
       }
   }
   if ($asl_dicom_series) {
       seq_B0_dicoms($asl_dicom_series,0);
       my $series_name = basename($asl_dicom_series);
       $asl_nifti = "$out_dir/${series_name}.nii.gz";
       dicom_series_2niftigz($asl_dicom_series,$asl_nifti, $series_type);
   }
   if ($force_prep) {
       prep4_B0_correction($mag_dest_dir, $phase_dest_dir, $out_dir); 
   } else {
     unless (-d $out_dir && already_prepared_4B0($out_dir)) {
         prep4_B0_correction($mag_dest_dir, $phase_dest_dir, $out_dir); 
     }
   }
   my $target_nifti = do_B0_correction($asl_nifti, $out_dir, $te_diff, $esp);
   unless (-e $target_nifti) {
       error("B0 correction failed!",3);
   }
   my $target_prefix = basename($target_nifti);
   $target_prefix =~ s/\..+$//;
   my $brik_file = "$out_dir/${target_prefix}+orig.BRIK";
   my $head_file = "$out_dir/${target_prefix}+orig.HEAD";
   unlink $brik_file if (-e $brik_file);
   unlink $head_file if (-e $head_file);

   create_brik($out_dir,$target_nifti, $target_prefix);

   exit 0;
}

main();
