package clinical.storage.plugins.gridftp;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.globus.ftp.DataSink;
import org.globus.ftp.DataSource;
import org.globus.ftp.FileInfo;
import org.globus.ftp.FileRandomIO;
import org.globus.ftp.GridFTPClient;
import org.globus.ftp.GridFTPSession;
import org.globus.ftp.RetrieveOptions;
import org.globus.ftp.Session;
import org.globus.ftp.exception.ClientException;
import org.globus.ftp.exception.ServerException;
import org.globus.myproxy.MyProxy;
import org.globus.replica.rls.LocalReplicaCatalog;
import org.globus.replica.rls.Mapping;
import org.globus.replica.rls.MappingResult;
import org.globus.replica.rls.RLSConnection;
import org.globus.replica.rls.RLSConnectionSource;
import org.globus.replica.rls.RLSException;
import org.globus.replica.rls.RLSStatusCode;
import org.globus.replica.rls.Results;
import org.globus.replica.rls.SimpleCatalogQuery;
import org.globus.replica.rls.impl.SimpleRLSConnectionSource;
import org.globus.util.GlobusURL;
import org.ietf.jgss.GSSCredential;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: GridFtpClient.java 750 2012-12-03 20:45:56Z bozyurt $
 */
public class GridFtpClient {
	protected String masterSiteGridFTPHost;
	protected String myproxyHost;
	protected String rlsURL;
	protected int myproxyPort = 7512;
	protected int gridFtpPort = 2811;
	protected GSSCredential credential = null;
	protected RLSConnectionSource source = null;
	protected RLSConnection con = null;
	protected boolean verbose = false;
	protected boolean allowDownloadParallelism = false;
	protected Map<String, GridFTPClient> clientMap = new HashMap<String, GridFTPClient>(
			17);
	private Log log = LogFactory.getLog(GridFtpClient.class);

	public GridFtpClient(String gridFTPHost, String myproxyHost, String rlsURL,
			int myproxyPort, int gridFtpPort) {
		super();
		this.masterSiteGridFTPHost = gridFTPHost;
		this.myproxyHost = myproxyHost;
		this.myproxyPort = myproxyPort;
		this.rlsURL = rlsURL;
		this.gridFtpPort = gridFtpPort;
		if (System.getProperty("GLOBUS_TCP_PORT_RANGE") != null) {
			allowDownloadParallelism = true;
		}
		// System.setProperty("GLOBUS_TCP_PORT_RANGE", "50000,51000");
	}

	public GridFtpClient(String gridFTPHost, String myproxyHost, String rlsURL) {
		this(gridFTPHost, myproxyHost, rlsURL, 7512, 2811);
	}

	public void connect(String username, String pwd) throws Exception {
		MyProxy proxy = new MyProxy(myproxyHost, myproxyPort);
		credential = proxy.get(username, pwd, 0);
		if (verbose) {
			System.out.println(credential.getName());
		}
		source = new SimpleRLSConnectionSource();
		GlobusURL url = new GlobusURL(rlsURL);
		con = source.connect(url, credential);
		// connect to the master site
		GridFTPClient c = new GridFTPClient(masterSiteGridFTPHost, gridFtpPort);
		c.authenticate(credential);
		clientMap.put(masterSiteGridFTPHost, c);
	}

	public void connect(GSSCredential credential) throws Exception {
		System.out.println("in connect(credential)");
		System.out.println("Credential:" + credential.getName());
		System.out.println("Credential Remaining Lifetime:" + credential.getRemainingLifetime());
		source = new SimpleRLSConnectionSource();
		GlobusURL url = new GlobusURL(rlsURL);
		con = source.connect(url, credential);
		// connect to the master site
		System.out.println("masterSiteGridFTPHost:" + masterSiteGridFTPHost
				+ " gridFtpPort:" + gridFtpPort);
		GridFTPClient c = new GridFTPClient(masterSiteGridFTPHost, gridFtpPort);

		c.authenticate(credential);
		clientMap.put(masterSiteGridFTPHost, c);
	}

	public void shutdown() {
		for (GridFTPClient c : clientMap.values()) {
			if (c != null) {
				try {
					c.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		clientMap.clear();

		if (con != null) {
			try {
				con.close();
				con = null;
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	protected void connect2Ftp(String aGridFtpHost) throws Exception {
		GridFTPClient c = new GridFTPClient(aGridFtpHost, gridFtpPort);
		c.authenticate(credential);
		clientMap.put(aGridFtpHost, c);
	}

	protected String getPhysicalLocation(String logicalName) throws RLSException {
		LocalReplicaCatalog lrc = con.catalog();
		Results results = lrc.query(new SimpleCatalogQuery(
				SimpleCatalogQuery.queryMappingsByLogicalName, logicalName, null));
		if (results.getRC() == RLSStatusCode.RLS_SUCCESS) {
			List<?> batch = results.getBatch();
			for (Object o : batch) {
				MappingResult mr = (MappingResult) o;
				if (mr.getRC() == RLSStatusCode.RLS_SUCCESS) {
					return mr.getTarget();
				}
			}
		}
		return null;
	}

	protected String getHost(String physicalLoc) {
		String s = physicalLoc.replaceFirst("^gsiftp:\\/\\/", "");
		int idx = s.indexOf('/');
		if (idx == -1)
			throw new RuntimeException("Not a valid GridFTP URL!:" + physicalLoc);
		String gridFtpHost = s.substring(0, idx);
		return gridFtpHost;
	}

	protected String getPath(String physicalLoc) {
		String s = physicalLoc.replaceFirst("^gsiftp:\\/\\/", "");
		int idx = s.indexOf('/');
		if (idx == -1)
			throw new RuntimeException("Not a valid GridFTP URL!:" + physicalLoc);
		return s.substring(idx);
	}

	protected int createMapping(String remoteFileLogical, String gridFtpHost,
			LocalReplicaCatalog lrc) throws RLSException {
		StringBuilder sb = new StringBuilder(100);
		sb.append("gsiftp://").append(gridFtpHost);
		sb.append(remoteFileLogical);
		String physicalLoc = sb.toString();
		List<Mapping> mpList = new ArrayList<Mapping>(1);
		mpList.add(new Mapping(remoteFileLogical, physicalLoc));
		return createMappings(mpList, lrc, 0);
	}

	protected int createMappings(List<String> remoteFileLogicalList,
			String gridFtpHost, LocalReplicaCatalog lrc) throws RLSException {
		final int BUFFER_SIZE = 100;
		List<Mapping> mpList = new ArrayList<Mapping>(BUFFER_SIZE);
		StringBuilder sb = new StringBuilder(100);
		sb.append("gsiftp://").append(gridFtpHost);
		String targetLocPrefix = sb.toString();
		int mappingCount = 0;
		int count = 0;
		@SuppressWarnings("unused")
		int totCount = 0;
		for (String rfl : remoteFileLogicalList) {
			if (count < BUFFER_SIZE) {
				Mapping mapping = new Mapping(rfl, targetLocPrefix + rfl);
				mpList.add(mapping);
			} else {
				count = 0;
				mappingCount = createMappings(mpList, lrc, mappingCount);
				mpList.clear();
				Mapping mapping = new Mapping(rfl, targetLocPrefix + rfl);
				mpList.add(mapping);
			}

			count++;
			totCount++;
		}
		if (!mpList.isEmpty()) {
			mappingCount = createMappings(mpList, lrc, mappingCount);
		}

		return mappingCount;
	}

	protected int createMappings(List<Mapping> mpList, LocalReplicaCatalog lrc,
			int mappingCount) throws RLSException {
		List<?> failedMappings = lrc.createMappings(mpList);
		mappingCount += mpList.size();
		if (failedMappings != null && !failedMappings.isEmpty()) {
			for (Object o : failedMappings) {
				MappingResult mr = (MappingResult) o;
				System.err.println("Failed mapping:" + mr);
			}
			mappingCount -= failedMappings.size();
		}
		return mappingCount;
	}

	public void putFile(File localFile, String remoteDir, String gridFtpHost)
			throws Exception {
		if (con == null)
			throw new RuntimeException(
					"Not properly initialized! Call connect() first!");

		GridFTPClient client = clientMap.get(gridFtpHost);
		if (client == null) {
			throw new RuntimeException(
					"Not properly initialized! Call connect() first!");
		}

		client.setType(Session.TYPE_IMAGE);
		client.setMode(GridFTPSession.MODE_EBLOCK);
		long size = localFile.length();
		// 0.5M to 50M
		if (size > 500000L && size <= 50000000L) {
			client.setOptions(new RetrieveOptions(2));
		} else if (size > 50000000L) {
			client.setOptions(new RetrieveOptions(8));
		}

		client.setPassive();
		client.setLocalActive();
		String remoteFile = new File(remoteDir, localFile.getName())
				.getAbsolutePath();
		DataSource source = null;

		try {
			source = new FileRandomIO(new RandomAccessFile(localFile, "r"));
			if (!client.exists(remoteDir)) {
				try {
					makeFullDirPath(remoteDir, client);
					// client.makeDir(remoteDir);
				} catch (Exception e) {
					log.error("cannot create directory:" + remoteDir);
					throw e;
				}
			}
			client.put(remoteFile, source, null);
			LocalReplicaCatalog lrc = con.catalog();
			createMapping(remoteFile, gridFtpHost, lrc);

		} finally {
			if (source != null) {
				source.close();
			}
		}
	}

	protected void makeFullDirPath(String remoteDir, GridFTPClient client)
			throws ServerException, IOException {
		String[] parts = remoteDir.split("\\/+");
		for (int i = 1; i <= parts.length; i++) {
			String partialPath = buildPathFromParts(parts, i);
			if (!client.exists(partialPath)) {
				client.makeDir(partialPath);
			}
		}
	}

	public static String buildPathFromParts(String[] parts, int numParts) {
		StringBuilder sb = new StringBuilder(100);
		for (int i = 0; i < numParts; i++) {
			sb.append('/').append(parts[i]);
		}
		return sb.toString();
	}

	public void downloadFolderNoRLSLookup(String destFolder, File localFolder,
			boolean useParallelism) throws Exception {
		String aGridFtpHost = this.masterSiteGridFTPHost;
		GridFTPClient client = clientMap.get(aGridFtpHost);
		if (client == null) {
			client = new GridFTPClient(aGridFtpHost, gridFtpPort);
			client.authenticate(credential);
			clientMap.put(aGridFtpHost, client);
		}
		if (!allowDownloadParallelism) {
			useParallelism = false;
		}
		downloadFolder(client, destFolder, localFolder, useParallelism);
	}

	public void downloadFolder(String destFolder, File localFolder,
			boolean useParallelism) throws Exception {
		String physicalLoc = getPhysicalLocation(destFolder);
		if (physicalLoc == null) {
			log.error("No physical location map info for :" + destFolder);
			return;
		}
		String aGridFtpHost = getHost(physicalLoc);
		System.out.println("GridFTPHost:" + aGridFtpHost);
		GridFTPClient client = clientMap.get(aGridFtpHost);
		if (client == null) {
			client = new GridFTPClient(aGridFtpHost, gridFtpPort);
			client.authenticate(credential);
			clientMap.put(aGridFtpHost, client);
		}
		if (!allowDownloadParallelism) {
			useParallelism = false;
		}
		downloadFolder(client, destFolder, localFolder, useParallelism);
	}

	@SuppressWarnings("unchecked")
	public void downloadFolder(GridFTPClient client, String destFolder,
			File localFolder, boolean useParallelism) throws Exception {

		client.setType(Session.TYPE_IMAGE);
		if (useParallelism) {
			client.setMode(GridFTPSession.MODE_EBLOCK);
			client.setOptions(new RetrieveOptions(4));
			client.setLocalPassive();
			client.setActive();
		} else {
			client.setPassive();
			client.setLocalActive();
		}

		client.changeDir(destFolder);
		Vector<FileInfo> remDirList = client.list();
		File dest = null;
		File localFile;
		for (FileInfo fi : remDirList) {
			if (fi.getName().equals(".") || fi.getName().equals("..")) {
				continue;
			}
			localFile = new File(localFolder, fi.getName());
			dest = new File(destFolder, fi.getName());
			if (fi.isDirectory()) {
				localFile.mkdirs();
				downloadFolder(client, dest.getAbsolutePath(), localFile,
						useParallelism);
			} else {
				downloadFile(client, dest.getAbsolutePath(), localFile,
						useParallelism);
			}
		}
	}

	public void downloadFile(String destFile, File localFile,
			boolean useParallelism) throws Exception {
		String physicalLoc = getPhysicalLocation(destFile);
		if (physicalLoc == null) {
			log.error("No physical location map info for :" + destFile);
			return;
		}
		String aGridFtpHost = getHost(physicalLoc);
		GridFTPClient client = clientMap.get(aGridFtpHost);
		if (client == null) {
			client = new GridFTPClient(aGridFtpHost, gridFtpPort);
			client.authenticate(credential);
			clientMap.put(aGridFtpHost, client);
		}
		if (!allowDownloadParallelism) {
			useParallelism = false;
		}
		downloadFile(client, destFile, localFile, useParallelism);
	}

	public void downloadFileNoRLSLookup(String destFile, File localFile,
			boolean useParallelism) throws Exception {
		String aGridFtpHost = this.masterSiteGridFTPHost;
		GridFTPClient client = clientMap.get(aGridFtpHost);
		if (client == null) {
			client = new GridFTPClient(aGridFtpHost, gridFtpPort);
			client.authenticate(credential);
			clientMap.put(aGridFtpHost, client);
		}
		if (!allowDownloadParallelism) {
			useParallelism = false;
		}
		downloadFile(client, destFile, localFile, useParallelism);
	}

	public void downloadFile(GridFTPClient client, String destFile,
			File localFile, boolean useParallelism) throws ClientException,
			IOException, ServerException {
		client.setType(Session.TYPE_IMAGE);
		if (!useParallelism) {
			client.setPassive();
			client.setLocalActive();
			client.get(destFile, localFile);
		} else {
			client.setMode(GridFTPSession.MODE_EBLOCK);
			client.setOptions(new RetrieveOptions(4));
			client.setLocalPassive();
			client.setActive();
			DataSink sink = null;
			try {
				sink = new FileRandomIO(new RandomAccessFile(localFile, "rw"));
				client.get(destFile, sink, null);
			} finally {
				if (sink != null)
					sink.close();
			}
		}
	}

	public String getMasterSiteGridFTPHost() {
		return masterSiteGridFTPHost;
	}

}
