package clinical.test.scheduler;

import static org.junit.Assert.assertNull;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.sql.Connection;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import clinical.server.dao.JobsDAO;
import clinical.server.vo.Jobs;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.DownloadCachePolicy;
import clinical.web.scheduler.IJob;
import clinical.web.scheduler.JobInfo;
import clinical.web.scheduler.JobRecord;
import clinical.web.scheduler.JobScheduler;
import clinical.web.scheduler.JobScheduler.JobSubmissionType;
import clinical.web.services.SecurityService;

public class SchedulerTests {
	protected ConnectionSupportMixin mixin;

	@Before
	public void setUp() throws Exception {

		mixin = new ConnectionSupportMixin("users.xml");
		mixin.startup();
		SecurityService ss = SecurityService.getInstance(mixin
				.getDbPoolService(), mixin.getDbID(), mixin.getDbType());
		ss.startup();

		String schedulerDir = "/tmp/schedulerDir";
		new File(schedulerDir).mkdir();
		DownloadCachePolicy cachePolicy = DownloadCachePolicy
				.getInstance(schedulerDir);
		JobScheduler scheduler = JobScheduler.getInstance(schedulerDir,
				cachePolicy);
		Thread thread = new Thread(scheduler);
		thread.setDaemon(false);
		thread.setPriority(Thread.NORM_PRIORITY - 1);
		thread.start();
	}

	@After
	public void tearDown() throws Exception {
		System.out.println("tearDown");
		String[] jobIds = { "test1", "batch1", "batch2", "batch3", "batch4",
				"batch5", "batch6", "batch7" };
		JobScheduler scheduler = JobScheduler.getInstance();
		scheduler.shutdown();
		if (mixin != null) {
			Connection con = mixin.getConnection();
			try {
				JobsDAO jobsDAO = DAOFactory.createJobsDAO(mixin.getDbID());
				Jobs cr = new Jobs();
				for (String jobId : jobIds) {
					cr.setJobid(jobId);
					cr.setJobuser(mixin.getUi().getPerceivedName());
					jobsDAO.delete(con, cr);
				}
			} finally {
				if (con != null) {
					mixin.releaseConnection(con);
				}
			}
			mixin.shutdown();
		}
	}

	@Test
	public void testBatchQueue() throws Exception {
		String[] jobIds = { "batch1", "batch2", "batch3", "batch4", "batch5",
				"batch6", "batch7" };
		UserInfo ui = mixin.getUi();
		String dbID = mixin.getDbID();

		String jobUser = null;
		JobScheduler scheduler = JobScheduler.getInstance();
		for (String jobId : jobIds) {
			IJob job = new AutoCBFProcessingJobMock(jobId, ui,
					"Mock Batch Process", dbID);
			scheduler.addJob(job, JobSubmissionType.BATCH);
			jobUser = job.getUser();
		}

		while (true) {
			synchronized (this) {
				try {
					this.wait(2000l);
				} catch (InterruptedException e) {
				}
			}
			boolean hasRunning = false;
			int numRunning = 0;
			for (String jobId : jobIds) {
				JobRecord jobRecord = scheduler.getJobRecord(ui, dbID, jobUser,
						jobId);
				if (jobRecord != null) {
					String status = jobRecord.getStatus();
					if (status.equals(JobInfo.RUNNING)) {
						numRunning++;
						hasRunning = true;
						System.out.println("running job " + jobId);
					}
				}
			}
			if (!hasRunning) {
				break;
			} else {
				System.out.println("# of concurrent jobs:" + numRunning);
				System.out.println("-----------------------------------");
			}
		}

	}

	public void testSplitJobs() throws Exception {
		String jobId = "test1";
		UserInfo ui = mixin.getUi();
		String dbID = mixin.getDbID();
		IJob job = new CBFProcessingJobMock(jobId, ui, "Mock Split Process",
				dbID);

		assertNull(job.getContextAsJSON());
		JobScheduler scheduler = JobScheduler.getInstance();
		scheduler.addJob(job, JobSubmissionType.INDIVIDUAL);
		BufferedReader console = new BufferedReader(new InputStreamReader(
				System.in));
		while (true) {
			synchronized (this) {
				try {
					this.wait(2000l);
				} catch (InterruptedException e) {
				}
			}
			JobRecord jobRecord = scheduler.getJobRecord(ui, dbID, job
					.getUser(), jobId);
			if (jobRecord != null) {
				String status = jobRecord.getStatus();
				System.out.println("status:" + status);
				if (status.equals(JobInfo.WAITING)) {
					System.out.printf(
							"Do you want to resume the job (%s) (y/N/c)? ",
							jobId);
					String ans = console.readLine().trim();
					if (ans.equalsIgnoreCase("y")) {
						scheduler.resumeJob(ui, dbID, job.getUser(), jobId);
					} else if (ans.equalsIgnoreCase("c")) {
						System.out.println("canceling job:" + jobId);
						scheduler.cancelJob(ui, dbID, job.getUser(), jobId);
					}
				} else if (status.equals(JobInfo.FINISHED)
						|| status.equals(JobInfo.FINISHED_WITH_ERR)
						|| status.equals(JobInfo.CANCELED)) {
					break;
				}
			} else {
				break;
			}
		}
		// TODO handle cancel after shelving without first resuming
		// TODO generalize JobManagementServiceImpl.removeOrphanJobs()

	}

	static void cleanup() {
		SchedulerTests t = new SchedulerTests();
		try {
			t.setUp();
			t.tearDown();
		} catch (Exception x) {
			x.printStackTrace();
		}
	}

	public static void main(String[] args) throws Exception {
		//cleanup();
		
		org.junit.runner.JUnitCore
				.main("clinical.test.scheduler.SchedulerTests");
	}

}
