package clinical.tools.dbadmin;

import java.io.File;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jdom.Element;

import clinical.server.dao.ConfParamsDAO;
import clinical.server.dao.ConfParamsGroupDAO;
import clinical.server.vo.ConfParams;
import clinical.server.vo.ConfParamsGroup;
import clinical.server.vo.Databaseuser;
import clinical.utils.FileUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.MinimalServiceFactory;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class ConfigParamsAdmin extends AbstractAdmin {
	List<ConfigParamRec> cpList = new ArrayList<ConfigParamRec>();
	Map<String, Integer> groupName2IdMap = new HashMap<String, Integer>(11);

	public ConfigParamsAdmin(String xmlFile, String usersXmlFile) throws Exception  {
		super();
		Element rootEl = FileUtils.loadXML(xmlFile);
		List<?> children = rootEl.getChildren("config-param");
		for (Object o : children) {
			Element ce = (Element) o;
			cpList.add( ConfigParamRec.fromXml(ce));
		}
		csm = new ConnectionSupportMixin(usersXmlFile);
		csm.startup();
		MinimalServiceFactory.setMimimalOpMode(true);
		init();
	}
	
	void init() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			ConfParamsGroupDAO dao = DAOFactory.createConfParamsGroupDAO(csm.getDbID());
			List<ConfParamsGroup> list = dao.find(con, new ConfParamsGroup());
			for(ConfParamsGroup pg : list) {
				this.groupName2IdMap.put(pg.getGroupName(), pg.getUniqueId().intValue());
			}
			
		} finally {
			csm.releaseConnection(con);
		}		
	}
	
	public void save() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			Map<String, ConfParams> map = getConfParams(con);
			for(ConfigParamRec cpRec : this.cpList) {
				ConfParams cp = map.get(cpRec.name);
				if (cp == null) {
					System.out.println("inserting " + cpRec);
					insert(con, cpRec);
				} else {
					System.out.println("updating " + cpRec);
					update(con, cpRec, cp);
				}
			}
			con.commit();
		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}
	
	
	public Map<String, ConfParams> getConfParams(Connection con) throws Exception {
		ConfParamsDAO dao = DAOFactory.createConfParamsDAO(csm.getDbID());
		List<ConfParams> list = dao.find(con, new ConfParams());
		Map<String, ConfParams> map = new HashMap<String, ConfParams>();
		for(ConfParams cp  : list) {
			map.put(cp.getName(), cp);
		}
		return map;
	}
	
	public void update(Connection con, ConfigParamRec cpRec, ConfParams oldCp)
			throws Exception {
		ConfParams cr = new ConfParams();
		cr.setName(oldCp.getName());
		ConfParams cp = new ConfParams();
		boolean ok = false;
	    if (!cpRec.value.equals(oldCp.getValue())) {
	    	cp.setValue(cpRec.value);
	    	ok = true;
	    }
	    if (cpRec.descr != null && ! cpRec.descr.equals(oldCp.getDescription())) {
	    	cp.setDescription(cpRec.descr);
	    	ok = true;
	    }
	    if (ok) {
	    	ConfParamsDAO dao = DAOFactory.createConfParamsDAO(csm.getDbID());
	    	dao.update(con, cp, cr);
	    }
	}
	
	public void insert(Connection con, ConfigParamRec cpRec) throws Exception {
		ConfParamsDAO dao = DAOFactory.createConfParamsDAO(csm.getDbID());
		ConfParams cp = new ConfParams();
		cp.setName(cpRec.name);
		cp.setValue(cpRec.value);
		cp.setType(cpRec.type);
		Integer groupId = groupName2IdMap.get(cpRec.groupName);
		cp.setParamGroupId( new BigDecimal(groupId));
		if (cpRec.descr != null) {
			cp.setDescription(cpRec.descr);
		}
		Databaseuser adminDU = getDatabaseUser(csm.getUi(),
				Constants.USERCLASS_ADMIN);
		cp.setOwner(adminDU.getUniqueid());
		cp.setModUser(adminDU.getUniqueid());
		cp.setModTime(new Date());
		dao.insert(con, cp);
	}
	
	public static class ConfigParamRec {
		String name;
		String value;
		String type;
		String descr;
		String groupName = "Advanced";
		public final static String STRING = "string";
		public final static String INT = "int";
		public final static String BOOL = "bool";
		
		public ConfigParamRec() {
		}
		
		public Element toXml() {
			Element el = new Element("config-param");
			el.setAttribute("name",name);
			el.setAttribute("value", value);
			el.setAttribute("type", type);
			el.setAttribute("groupName", groupName);
			if (descr != null) {
				el.setAttribute("descr", descr);
			}
			return el;
		}
		
		public static ConfigParamRec fromXml(Element el) {
			ConfigParamRec cp = new ConfigParamRec();
			cp.name = el.getAttributeValue("name");
			cp.value = el.getAttributeValue("value");
			cp.type = el.getAttributeValue("type");
			cp.groupName = el.getAttributeValue("groupName");
			if (el.getAttribute("descr") != null) {
				cp.descr = el.getAttributeValue("descr");
			}
			return cp;
		}	
		
		public String toString() {
		  StringBuilder sb = new StringBuilder(200);
		  sb.append("ConfigParamRec::[");
		  sb.append("name:").append(name);
		  sb.append(",value:").append(value);
		  sb.append(",type:").append(type);
		  sb.append(",groupName:").append(groupName);
		  if (descr != null) {
			  sb.append(",descr:").append(descr);
		  }
		  sb.append(']');
		  return sb.toString();
		}
	}
	
	public static void usage() {
		System.err.println("Usage:ConfigParamsAdmin <config-params-xml-file>\n");
		System.exit(1);
	}
	
	public static void main(String[] args) throws Exception {
		if (args.length != 1) {
			usage();
		}
		String cpXml = args[0];
		if (!new File(cpXml).isFile()) {
			System.err.println("Not a file:" + cpXml);
			usage();
		}
		ConfigParamsAdmin cpa = null;
		try {
			cpa = new ConfigParamsAdmin(cpXml, "users.xml");
			
			cpa.save();
		} finally {
			if (cpa != null)
				cpa.shutdown();
		}
		
	}
}
