package clinical.tools.dbadmin;

import java.io.File;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jdom.Element;

import clinical.server.dao.ProtocolDAO;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Protocol;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */

public class ProtocolAdmin extends AbstractAdmin {
	List<ProtocolRec> prList = new ArrayList<ProtocolRec>();

	public ProtocolAdmin(String xmlFile, String usersXmlFile) throws Exception {
		super();
		Element rootEl = FileUtils.loadXML(xmlFile);
		List<?> children = rootEl.getChildren("protocol");
		for (Object o : children) {
			Element ce = (Element) o;
			prList.add(ProtocolRec.fromXml(ce));
		}

		csm = new ConnectionSupportMixin(usersXmlFile);
		csm.startup();
		MinimalServiceFactory.setMimimalOpMode(true);

	}

	public void save() throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			Map<String, Protocol> map = getProtocols(con);

			for (ProtocolRec pr : prList) {
				String key = pr.protocolId + ":" + pr.protocolVersion;
				Protocol protocol = map.get(key);
				if (protocol == null) {
					System.out.println("inserting " + pr);
					insert(con, pr);
				} else {
					System.out.println("updating " + pr);
					update(con, pr, protocol);
				}
			}
			con.commit();
		} catch (Exception x) {
			con.rollback();
			x.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	public Map<String, Protocol> getProtocols(Connection con) throws Exception {
		ProtocolDAO dao = DAOFactory.createProtocolDAO(csm.getDbID());
		List<Protocol> list = dao.find(con, new Protocol());
		Map<String, Protocol> protocolMap = new HashMap<String, Protocol>();
		for (Protocol p : list) {
			String key = p.getProtocolid() + ":" + p.getProtocolversion();
			protocolMap.put(key, p);
		}
		return protocolMap;
	}

	public void update(Connection con, ProtocolRec pr, Protocol oldProtocol)
			throws Exception {
		Protocol cr = new Protocol();
		cr.setUniqueid(oldProtocol.getUniqueid());
		Protocol updateBean = new Protocol();
		if (pr.descr != null) {
			updateBean.setDescription(pr.descr);
			ProtocolDAO dao = DAOFactory.createProtocolDAO(csm.getDbID());
			dao.update(con, updateBean, cr);
		}
	}

	public void insert(Connection con, ProtocolRec pr) throws Exception {
		ProtocolDAO dao = DAOFactory.createProtocolDAO(csm.getDbID());

		Protocol protocol = new Protocol();
		protocol.setName(pr.name);
		protocol.setProtocolid(pr.protocolId);
		protocol.setProtocolversion(GenUtils.toBigDecimal(pr.protocolVersion));
		protocol.setDescription(pr.descr);
		protocol.setModtime(new Date());

		Databaseuser adminDU = getDatabaseUser(csm.getUi(),
				Constants.USERCLASS_ADMIN);
		protocol.setOwner(adminDU.getUniqueid());
		protocol.setModuser(adminDU.getUniqueid());

		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(csm.getDbID());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.PROTOCOL_TABLE, "uniqueid");
		protocol.setUniqueid(uniqueid);

		int tableID = DBUtils.getTableID(csm.getDbID(), con,
				Constants.PROTOCOL_TABLE);

		protocol.setTableid(GenUtils.toBigDecimal(tableID));

		dao.insert(con, protocol);
	}


	public static class ProtocolRec {
		String protocolId;
		int protocolVersion;
		String name;
		String descr;

		public ProtocolRec() {
		}

		public Element toXml() {
			Element el = new Element("protocol");
			el.setAttribute("protocolId", protocolId);
			el.setAttribute("protocolVersion", String.valueOf(protocolVersion));
			el.setAttribute("name", name);
			if (descr != null) {
				Element de = new Element("descr");
				de.addContent(descr);
				el.addContent(de);
			}
			return el;
		}

		public static ProtocolRec fromXml(Element el) {
			ProtocolRec pr = new ProtocolRec();
			pr.protocolId = el.getAttributeValue("protocolId");
			pr.protocolVersion = Integer.parseInt(el
					.getAttributeValue("protocolVersion"));
			if (el.getAttribute("name") != null) {
				pr.name = el.getAttributeValue("name");
			} else {
				pr.name = pr.protocolId;
			}
			if (el.getChild("descr") != null) {
				pr.descr = el.getChildTextTrim("descr");
			}
			return pr;
		}

		public String toString() {
			StringBuilder sb = new StringBuilder();
			sb.append("ProtocolRec::[");
			sb.append("protocolId:").append(protocolId);
			sb.append(",protocolVersion:").append(protocolVersion);
			sb.append("]");
			return sb.toString();
		}
	}// ;

	public static void testit() throws Exception {
		String rootDir = "/Users/bozyurt/dev/java/cbf_birn/hid/clinical/branches/BRANCH_cbfbirn/clinical";
		String protocolXml = rootDir + "/protocols.xml";

		ProtocolAdmin pa = null;
		try {
			pa = new ProtocolAdmin(protocolXml, "users.xml");

			pa.save();
		} finally {
			if (pa != null)
				pa.shutdown();
		}
	}

	public static void usage() {
		System.err.println("Usage:ProtocolAdmin <protocol-xml-file>\n");
		System.exit(1);
	}

	public static void main(String[] args) throws Exception {
		if (args.length != 1) {
			usage();
		}
		String protocolXml = args[0];
		if (!new File(protocolXml).isFile()) {
			System.err.println("Not a file:" + protocolXml);
			usage();
		}
		ProtocolAdmin pa = null;
		try {
			pa = new ProtocolAdmin(protocolXml, "users.xml");

			pa.save();
		} finally {
			if (pa != null)
				pa.shutdown();
		}
	}

}
